/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: keyboard.cc,v 1.1 2004/06/19 11:36:07 jd Exp $

   Created 06/15/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file libs/interactivity/keyboard.cc
  \brief keyboard controler.
*/

#include "leg/libs/interactivity/keyboard.h"
#include "leg/support/utils/errors.h"
#include <iostream>

namespace leg
{
namespace libs
{
namespace interactivity
{
   KeyboardMap		Keyboard::map;
   KeyboardMap		Keyboard::release_map;
   KeyboardMap		Keyboard::last_map;
   KeyClickTimeMap	Keyboard::pressed_time_map;
   KeyClickMap		Keyboard::click_map;
   Timing		Keyboard::timing;
   GameTime		Keyboard::click_time = .25;
   Key			Keyboard::current_key = Key(0);
   char			Keyboard::current_char = 0;
   std::string		Keyboard::current_string = "";
 
   Keyboard::Keyboard():Controler()
   {
   }

   Keyboard::~Keyboard ()
   {
   }

   bool
   Keyboard::IsKeyPressed (Key k)
   {
      return (map[k] == true);
   }

   bool
   Keyboard::IsKeyReleased (Key k)
   {
      return (release_map[k] == true);
   }
   
   bool
   Keyboard::IsKeyClicked (Key mb)
   {
      return (click_map[mb] == 1);
   }

   bool
   Keyboard::IsKeyDoubleClicked (Key mb)
   {
      return (click_map[mb] == 2);
   }

   void
   Keyboard::SetClickTimeDuration (GameTime duration)
   {
      if (duration > 0)
	 click_time = duration;
      else
	 support::utils::Warning ("Click time duration could not be negative, keep last value",
	       "libs::interactivity::Keyboard::SetClickTimeDuration");
   }

   GameTime
   Keyboard::GetClickTimeDuration()
   {
      return click_time;
   }

   char
   Keyboard::GetCurrentChar()
   {
      return current_char;
   }

   Key
   Keyboard::GetCurrentKey()
   {
      return current_key;
   }

   const std::string&
   Keyboard::GetCurrentString()
   {
      return current_string;
   }

   void
   Keyboard::SetSaveString (bool val)
   {
//      user_event->SetSaveStringKey (val);
   }

   bool
   Keyboard::GetSaveString()
   {
//      return user_event->GetSaveStringKey();
   }

   void
   Keyboard::PressKey (Key k)
   {
      last_map[k] = map[k];
      map[k] = true;
      release_map[k] = false;
      current_key = k;
      UpdateKeyStatus (k,true);
      UpdateChar (k,true);
   }

   void
   Keyboard::UnpressKey (Key k)
   {
      last_map[k] = map[k];
      map[k] = false;
      release_map[k] = true;
      if (current_key == k)
	 current_key = (Key)0;
      UpdateKeyStatus (k,false);
      UpdateChar (k,false);
   }

   void
   Keyboard::UpdateKeyStatus (Key n, bool press)
   {
      if (press){
	 pressed_time_map[n] = timing.GetCurrentTime();
      }
      else{
	 if (last_map[n]){
	    GameTime gt = timing.GetCurrentTime();
	    if ((gt - pressed_time_map[n]) <= click_time)
		  ++click_map[n];
	    else
	       click_map[n] = 0;
	 }
	 else{
	    pressed_time_map[n] = 0;
	    click_map[n] = 0;
	 }
      }
   }

   void
   Keyboard::UpdateChar (Key k, bool press)
   {
      static bool maj = false;
      static bool ver_maj = false;
      
      switch (k){
	 case support::window::key::key_escape:
	 case support::window::key::key_f1:
	 //case support::window::key::key_f2:
	 case support::window::key::key_f3:
	 case support::window::key::key_f4:
	 case support::window::key::key_f5:
	 case support::window::key::key_f6:
	 case support::window::key::key_f7:
	 case support::window::key::key_f8:
	 case support::window::key::key_f9:
	 case support::window::key::key_f10:
	 case support::window::key::key_f11:
	 case support::window::key::key_f12:
	 case support::window::key::key_print_screen:
	 case support::window::key::key_stop_loop:
	 //case support::window::key::key_pause:
	 case support::window::key::key_backspace:
	 case support::window::key::key_inser:
	 case support::window::key::key_begin:
	 case support::window::key::key_page_up:
	 case support::window::key::key_numlock:
	 case support::window::key::key_tab:
	 case support::window::key::key_suppr:
	 case support::window::key::key_end:
	 case support::window::key::key_page_down:
	 case support::window::key::key_return:
	 case support::window::key::key_up:
	 case support::window::key::key_l_ctrl:
	 case support::window::key::key_l_alt:
	 case support::window::key::key_r_alt:
	 case support::window::key::key_r_ctrl:
	 case support::window::key::key_left:
	 case support::window::key::key_down:
	 case support::window::key::key_right:
	    break;
	 case support::window::key::key_l_shift:
	 case support::window::key::key_r_shift:
	    if (press)
	       maj = true;
	    else
	       maj = false;
	    break;
	 case support::window::key::key_caps_lock:
	    if (press)
	       ver_maj = !ver_maj;
	    break;
	 default:
	    if (press){
	       current_char = (maj || ver_maj) ? toupper (k) : k;

	       // need to check numerical
	       
	       return;
	    }
      }
      
      current_char = '\0';
   }
}
}
}
