/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: simpleentity.cc,v 1.1 2004/07/15 23:28:23 jd Exp $

   Created 07/03/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/physics/simpleentity.cc
  \brief An entity for physics.
*/

#include "leg/libs/physics/simpleentity.h"
#include "leg/support/utils/errors.h"

namespace leg
{
namespace libs
{
namespace physics
{
   using leg::support::utils::Warning;
   using leg::support::utils::Error;

   SimpleEntity::SimpleEntity(): Entity(),
				 revolution_center (0),
				 gravity (0),
				 time (0),
				 last_time (0),
				 speed (0),
				 mass (0),
				 moving_config (move_follows_orientation),
				 gravity_config (no_gravity)
   {
   }

   SimpleEntity::SimpleEntity (const SimpleEntity& e): Entity (static_cast<const Entity&> (e))
   {
      Copy (e);
   }
   
   const SimpleEntity& 
   SimpleEntity::operator = (const SimpleEntity& e)
   {
      Copy (e);
      return *this;
   }

   SimpleEntity::~SimpleEntity()
   {
   }

   void
   SimpleEntity::SetPosition (Vector& pos)
   {
      position = pos;
   }

   const Vector&
   SimpleEntity::GetPosition() const
   {
      return position;
   }

   void
   SimpleEntity::SetVelocity (Vector& vel)
   {
      velocity = vel;
   }

   const Vector&
   SimpleEntity::GetVelocity () const
   {
      return velocity;
   }

   void
   SimpleEntity::SetSpeed (real magnitude)
   {
      speed = magnitude;
      velocity.Normalize (magnitude);
   }

   real
   SimpleEntity::GetSpeed() const
   {
      return speed;
   }

   void
   SimpleEntity::SetAcceleration (Vector& accel)
   {
      acceleration = accel;
   }

   const Vector&
   SimpleEntity::GetAcceleration() const
   {
      return acceleration;
   }

   void
   SimpleEntity::SetAccelerationMagnitude (real magnitude)
   {
      acceleration.Normalize (magnitude);
   }

   real
   SimpleEntity::GetAccelerationMagnitude()
   {
      return acceleration.Magnitude();
   }

   GameTime
   SimpleEntity::GetTime()
   {
      return this->time;
   }

   void
   SimpleEntity::SetTime (GameTime gt)
   {
      time = gt;
   }

   Orientation
   SimpleEntity::GetOrientation() const
   {
      return orientation;
   }

   Orientation
   SimpleEntity::GetOrientationFromVelocity() const
   {
      Vector dum = velocity;
      const real alpha = dum.Angle (Vector (0,1,0));
      Quaternion q1 (-alpha,Vector (0,1,0));
      Vector dum2 = q1.Rotate (dum);
      
      const real beta = dum2.Angle (Vector (1,0,0));
      Quaternion q2 (-beta,Vector (1,0,0));
      Vector dum3 = q2.Rotate (dum2);

      const real gamma = dum3.Angle (Vector (0,0,1));

      return Orientation (alpha,beta,gamma);
   }

   void
   SimpleEntity::SetOrientation (Orientation& o)
   {
      orientation = o;
   }

   void
   SimpleEntity::SetMass (real m)
   {
      mass = m;
   }

   real
   SimpleEntity::GetMass()
   {
      return mass;
   }
   
   void
   SimpleEntity::Update()
   {
      GameTime dt = PreUpdate();
      Update (dt);
      PostUpdate();
   }
   
   void
   SimpleEntity::Update (GameTime dt)
   {
      UpdateAngularSpeed (dt); // angular speed may depend on acceleration
      UpdateOrientation (dt);
      UpdateAcceleration (dt);
      if (moving_config == move_follows_orientation)
	 UpdateVelocityFromOrientation (dt);
      else if (moving_config == move_follows_revolution)
	 UpdateRevolution (dt);
      UpdateVelocity (dt);
      UpdatePosition (dt);
   }

   GameTime
   SimpleEntity::PreUpdate()
   {
      UpdateTime();
      return this->time - last_time;
   }

   void
   SimpleEntity::PostUpdate()
   {
      last_time = this->time;
   }
/*
   Entity&
   SimpleEntity::Clone ()
   {
      return *new SimpleEntity (*this);
   }
*/
   void
   SimpleEntity::UpdatePosition (GameTime dt)
   {
      position += (velocity * dt);
   }
   
   void
   SimpleEntity::UpdateVelocity (GameTime dt)
   {
      velocity += (acceleration * dt);
      speed = velocity.Magnitude();
   }

   void
   SimpleEntity::UpdateVelocityFromOrientation (GameTime dt)
   {
      const real alpha = orientation (0);
      const real beta = orientation (1);
      
      Vector dum1 (0,0,1);
      Vector dum2 = Quaternion (beta,Vector (1,0,0)).Rotate (dum1);
      Vector dum3 = Quaternion (alpha,Vector (0,1,0)).Rotate (dum2);

      dum3.Normalize (speed);
      velocity = dum3;
   }
   
   void
   SimpleEntity::UpdateAcceleration (GameTime dt)
   {
      if (gravity_config == receive_gravity)
	 acceleration = forces.W+forces.R;
      if (forces.F1 != Vector (0,0,0))
	 acceleration += forces.F1;
      if (forces.F2 != Vector (0,0,0))
	 acceleration += forces.F2;

      if (mass)
	 acceleration /= mass;
//      else
//	 support::utils::Error ("zero mass !","libs::physics::SimpleEntity::UpdateAcceleration");
   }

   void
   SimpleEntity::UpdateOrientation (GameTime dt)
   {
      orientation += (angular_speed * dt);
   }
   
   void 
   SimpleEntity::UpdateAngularSpeed (GameTime dt)
   {
      angular_speed += (angular_acceleration * dt);
   }

   void
   SimpleEntity::UpdateRevolution (GameTime dt)
   {
      /*
      if (!revolution_center){
	 Warning ("warning","warning");
	 return;
      }
      
      Vector c = revolution_center->GetPosition();
      Point center (c (0), c(1), c(2));
      Point me (position (0), position (1), position (2));
      real distance = me.DistanceTo (center);
      ...
      */
   }

   void
   SimpleEntity::UpdateTime ()
   {
      this->time = timing.GetCurrentTime();
   }

   void
   SimpleEntity::Copy (const SimpleEntity& e)
   {
      position = e.position;
      velocity = e.velocity;
      acceleration = e.acceleration;
      orientation = e.orientation;
      angular_acceleration = e.angular_acceleration;
      angular_speed = e.angular_speed;
      gravity_center = e.gravity_center;
      revolution_center = e.revolution_center;
      gravity = e.gravity;
      timing = e.timing;
      time = e.time;
      last_time = e.last_time;
      speed = e.speed;
      mass = e.mass;
      moving_config = e.moving_config;
      gravity_config = e.gravity_config;
      forces = e.forces;
   }

   void
   SimpleEntity::SetRevolutionCenter (Entity& entity)
   {
      revolution_center = &entity;
      moving_config = move_follows_revolution;
   }

   const Entity*
   SimpleEntity::GetRevolutionCenter() const
   {
      return revolution_center;
   }

   void
   SimpleEntity::SetMovingConfig (MovingConfig conf)
   {
      moving_config = conf;
   }

   MovingConfig
   SimpleEntity::GetMovingConfig() const
   {
      return moving_config;
   }

   void
   SimpleEntity::SetGravityConfig (GravityConfig conf)
   {
      gravity_config = conf;
   }

   GravityConfig
   SimpleEntity::GetGravityConfig() const
   {
      return gravity_config;
   }
   
   void
   SimpleEntity::UpdateGravity (Vector& g, real height)
   {
      gravity = &g;
      
      // we should update the gravity center
      gravity_center = position;

      real delta = (gravity_center(1) - height);
      
      if (delta > distance_precision)
	 Fall();
      else if (delta > -distance_precision){ // below is surely bad
	 forces.W = Vector (0,0,0);
	 velocity (1) = 0;
      }
      else
	 forces.W = forces.R = Vector (0,0,0);
   }

   void
   SimpleEntity::Fall()
   {
      forces.W = *gravity;
      forces.W *= mass;
   }

   void
   SimpleEntity::SetExternalForce (Vector& f)
   {
      forces.F2 = f;
   }

   const Vector&
   SimpleEntity::GetExternalForce()
   {
      return forces.F2;
   }
}
}
}

