/* This file is part of GNU Libraries and Engines for Games.

   $Id: timer.cc,v 1.1 2004/05/13 04:16:02 jechk Exp $
   $Log: timer.cc,v $
   Revision 1.1  2004/05/13 04:16:02  jechk
   Split timing into two parts; moved message to libs.

   Revision 1.3  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 2/18/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/timing/timer.cc
  \brief The Timer class, a thread that is created at certain times.
*/

#include <iostream>

#include "leg/libs/timing/timing.h"
#include "leg/support/utils/errors.h"

namespace leg
{

namespace libs
{

namespace timing
{

Timer::Timer (Timing &timing)
  : support::threads::Thread (),
    timing (timing),
    next_identifier (0)
{
  timing.thread.AddTimer (this);
}

Timer::~Timer ()
{
  if (!settings.empty ())
    {
      throw support::utils::Warning ("A set timer has been deleted.  This might result in a memory leak.");
    }
  timing.thread.RemoveTimer (this);
}

int
Timer::Set (support::timing::game_time when, void *data)
{
  settings.push_front (TimerSetting (next_identifier,
				     timing.GameTimeToTimeval (when),
				     data));

  return next_identifier++;
}

int
Timer::SetRepeating (support::timing::game_time when, support::timing::game_time interval, void *data)
{
  settings.push_front (TimerSetting (next_identifier,
				     timing.GameTimeToTimeval (when),
				     timing.GameTimeToTimevalInterval (interval),
				     data));

  return next_identifier++;
}

void
Timer::Unset (int identifier)
{
  for (std::list<TimerSetting>::iterator i = settings.begin ();
       i != settings.end (); ++i)
    {
      if (i->identifier == identifier)
	{
	  i->active = false;
	  return;
	}
    }
}

void
Timer::Ping (struct timeval tv)
{
  using namespace support::timing;

  for (std::list<TimerSetting>::iterator i = settings.begin ();
       i != settings.end (); ++i)
    {
      if (i->active)
	{
	  if (i->when.tv_sec < tv.tv_sec
	      || (i->when.tv_sec == tv.tv_sec
		  && i->when.tv_usec < tv.tv_usec))
	    {
	      void *data = i->data;

	      if (i->is_repeating)
		{
		  i->when = i->when + i->interval;
		}
	      else
		{
		  i = settings.erase (i);
		}
	      SetParameters (data);
	      Create ();
	    }
	}
      else
	{
	  i = settings.erase (i);
	}
    }
}

Timer::TimerSetting::TimerSetting (int identifier,
				   struct timeval when,
				   void *data)
  : identifier (identifier),
    when (when),
    is_repeating (false),
    active (true),
    data (data)
{
}

Timer::TimerSetting::TimerSetting (int identifier, 
				   struct timeval when,
				   struct timeval interval,
				   void *data)
  : identifier (identifier),
    when (when),
    is_repeating (true),
    interval (interval),
    active (true),
    data (data)
{
}

}

}

}
