/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: polynomial.cc,v 1.2 2004/03/11 06:44:52 jechk Exp $
   $Log: polynomial.cc,v $
   Revision 1.2  2004/03/11 06:44:52  jechk
   Made the Sq function generic; added some TODO comments.

   Revision 1.1  2004/03/08 23:07:07  jechk
   Added polynomial files.



   Created 3/6/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/maths/polynomial.cc
  \brief A class representing nth-order polynomials.
*/

#include <cstdarg>
#include <iostream>

#include "leg/support/maths/polynomial.h"

namespace leg
{

namespace support
{

namespace maths
{

Polynomial::Polynomial ()
  : order (0)
{
  coeff.push_back (0.0);
}

Polynomial::Polynomial (unsigned int order, ...)
{
  va_list ap;

  va_start (ap, order);
  for (unsigned int i = 0; i <= order; ++i)
    {
      coeff.push_back (va_arg (ap, real));
    }
  va_end (ap);

  CalcOrder ();
}

Polynomial::Polynomial (std::vector<real> coeff)
  : coeff (coeff)
{
  CalcOrder ();
}

Polynomial::Polynomial (const Polynomial &p)
  : coeff (p.coeff),
    order (p.order)
{
}

Polynomial::~Polynomial ()
{
}

real
Polynomial::operator () (real x) const
{
  real p = coeff[order];

  for (unsigned int i = order; i >= 1; --i)
    {
      p = coeff[i - 1] + x * p;
    }

  return p;
}

Polynomial
Polynomial::Derivative (unsigned int n) const
{
  if (n == 0 || n > order)
    {
      return Polynomial ();
    }
  else
    {
      std::vector<real> new_coeff;

      for (unsigned int i = n; i <= order; ++i)
	{
	  new_coeff.push_back (coeff[i] * Permutation (i, n));
	}
      return Polynomial (new_coeff);
    }
}

Polynomial
Polynomial::Antiderivative (unsigned int n) const
{
  if (n == 0)
    {
      return Polynomial ();
    }
  else
    {
      std::vector<real> new_coeff;
      
      for (unsigned int i = 0; i < n; ++i)
	{
	  new_coeff.push_back (0.);
	}
      for (unsigned int i = 0; i <= order; ++i)
	{
	  new_coeff.push_back (coeff[i] / (real) Permutation (i + n, n));
	}
      return Polynomial (new_coeff);
    }
}

real
Polynomial::Integral (real a, real b) const
{
  Polynomial indef = Antiderivative ();
  return indef (b) - indef (a);
}

std::vector<real>
Polynomial::Roots (real epsilon)
{
  std::vector<real> roots;

  // TODO: write this.  I am considering using QR reduction which
  // would require a better matrix class.

  return roots;
}

void
Polynomial::CalcOrder ()
{
  order = coeff.size () - 1;
  while (order && !coeff[order])
    {
      coeff.pop_back ();
      --order;
    }
}

std::ostream &operator << (std::ostream &s, Polynomial p)
{
  bool first = false;
  real order = p.Order ();

  if (p[0] || order == 0)
    {
      s << p[0];
      first = true;
    }
  for (unsigned int i = 1; i <= order; ++i)
    {
      if (p[i])
	{
	  if (first)
	    {
	      s << " + ";
	    }
	  s << p[i] << " x^" << i;
	  first = true;
	}
    }
  
  return s;
}

}

}

}
