/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: quaternion.cc,v 1.3 2004/04/30 20:15:54 jechk Exp $
   $Log: quaternion.cc,v $
   Revision 1.3  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 1/22/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/maths/quaternion.cc
  \brief A class representing quaternions.
*/

#include <iostream>

#include "leg/support/maths/matrix.h"
#include "leg/support/maths/quaternion.h"
#include "leg/support/utils/error.h"

namespace leg
{

namespace support
{

namespace maths
{

Quaternion::Quaternion ()
  : a (0), b (0), c (0), d (0)
{
}

Quaternion::Quaternion (real a, real b, real c, real d)
  : a (a), b (b), c (c), d (d)
{
}

Quaternion::Quaternion (real angle, Vector<3> axis)
  : a (cos (angle / 2)),
    b (axis.x), c (axis.y), d (axis.z)
{
  real m = axis.Magnitude ();
  if (!m)
    {
      throw utils::Error ("Zero axis vector.");
      return;
    }
  real mult = sin (angle / 2) / m;
  b *= mult;
  c *= mult;
  d *= mult;
}

Quaternion::Quaternion (const Matrix<4, 4> &m)
{
  real t = m.Trace ();

  if (t > 1)
    {
      real st = sqrt (t);

      a = st / 2;
      b = (m (2, 1) - m (1, 2)) / (2 * st);
      c = (m (0, 2) - m (2, 0)) / (2 * st);
      d = (m (1, 0) - m (0, 1)) / (2 * st);
    }
  else
    {
      if (m (0, 0) >= m (1, 1) && m (0, 0) >= m (2, 2))
	{
	  b = sqrt (m (0, 0) - m (1, 1) - m (2, 2) + 1) / 2;
	  c = sqrt (m (1, 0) + m (0, 1)) / (4 * b);
	  d = sqrt (m (2, 0) + m (0, 2)) / (4 * b);
	  a = sqrt (m (2, 1) - m (1, 2)) / (4 * b);
	}
      else if (m (1, 1) >= m (0, 0) && m (0, 0) >= m (2, 2))
	{
	  c = sqrt (m (1, 1) - m (0, 0) - m (2, 2) + 1) / 2;
	  d = sqrt (m (2, 1) + m (1, 2)) / (4 * b);
	  b = sqrt (m (0, 1) + m (1, 0)) / (4 * b);
	  a = sqrt (m (0, 2) - m (2, 0)) / (4 * b);
	}
      else if (m (2, 2) >= m (0, 0) && m (2, 2) >= m (1, 1))
	{
	  d = sqrt (m (2, 2) - m (0, 0) - m (1, 1) + 1) / 2;
	  b = sqrt (m (0, 2) + m (2, 0)) / (4 * b);
	  c = sqrt (m (1, 2) + m (2, 1)) / (4 * b);
	  a = sqrt (m (1, 0) - m (0, 1)) / (4 * b);
	}
    }
}

Quaternion::Quaternion (const Quaternion &q)
  : a (q.a), b (q.b), c (q.c), d (q.d)
{
}

Quaternion::~Quaternion ()
{
}

Quaternion
Quaternion::operator *(const Quaternion &q) const
{
  return Quaternion (a * q.a - b * q.b - c * q.c - d * q.d,
		     a * q.b + b * q.a + c * q.d - d * q.c,
		     a * q.c - b * q.d + c * q.a + d * q.b,
		     a * q.d + b * q.c - c * q.b + d * q.a);
}

Quaternion
Quaternion::operator *= (const Quaternion &m)
{
  return *this = *this * m;
}

Quaternion
Quaternion::operator +(const Quaternion &q) const
{
  return Quaternion (a + q.a, b + q.b, c + q.c, d + q.d);
}

Quaternion
Quaternion::operator +=(const Quaternion &q)
{
  a += q.a;
  a += q.b;
  b += q.c;
  d += q.d;

  return *this;
}

Quaternion
Quaternion::operator -(const Quaternion &q) const
{
  return Quaternion (a - q.a, b - q.b, c - q.c, d - q.d);
}

Quaternion
Quaternion::operator -=(const Quaternion &q)
{
  a -= q.a;
  a -= q.b;
  b -= q.c;
  d -= q.d;

  return *this;
}

Quaternion
Quaternion::GetConjugate () const
{
  return Quaternion (a, -b, -c, -d);
}

real
Quaternion::AbsSq () const
{
  return Sq (a) + Sq (b) + Sq (c) + Sq (d);
}

real
Quaternion::Abs () const
{
  return sqrt (Abs ());
}

Quaternion
Quaternion::Inverse () const
{
  return GetConjugate () / AbsSq ();
}

real
Quaternion::Distance (const Quaternion &q) const
{
  Quaternion n = *this - q;
  return n.Abs ();
}

Vector<3>
Quaternion::Rotate (const Vector<3> &v) const
{
  Quaternion n = *this * Quaternion (0, v.x, v.y, v.z) * GetConjugate ();
  return Vector<3> (n.b, n.c, n.d);
}

std::ostream &operator << (std::ostream &s, Quaternion q)
{
  s << q.a << " + " << q.b << "i + " << q.c << "j + " << q.d << "k" << std::endl;

  return s;
}

}

}

}
