/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 03/27/05 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004-2005 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/model/formatleg.cc
  \brief leg model loader/saver.
*/

#include "leg/support/model/formatmd2.h"
#include "leg/support/model/modelclass.h"
#include "leg/support/model/meshes.h"
#include <fstream>
#include <string>

namespace leg
{
namespace support
{
namespace model
{

   void
   CalculateNormal (const GLfloat *const p, GLfloat *const n)
   {
      GLfloat a[3], b[3], result[3];
      GLfloat inv_length;

      a[0] = p[0] - p[3];
      a[1] = p[1] - p[4];
      a[2] = p[2] - p[5];
      b[0] = p[0] - p[6];
      b[1] = p[1] - p[7];
      b[2] = p[2] - p[8];

      result[0] = a[1] * b[2] - b[1] * a[2];
      result[1] = b[0] * a[2] - a[0] * b[2];
      result[2] = a[0] * b[1] - b[0] * a[1];

      inv_length = 1 / ((GLfloat)sqrt(result[0]*result[0] + result[1]*result[1] + result[2]*result[2]));

      n[0] = result[0] * inv_length;
      n[1] = result[1] * inv_length;
      n[2] = result[2] * inv_length;
   }

   float Md2Model::scale = .05;

   Md2Model::Md2Model()
   {
      percent = 0.2;
      state = idle;
   }

   void
   Md2Model::Default()
   {
      std::cout << "Setting to default." << std::endl;
      percent = 0.2;
      state = idle;
   }
   
   void
   Md2Model::Update()
   {
      vector_t *vList;        // current frame vertices
      vector_t *nextVList;    // next frame vertices
      int i;                  // index counter
      float x1, y1, z1;       // current frame point values
      float x2, y2, z2;       // next frame point values
      int start_frame;
      int end_frame;
 
      switch (state){
	 case idle:
	    start_frame = 0;
	    end_frame = 39;
	    break;
	 case crouch:
	    start_frame = 136;
	    end_frame = 154;
	    break;
	 case run:
	    start_frame = 40;
	    end_frame = 46;
	    break;
	 case jump:
	    start_frame = 67;
	    end_frame = 73;
	    break;
	 case hit:
	    start_frame = 47;
	    end_frame = 60;
	    break;
	 default:
	    start_frame = -1;
	    end_frame = -1;
	    break;
      }
 
      if ((start_frame > md2->currentFrame))
	 md2->currentFrame = start_frame;
      
      if ((start_frame < 0) || (end_frame < 0)){
	 std::cout << "Error in FormatMd2: negative frame" << std::endl;
	 throw std::exception();
      }
      
      if (start_frame >= md2->numFrames){
	 std::cout << "Error in FormatMd2: start frame too high" << std::endl;
	 std::cout << "start: " << start_frame << " ; max: " << md2->numFrames << std::endl;
	 throw std::exception();
      }
 
      if (end_frame >= md2->numFrames){
	 std::cout << "Error in FormatMd2: end frames too high" << std::endl;
	 throw std::exception();
      }
      
      if (md2->interpol >= 1.0){
	 md2->interpol = 0.0f;
	 md2->currentFrame++;
	 
	 if (md2->currentFrame >= end_frame)
	    md2->currentFrame = start_frame;
         
	 md2->nextFrame = md2->currentFrame + 1;
         
	 if (md2->nextFrame >= end_frame)
	    md2->nextFrame = start_frame;
      }
      
      vList = &md2->vertexList[md2->numVertices*md2->currentFrame];
      nextVList = &md2->vertexList[md2->numVertices*md2->nextFrame];

      if (!vList){
	 std::cout << "vList is NULL !" << std::endl;
	 throw std::exception();
      }

      if (!nextVList){
	 std::cout << "nextVList is NULL !" << std::endl;
	 throw std::exception();
      }
      
      Meshes *new_mesh = 0;
      Meshes *old_mesh = 0;

      std::list<Meshes*>::iterator mli;
      mli = mesh_list.begin();

      while (mli != mesh_list.end()){
	 old_mesh = *mli;
	 delete old_mesh;
	 mesh_list.erase (mli);
	 mli = mesh_list.begin();
      }
      
      new_mesh = new Meshes (md2->numTriangles);
      GLfloat n[3];
      
      for (i = 0; i < md2->numTriangles; ++i){
	 x1 = vList[md2->triIndex[i].meshIndex[0]].point[0];
	 z1 = -vList[md2->triIndex[i].meshIndex[0]].point[1];
	 y1 = vList[md2->triIndex[i].meshIndex[0]].point[2];
	 x2 = nextVList[md2->triIndex[i].meshIndex[0]].point[0];
	 z2 = -nextVList[md2->triIndex[i].meshIndex[0]].point[1];
	 y2 = nextVList[md2->triIndex[i].meshIndex[0]].point[2];
         
	 new_mesh->vert.d[9*i]   = (x1 + md2->interpol * (x2 - x1))*scale;
	 new_mesh->vert.d[9*i+1] = (y1 + md2->interpol * (y2 - y1))*scale;
	 new_mesh->vert.d[9*i+2] = (z1 + md2->interpol * (z2 - z1))*scale;

	 if (new_mesh->vert.d[9*i] < min_v.x)
	    min_v.x = new_mesh->vert.d[9*i];
	 else if (new_mesh->vert.d[9*i] > max_v.x)
	    max_v.x = new_mesh->vert.d[9*i];
	 if (new_mesh->vert.d[9*i+1] < min_v.y)
	    min_v.y = new_mesh->vert.d[9*i+1];
	 else if (new_mesh->vert.d[9*i+1] > max_v.y)
	    max_v.y = new_mesh->vert.d[9*i+1];
         if (new_mesh->vert.d[9*i+2] < min_v.z)
	    min_v.z = new_mesh->vert.d[9*i+2];
	 else if (new_mesh->vert.d[9*i+2] > max_v.z)
	    max_v.z = new_mesh->vert.d[9*i+2];

	 x1 = vList[md2->triIndex[i].meshIndex[2]].point[0];
	 z1 = -vList[md2->triIndex[i].meshIndex[2]].point[1];
	 y1 = vList[md2->triIndex[i].meshIndex[2]].point[2];
	 x2 = nextVList[md2->triIndex[i].meshIndex[2]].point[0];
	 z2 = -nextVList[md2->triIndex[i].meshIndex[2]].point[1];
	 y2 = nextVList[md2->triIndex[i].meshIndex[2]].point[2];
         
	 new_mesh->vert.d[9*i+3] = (x1 + md2->interpol * (x2 - x1))*scale;
	 new_mesh->vert.d[9*i+4] = (y1 + md2->interpol * (y2 - y1))*scale;
	 new_mesh->vert.d[9*i+5] = (z1 + md2->interpol * (z2 - z1))*scale;
         
	 if (new_mesh->vert.d[9*i+3] < min_v.x)
	    min_v.x = new_mesh->vert.d[9*i+3];
	 else if (new_mesh->vert.d[9*i+3] > max_v.x)
	    max_v.x = new_mesh->vert.d[9*i+3];
	 if (new_mesh->vert.d[9*i+4] < min_v.y)
	    min_v.y = new_mesh->vert.d[9*i+4];
	 else if (new_mesh->vert.d[9*i+4] > max_v.y)
	    max_v.y = new_mesh->vert.d[9*i+4];
         if (new_mesh->vert.d[9*i+5] < min_v.z)
	    min_v.z = new_mesh->vert.d[9*i+5];
	 else if (new_mesh->vert.d[9*i+5] > max_v.z)
	    max_v.z = new_mesh->vert.d[9*i+5];

	 x1 = vList[md2->triIndex[i].meshIndex[1]].point[0];
	 z1 = -vList[md2->triIndex[i].meshIndex[1]].point[1];
	 y1 = vList[md2->triIndex[i].meshIndex[1]].point[2];
	 x2 = nextVList[md2->triIndex[i].meshIndex[1]].point[0];
	 z2 = -nextVList[md2->triIndex[i].meshIndex[1]].point[1];
	 y2 = nextVList[md2->triIndex[i].meshIndex[1]].point[2];
         
	 new_mesh->vert.d[9*i+6] = (x1 + md2->interpol * (x2 - x1))*scale;
	 new_mesh->vert.d[9*i+7] = (y1 + md2->interpol * (y2 - y1))*scale;
	 new_mesh->vert.d[9*i+8] = (z1 + md2->interpol * (z2 - z1))*scale;
         
	 if (new_mesh->vert.d[9*i+6] < min_v.x)
	    min_v.x = new_mesh->vert.d[9*i+6];
	 else if (new_mesh->vert.d[9*i+6] > max_v.x)
	    max_v.x = new_mesh->vert.d[9*i+6];
	 if (new_mesh->vert.d[9*i+7] < min_v.y)
	    min_v.y = new_mesh->vert.d[9*i+7];
	 else if (new_mesh->vert.d[9*i+7] > max_v.y)
	    max_v.y = new_mesh->vert.d[9*i+7];
         if (new_mesh->vert.d[9*i+8] < min_v.z)
	    min_v.z = new_mesh->vert.d[9*i+8];
	 else if (new_mesh->vert.d[9*i+8] > max_v.z)
	    max_v.z = new_mesh->vert.d[9*i+8];

	 CalculateNormal(&new_mesh->vert.d[9*i], &n[0]);
	 
	 std::memcpy (&new_mesh->norm.d[9*i], &n[0], 3*sizeof (GLfloat));
	 std::memcpy (&new_mesh->norm.d[9*i+3],&n[0],3*sizeof (GLfloat));
	 std::memcpy (&new_mesh->norm.d[9*i+6],&n[0],3*sizeof (GLfloat));
	 
	 std::memset (&new_mesh->col.d[12*i], 0xffffffff, 12*sizeof (GLfloat));
	 
	 new_mesh->tex.d[6*i] = md2->st[md2->triIndex[i].stIndex[0]].s;
	 new_mesh->tex.d[6*i+1] = md2->st[md2->triIndex[i].stIndex[0]].t;
	 new_mesh->tex.d[6*i+2] = md2->st[md2->triIndex[i].stIndex[2]].s;
	 new_mesh->tex.d[6*i+3] = md2->st[md2->triIndex[i].stIndex[2]].t;
	 new_mesh->tex.d[6*i+4] = md2->st[md2->triIndex[i].stIndex[1]].s;
	 new_mesh->tex.d[6*i+5] = md2->st[md2->triIndex[i].stIndex[1]].t;
      }

      md2->interpol += percent;  // increase percentage of interpolation between frames
      mesh_list.push_back (new_mesh);
   }

   std::list<Meshes*>&
   Md2Model::GetMeshes()
   {
      return mesh_list;
   }



   FormatMd2::FormatMd2(): mdl (0)
   {
   }

   FormatMd2::FormatMd2 (const FormatMd2& v)
   {
   }

   FormatMd2::~FormatMd2()
   {
      if (mdl){
	 delete mdl;
	 mdl = 0;
      }
   }
   
   Md2Model&
   FormatMd2::Load (const std::string& f)
   {
      std::cout << "Loading md2 format" << std::endl;
      if (mdl){
	 delete mdl;
	 mdl = 0;
      }

      mdl = new Md2Model;
      mdl->md2 = new CMD2Model;
      mdl->md2->Load (f.c_str(),"kljlkj");

      return *mdl;
   }

   void
   FormatMd2::Save (const std::string& filename, Model& mdl)
   {
      throw std::exception();
   }

   BaseFormatLoader&
   FormatMd2::Clone()
   {
      throw std::exception();
   }
   
   void
   FormatMd2::Copy (const FormatMd2& f)
   {
      throw std::exception();
   }

}
}
}
