/* This file is part of GNU Libraries and Engines for Games.

   $Id: thread.cc,v 1.6 2004/04/30 20:15:54 jechk Exp $
   $Log: thread.cc,v $
   Revision 1.6  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.5  2004/03/10 20:44:01  jd
   support for new error handlings

   Revision 1.4  2004/03/04 07:22:36  jechk
   Clean up and fixes in Guile code.

   Revision 1.3  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 01/23/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/threads/thread.cc
  \brief First support of threads for Leg.
*/

#include "leg/support/threads/thread.h"
#include "leg/support/utils/errors.h"
#include <iostream>

#if GUILE
#include <libguile.h>
#endif

namespace leg
{
namespace support
{

using namespace utils;

namespace threads
{
   unsigned int Thread::count=0;

   static void* ThreadFunc (void* th)
   {
      if (!th)
	 throw Error ("leg::support::threads::threadFunc( void*): ERROR : pointer must not be null !");
      Thread* tmp= static_cast<Thread*> (th);
      tmp->Go (tmp->GetParameters ());
		
      return 0; //TODO: why returning null here ?
   }

#if GUILE
   static SCM GuileThreadFunc (void* th)
   {
     ThreadFunc (th);

     return SCM_BOOL_F;
   }

   static SCM
   CatchHandler (void *data, SCM tag, SCM throw_args)
   {
      std::string error = "Guile error in thread: ";
      if (SCM_SYMBOLP (tag))
         {
	    error += SCM_SYMBOL_CHARS (tag);
	 }
      throw Error (error);
   }
#endif // GUILE
   
   Thread::Thread ():	parameters (0),
			schedule_algo (normal),
			cancel_state (enable),
			cancel_type (deferred)
   {
      priority.priority= 0;
      priority.min= 0;
      priority.max= 0;
      ++count;
   }

   Thread::~Thread ()
   {
      if (pthread_equal (pthread_self (), thread))
	 {
	    pthread_cancel (thread);
	    pthread_exit (0);
	 }
      --count;
      if( count<0)
	 throw Error ("leg::support::threads::Thread::~Thread(): error in counting.");
   }

   void Thread::Create ()
   {
#if GUILE
      if (scm_initialized_p)
         {
	    SCM s = scm_spawn_thread (GuileThreadFunc, this, CatchHandler, NULL);
	    thread = scm_c_scm2thread (s);

	    return;
	 }
#endif
      int ret=0;

      ret= pthread_create (&thread, &attributes.GetAttributes (), ThreadFunc, this);

      if (ret!=0)
	 throw Error ("error while creating pthread", "leg::support::threads::Thread:Create()", ret);
   }

   void* Thread::Join () const
   {
      void ** returned=0;
      pthread_join (thread, returned);
      return *returned;
   }

   void 
   Thread::CalculatePriorityRange ()
   {
      priority.min= sched_get_priority_min (schedule_algo);
      priority.max= sched_get_priority_max (schedule_algo);
   }

   void Thread::SetPriority (int prior)
   {
      if( prior<priority.min || prior>priority.max)
	 throw Error ("leg::support::threads::Thread::setPriority (int): Error: priority is not between acceptable range.");
      
      ScheduleParam sp;
      attributes.GetScheduleParam (&sp);
      sp.sched_priority= priority.priority;
      attributes.SetScheduleParam (&sp);
      priority.priority= prior;
   }
   
   void Thread::SetScheduleConfig (const ScheduleAlgorithm sched_algo)
   {
      attributes.SetSchedulePolicy (sched_algo);
      schedule_algo= sched_algo;
      CalculatePriorityRange ();
      SetPriority (priority.min);
   }

   bool Thread::OwnThread() const
   {
      return (pthread_self ()==thread);
   }
   
   bool Thread::SetCancelState (const ThreadCancelState st)
   {
      if (!OwnThread ())
	 return false;
      
      pthread_setcancelstate (st, &cancel_state);
      cancel_state= st;

      return true;
   }

   bool Thread::SetCancelType (const ThreadCancelType ty)
   {
      if (!OwnThread ())
	 return false;
      pthread_setcanceltype (ty, &cancel_type);
      cancel_type= ty;

      return true;
   }

   void Thread::TestCancel () const
   {
      if (!OwnThread ())
	 return;
      pthread_testcancel ();
   }
}
}
}
