#!/bin/bash
# prepare_lfs-install-cdrom.sh
#
# Based on 'Easy Boot CD of your LFS'-Hint written by
#	Thomas Foecking <thomas@foecking.de>
#	Christian Hesse <mail@earthworm.de>
#
source scripts/scripts_settings

cp -R ../lfs-books/ ../lfs-commands/ ../lfs-packages/ ../lfs-install/ $DIRECTORY_LFS/root/
cp $DIRECTORY_DOWNLOAD/syslinux* $DIRECTORY_LFS/usr/src/

# Move /dev /var /tmp /root /home to /fake/needwrite
# --------------------------------------------------
mkdir -p $DIRECTORY_LFS/fake/{needwrite,ramdisk}

cd $DIRECTORY_LFS/
mv dev/ etc/ mnt/ var/ tmp/ home/ fake/needwrite/


# Create symlinks /... -> /fake/needwrite/...
# -------------------------------------------
cd $DIRECTORY_LFS/
ln -s fake/needwrite/dev dev
ln -s fake/needwrite/etc etc
ln -s fake/needwrite/mnt mnt
ln -s fake/needwrite/var var
ln -s fake/needwrite/tmp tmp
ln -s fake/needwrite/home home


# Create boot script which mounts the ramdisk
# -------------------------------------------
cat > $DIRECTORY_LFS/etc/rc.d/init.d/create_ramdisk << "EOF"
#!/bin/sh

dev_ram=/dev/ram0
dir_ramdisk=/fake/ramdisk
dir_needwrite=/fake/needwrite

source /etc/rc.d/init.d/functions

case "$1" in
	start)
		echo -n "Creating ext2fs on $dev_ram ...              "
		/sbin/mke2fs -m 0 -i 1024 -q $dev_ram > /dev/null 2>&1
		evaluate_retval
		sleep 1
		echo -n "Mounting ramdisk on $dir_ramdisk ...         "
		mount -n $dev_ram $dir_ramdisk
		evaluate_retval
		sleep 1
		echo -n "Copying files to ramdisk ...                 "
		cp -dpR $dir_needwrite/* $dir_ramdisk > /dev/null 2>&1
		evaluate_retval
		sleep 1
		echo -n "Remount ramdisk to $dir_needwrite ...        "
		umount -n $dir_ramdisk > /dev/null 2>&1
		sleep 1
		mount -n $dev_ram $dir_needwrite
		sleep 1
		;;
	*)
		echo "Usage: $0 {start}"
		exit 1
		;;
esac
EOF

chmod 0755 $DIRECTORY_LFS/etc/rc.d/init.d/create_ramdisk

cd $DIRECTORY_LFS/etc/rc.d/rcsysinit.d
ln -s ../init.d/create_ramdisk S00create_ramdisk


# Install the bootloader isolinux
# -------------------------------
cd $DIRECTORY_LFS/usr/src
tar xfvz syslinux*

mkdir $DIRECTORY_LFS/isolinux
cp syslinux*/isolinux.bin $DIRECTORY_LFS/isolinux/
rm -rf syslinux*

mv $DIRECTORY_LFS/boot/* $DIRECTORY_LFS/isolinux

cd $DIRECTORY_LFS/
rmdir boot
ln -s isolinux boot

cat > $DIRECTORY_LFS/isolinux/isolinux.cfg << "EOF"
default bootcd

label bootcd
	kernel lfskernel
	append initrd=initrd.gz root=/dev/ram0 init=/linuxrc ramdisk_size=16384
EOF


# Create initial ramdisk
# ----------------------
cat > $DIRECTORY_LFS/etc/fstab << "EOF"
# Begin /etc/fstab

# filesystem   mount-point fs-type    options     dump    fsck-order

proc           /proc       proc       defaults    0       0
#devpts         /dev/pts    devpts     gid=4,mode=620 0    0
#usbdevfs       /proc/bus/usb usbdevfs defaults    0       0

# End /etc/fstab
EOF

rm $DIRECTORY_LFS/etc/rc.d/rc3.d/S20network
rm $DIRECTORY_LFS/etc/rc.d/rc0.d/K80network
rm $DIRECTORY_LFS/etc/rc.d/rc6.d/K80network
rm $DIRECTORY_LFS/etc/rc.d/rcsysinit.d/S40mountfs
rm $DIRECTORY_LFS/etc/rc.d/rcsysinit.d/S30checkfs

dd if=/dev/zero of=$DIRECTORY_LFS/boot/initrd bs=1024 count=6144
mke2fs -m 0 -i 1024 -F $DIRECTORY_LFS/boot/initrd

mount -o loop $DIRECTORY_LFS/boot/initrd $DIRECTORY_LFS/mnt
cd $DIRECTORY_LFS/mnt
mkdir bin lib dev proc mnt
cp $DIRECTORY_LFS/bin/{bash,mount,grep,umount,echo} bin/
cp $(find $DIRECTORY_LFS -name "test" -type f) bin/
cp $(find $DIRECTORY_LFS -name "chroot" -type f) bin/
cp $(find $DIRECTORY_LFS -name "pivot_root" -type f) bin/
cp $DIRECTORY_LFS/lib/{libncurses.so.5,libdl.so.2,libc.so.6,ld-linux.so.2} lib/
cp -dR $DIRECTORY_LFS/dev/{console,hd?,scd*,null,ram*} dev/
ln -s bash bin/sh
ln -s test bin/[

cat > $DIRECTORY_LFS/mnt/linuxrc << "EOF"
#!/bin/sh

#if you want use devfs and don't have /dev/hd?, /dev/scd* then
#you must mount your devfs here:


#ID is the volume id / label of the LFS boot CD if you use /bin/isoinfo
#or ID is a file in root of the LFS boot CD
EOF

echo "ID=\"LFS_$(date +%Y%m%d)\"" >> $DIRECTORY_LFS/mnt/linuxrc

cat >> $DIRECTORY_LFS/mnt/linuxrc << "EOF"
TMP_MOUNT="/mnt"

PATH="/bin:/sbin:/usr/bin:/usr/sbin"

#this script searches for cdrom devices and then tries to find
#the LFS boot CD in order to mount it as / (rootfs)
#it detects: IDE devices hda - hdn and SCSI devices scd0 - scd99
#or when using devfs: IDE/SCSI devices cdrom0 - cdrom99

#you need following programs:
#- /bin/isoinfo (optional)
#- /bin/sh (e.g. bash)
#- echo
#- [ (which is linked to test)
#- mount
#- umount
#- grep
#- pivot_root
#- chroot

#you need following devices:
#- /dev/hd* or /dev/scd* or /dev/cdrom/cdrom*
#- /dev/null

#you need following directories:
#- /proc
#- $TMP_MOUNT

if [ -e "/bin/isoinfo" ]; then
  CHECK_TYPE="isoinfo"
else
  CHECK_TYPE="try_mount"
fi

if [ ! -d "/proc/" ]; then
  mkdir /proc
fi

mount -n proc /proc -t proc

#1. search for cdrom devices and add them to CDROM_LIST

CDROM_LIST=""

#do we have devfs? (then its very easy)
if [ -d "/dev/cdrom/" ]; then
  for cdrom_device in /dev/cdrom/cdrom[0-99]
  do
    if [ -e "$cdrom_device" ]; then
      CDROM_LIST="$CDROM_LIST $cdrom_device"
    fi
  done
else #we search in proc tree for ide cdrom devices
  for ide_channel in /proc/ide/ide[0-9]
  do
    if [ ! -d "$ide_channel" ]; then
     break
    fi
    for ide_device in hda hdb hdc hdd hde hdf hdg hdh hdi hdj hdk hdl hdm hdn
    do
      device_media_file="$ide_channel/$ide_device/media"
      if [ -e "$device_media_file" ]; then
        grep -i "cdrom" $device_media_file > /dev/null 2>&1
        if [ $? -eq 0 ]; then
          if [ -e "/dev/$ide_device" ]; then
            CDROM_LIST="$CDROM_LIST /dev/$ide_device"
          fi
        fi
      fi
    done
  done
  for scsi_cdrom in /dev/scd[0-99]
  do
    if [ -e "$scsi_cdrom" ]; then
      CDROM_LIST="$CDROM_LIST $scsi_cdrom"
    fi
  done
fi

#2. now we try to find the LFS boot CD (we use ID as identification)

LFS_CDROM_DEVICE=""
echo "Using $CHECK_TYPE"
for cdrom_device in $CDROM_LIST
do
  echo -n "Checking $cdrom_device ... "

  if [ "$CHECK_TYPE" = "try_mount" ]; then
    mount -n -t iso9660 $cdrom_device $TMP_MOUNT > /dev/null 2>&1
    media_found=$?
  fi
  if [ "$CHECK_TYPE" = "isoinfo" ]; then
    isoinfo -d -i $cdrom_device > /dev/null 2>&1
    media_found=$?
    if [ $? -eq 2 ]; then
      isoinfo -V $cdrom_device > /dev/null 2>&1
      media_found=$?
    fi
  fi

  if [ $media_found -eq 0 ]; then

    echo -n "media found"
    if [ "$CHECK_TYPE" = "try_mount" ]; then
      [ -e "$TMP_MOUNT/$ID" ]
      media_lfs=$?
    fi
    if [ "$CHECK_TYPE" = "isoinfo" ]; then
      isoinfo -d -i $cdrom_device | grep -i "Volume id:" | grep "$ID" \
      > /dev/null 2>&1
      media_lfs=$?
      if [ $? -eq 2 ]; then
        isoinfo -V $cdrom_device | grep "$ID" > /dev/null 2>&1
        media_lfs=$?
      fi
    fi

    if [ "$CHECK_TYPE" = "try_mount" ]; then
      umount -n $cdrom_device > /dev/null 2>&1
    fi

    if [ $media_lfs -eq 0 ]; then
      echo ", LFS boot CD found. Ready!"
      LFS_CDROM_DEVICE="$cdrom_device"
      break;
    else
      echo ", not LFS boot CD."
    fi

  else
    echo "no media "
  fi
done


#3. mount LFS CD as / (root fs)
if [ "$LFS_CDROM_DEVICE" = "" ]; then

  echo "No LFS boot CD found!!!"
  exit 1

else

  echo "Booting from $LFS_CDROM_DEVICE ...                  "

  mount -n -o ro -t iso9660 $LFS_CDROM_DEVICE $TMP_MOUNT
  cd $TMP_MOUNT
  pivot_root . mnt
  umount -n /mnt/proc >/dev/null 2>&1
  exec chroot . sh -c 'umount -n /mnt >/dev/null 2>&1;\
  exec -a init.new /sbin/init 3'\
  <dev/console >dev/console 2>&1

fi
EOF

chmod 0755 $DIRECTORY_LFS/mnt/linuxrc

cd $DIRECTORY_LFS/
umount $DIRECTORY_LFS/mnt
gzip $DIRECTORY_LFS/boot/initrd


# Burn the Boot CD
# ----------------

touch $DIRECTORY_LFS/LFS_$(date +%Y%m%d)

cd $DIRECTORY_LFS/
mkisofs -R -l -L -D -b isolinux/isolinux.bin -c isolinux/boot.cat -no-emul-boot -boot-load-size 4 -boot-info-table -V "LFS_$(date +%Y%m%d)" $DIRECTORY_LFS > ../lfs-install-cdrom_$(date +%Y%m%d).iso
