/*
 * This file contains the private implementation of libbubblemon.
 * Libbubblemon is a library for making bubbling load monitors for
 * different platforms.  See for example
 * "http://www.nongnu.org/bubblemon/".
 *
 *  Copyright (C) 1999-2000 Johan Walles - d92-jwa@nada.kth.se
 *  http://www.nongnu.org/libbubblemon/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

// We want to use roundf() and truncf()
#define _ISOC99_SOURCE

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>
#include <math.h>

#include "bubblemon.h"
#include "bubblemon_private.h"
#include "msgInBottle.c"

// Get a timestamp with millisecond precision
static timestamp_t getCurrentTime(void)
{
  struct timeval now;
  int problemsReadingTime;
  
  problemsReadingTime = gettimeofday(&now, NULL);
  assert(problemsReadingTime == 0);
  
  return now.tv_usec / 1000 + now.tv_sec * 1000;
}

static bubblemon_color_t constant2color(const unsigned int constant)
{
  bubblemon_color_t returnMe;
  
  returnMe.components.r = (constant >> 24) & 0xff;
  returnMe.components.g = (constant >> 16) & 0xff; 
  returnMe.components.b = (constant >> 8)  & 0xff; 
  returnMe.components.a = (constant >> 0)  & 0xff;
  
  return returnMe;
}

// Create a new bubblemon, start by calling this function
//
// Returns NULL on out of memory
bubblemon_t bubblemon_create(void)
{
  bubblemon_private_t *this =
    (bubblemon_private_t *)calloc(1, sizeof(bubblemon_private_t));
  bubblemon_picture_t tempPic;
  int result;
  
  if (this == NULL) {
    return NULL;
  }
  
  this->maxWaterColor = constant2color(MAXWATERCOLOR);
  this->minWaterColor = constant2color(MINWATERCOLOR);
  this->maxAirColor = constant2color(MAXAIRCOLOR);
  this->minAirColor = constant2color(MINAIRCOLOR);
  this->growthColor1 = constant2color(GROWTHCOLOR1);
  this->growthColor2 = constant2color(GROWTHCOLOR2);
  
  this->floaterState = GONE;
  
  tempPic.width = msgInBottle.width;
  tempPic.height = msgInBottle.height;
  tempPic.pixels = (bubblemon_color_t *)msgInBottle.pixel_data;
  result = bubblemon_setFloaterImage(this, &tempPic,
				     msgInBottle.width / 2,
				     msgInBottle.height / 2);
  
  // Out of memory setting the floater image
  if (result != 0) {
    free(this);
    this = NULL;
    
    return NULL;
  }
  
  this->lastUpdate = getCurrentTime();
  
  return this;
}

// The amount parameter is 0-255
static inline bubblemon_color_t interpolateColor(const bubblemon_color_t c1,
						 const bubblemon_color_t c2,
						 const int amount)
{
  int a, r, g, b;
  bubblemon_color_t returnme;

  assert(amount >= 0 && amount < 256);

  r = ((((int)c1.components.r) * (255 - amount)) +
       (((int)c2.components.r) * amount)) / 255;
  g = ((((int)c1.components.g) * (255 - amount)) +
       (((int)c2.components.g) * amount)) / 255;
  b = ((((int)c1.components.b) * (255 - amount)) +
       (((int)c2.components.b) * amount)) / 255;
  a = ((((int)c1.components.a) * (255 - amount)) +
       (((int)c2.components.a) * amount)) / 255;

  /*
  assert(a >= 0 && a <= 255);
  assert(r >= 0 && r <= 255);
  assert(g >= 0 && g <= 255);
  assert(b >= 0 && b <= 255);

  assert(((c1.components.a <= a) && (a <= c2.components.a)) ||
	 ((c1.components.a >= a) && (a >= c2.components.a)));
  assert(((c1.components.r <= r) && (r <= c2.components.r)) ||
	 ((c1.components.r >= r) && (r >= c2.components.r)));
  assert(((c1.components.g <= g) && (g <= c2.components.g)) ||
	 ((c1.components.g >= g) && (g >= c2.components.g)));
  assert(((c1.components.b <= b) && (b <= c2.components.b)) ||
	 ((c1.components.b >= b) && (b >= c2.components.b)));
  */
  
  returnme.components.r = r;
  returnme.components.g = g;
  returnme.components.b = b;
  returnme.components.a = a;

  /*
  assert((amount != 0)   || (returnme.value == c1.value));
  assert((amount != 255) || (returnme.value == c2.value));
  */
  
  return returnme;
}

// Set the width of all our internal state
//
// Returns:
// 0 on success
// ENOMEM (as defined in <errno.h>) on out of memory
static int setSize(bubblemon_private_t *this, int newWidth, int newHeight)
{
  int x;
  
  assert(this != NULL);
  assert(newWidth > 0);
  
  if (newWidth == this->width &&
      newHeight == this->height)
  {
    return 0;
  }
  this->width = newWidth;
  this->height = newHeight;

  if (this->antialiasingMap != NULL) {
    free(this->antialiasingMap);
    this->antialiasingMap = NULL;
  }
  if (this->waterLevels != NULL) {
    free(this->waterLevels);
    this->waterLevels = NULL;
  }
  if (this->bubbles != NULL) {
    free(this->bubbles);
    this->bubbles = NULL;
    this->nBubbles = 0;
    this->maxBubbles = 0;
    this->createNNewBubbles = 0;
  }
  if (this->weeds != NULL) {
    free(this->weeds);
    this->weeds = NULL;
    this->msecsToNextWeedUpdate = 0;
    this->lastUpdatedWeed = 0;
  }

  assert(this->antialiasingMap == NULL);
  assert(this->waterLevels == NULL);
  assert(this->bubbles == NULL);
  assert(this->weeds == NULL);

  this->antialiasingMap =
    (antialias_t *)malloc(this->width * this-> height * sizeof(antialias_t));
  if (this->antialiasingMap == NULL) {
    goto antialiasingOOM;
  }
  
  this->waterLevels =
    (waterLevel_t *)calloc(this->width,
			   sizeof(waterLevel_t));
  if (this->waterLevels == NULL) {
    goto waterLevelsOOM;
  }
  
  this->maxBubbles = (this->width * this->height) / 10;
  this->bubbles =
    (bubble_t *)malloc(this->maxBubbles *
		       sizeof(bubble_t));
  if (this->bubbles == NULL) {
    goto bubblesOOM;
  }
  
  this->weeds =
    (weed_t *)calloc(this->width,
		     sizeof(weed_t));
  if (this->weeds == NULL) {
    goto weedsOOM;
  }
  // Colorize the weeds
  for (x = 0; x < this->width; x++) {
    static int stripey = 0;
    
    this->weeds[x].color =
      interpolateColor(this->growthColor1, this->growthColor2,
		       (rand() % 156) + stripey);
    stripey = 100 - stripey;
  }
  
  return 0;
  
 weedsOOM:
  free(this->bubbles);
  this->bubbles = NULL;
 bubblesOOM:
  this->maxBubbles = 0;
  this->nBubbles = 0;
  
  free(this->waterLevels);
  this->waterLevels = NULL;
 waterLevelsOOM:
  free(this->antialiasingMap);
  this->antialiasingMap = NULL;
 antialiasingOOM:
  this->width = 0;
  this->height = 0;
  
  return ENOMEM;
}

static void updateWaterlevels(bubblemon_private_t *this, int msecsDelta)
{
  float dt = msecsDelta / TIME_SLOWDOWN;
  int x;
  float waterLevels_goal; // Where is the surface heading?
  float waterLevels_max;  // How high can the surface be?
  
  assert(this != NULL);
  assert(msecsDelta > 0);

  waterLevels_max = this->height;
  
  waterLevels_goal =
    (((float)this->targetWaterlevel) / 100.0) * waterLevels_max;
  
  this->waterLevels[0].y = waterLevels_goal;
  this->waterLevels[this->width - 1].y = waterLevels_goal;
  
  for (x = 1; x < (this->width - 1); x++)
  {
    // Accelerate the current waterlevel towards its goal value
    float current_waterlevel_goal = (this->waterLevels[x - 1].y +
				     this->waterLevels[x + 1].y) / 2.0;
    
    this->waterLevels[x].dy +=
      (current_waterlevel_goal - this->waterLevels[x].y) * dt * VOLATILITY;
    
    this->waterLevels[x].dy *= VISCOSITY;
    
    if (this->waterLevels[x].dy > SPEED_LIMIT)
      this->waterLevels[x].dy = SPEED_LIMIT;
    else if (this->waterLevels[x].dy < -SPEED_LIMIT)
      this->waterLevels[x].dy = -SPEED_LIMIT;
  }
  
  for (x = 1; x < (this->width - 1); x++)
  {
    // Move the current water level
    this->waterLevels[x].y += this->waterLevels[x].dy * dt;
    
    if (this->waterLevels[x].y > waterLevels_max)
    {
      // Stop the wave if it hits the ceiling...
      this->waterLevels[x].y = waterLevels_max;
      this->waterLevels[x].dy = 0.0;
    }
    else if (this->waterLevels[x].y < 0.0)
    {
      // ... or the floor.
      this->waterLevels[x].y = 0.0;
      this->waterLevels[x].dy = 0.0;
    }
  }
}

static void updateBubbles(bubblemon_private_t *this, int msecsDelta)
{
  int i;
  int x;
  
  float dt = msecsDelta / TIME_SLOWDOWN;
  
  assert(this != NULL);
  assert(msecsDelta > 0);
  
  // How many new bubbles shall we create this turn?
  // The width * 2.0 is because there are two equally wide layers
  this->createNNewBubbles +=
    (((float)this->bubblingIntensity) / 100.0 *
     ((float)this->width) * 2.0 *
     ((float)msecsDelta) / 1000.0) *
    MAX_BUBBLE_INTENSITY;

  // Create a suitable number of new bubbles
  while (this->createNNewBubbles >= 1.0)
  {
    if (this->nBubbles >= this->maxBubbles) {
      this->createNNewBubbles = 0.0f;
      break;
    }
    
    // We don't allow bubbles on the edges 'cause we'd have to clip them
    this->bubbles[this->nBubbles].x = (rand() % (this->width - 2)) + 1;
    this->bubbles[this->nBubbles].y = -1.0;
    this->bubbles[this->nBubbles].dy = 0.0;
    
    // Alternate between creating foreground and background bubbles
    this->bubbles[this->nBubbles].layer =
      (this->lastNewBubbleLayer == BACKGROUND) ? FOREGROUND : BACKGROUND;
    this->lastNewBubbleLayer = this->bubbles[this->nBubbles].layer;
    
    if (RIPPLES != 0.0) {
      int bubbleX = this->bubbles[this->nBubbles].x;
      // Raise the water level above where the bubble is created
      if ((bubbleX - 2) >= 0)
	this->waterLevels[bubbleX - 2].y += RIPPLES;
      this->waterLevels[bubbleX - 1].y   += RIPPLES;
      this->waterLevels[bubbleX].y       += RIPPLES;
      this->waterLevels[bubbleX + 1].y   += RIPPLES;
      if ((bubbleX + 2) < this->width)
	this->waterLevels[bubbleX + 2].y += RIPPLES;
    }
    
    // Count the new bubble
    this->nBubbles++;
    this->createNNewBubbles--;
  }
  
  // Move the bubbles
  for (i = 0; i < this->nBubbles; i++)
  {
    // Accelerate the bubble upwards
    this->bubbles[i].dy -= GRAVITY * dt;
    
    // Move the bubble vertically
    this->bubbles[i].y +=
      this->bubbles[i].dy * dt *
      ((this->bubbles[i].layer == BACKGROUND) ? BGBUBBLE_SPEED : 1.0);
    
    // Did we lose it?
    if (this->bubbles[i].y > this->waterLevels[this->bubbles[i].x].y)
    {
      if (RIPPLES != 0.0)
      {
        // Lower the water level around where the bubble is
        // about to vanish
        this->waterLevels[this->bubbles[i].x - 1].y -= RIPPLES;
        this->waterLevels[this->bubbles[i].x].y     -= 3 * RIPPLES;
        this->waterLevels[this->bubbles[i].x + 1].y -= RIPPLES;
      }
      
      // We just lost it, so let's forget about it by over-writing it
      // with the last bubble
      if (i != this->nBubbles - 1) {
	memcpy(&this->bubbles[i],
	       &this->bubbles[this->nBubbles - 1],
	       sizeof(bubble_t));
      }
      this->nBubbles--;
      
      // We must not forget to check the previously last bubble, which
      // is now the current bubble.
      i--;
      continue;
    }
  }

  // Rippling can move waterlevels out of bounds
  for (x = 0; x < this->width; x++) {
    if (this->waterLevels[x].y < 0) {
      this->waterLevels[x].y = 0;
    } else if (this->waterLevels[x].y >= this->height) {
      this->waterLevels[x].y = this->height;
    }
  }
}

static void updateFloater(bubblemon_private_t *this, int msecsDelta)
{
  float dt = msecsDelta / TIME_SLOWDOWN;
  float gravityDdy;
  float dragDdy;
  int isInWater;
  
  assert(this != NULL);
  assert(msecsDelta > 0);

  switch (this->floaterState) {
  case GONE:
    // Drop a new floater if one is supposed to be visible
    if (this->floaterVisible) {
      this->floaterY =
	this->height +
	this->floaterImage.height -
	this->floaterYCenter;
      this->floaterDy = -0.5;  // Shove the floater down so it looks
			       // like it was dropped from some
			       // altitude
      this->floaterState = FALLING;
    }
    break;
    
  case FALLING:
    // Start FLOATING if all of the floater image has been shown
    if (this->floaterY + this->floaterYCenter < this->height) {
      this->floaterState = FLOATING;
    }
    break;
    
  case FLOATING:
    // Start SINKING if the floater is supposed to go away
    if (!this->floaterVisible) {
      this->floaterState = SINKING;
    }
    break;
    
  case SINKING:
    // Become GONE if the floater goes out of screen...
    if (this->floaterY + this->floaterYCenter < 0.0) {
      this->floaterState = GONE;
    } else if (this->floaterVisible) {
      // ... or go back to FLOATING if the floater is supposed to be
      // visible again
      this->floaterState = FLOATING;
    }
    break;
    
  default:
    assert("Internal error" == 0);
  }

  isInWater =
    this->floaterY <= this->waterLevels[this->width / 2].y;
  gravityDdy = GRAVITY * dt;
  
  switch (this->floaterState) {
  case GONE:
    return;
    
  case FALLING:
  case SINKING:
    // Intentionally left blank
    break;
    
  case FLOATING:
    if (isInWater) {
      gravityDdy = -gravityDdy;
    }
    break;
  }
  
  if (isInWater) {
    dragDdy = (this->floaterDy * this->floaterDy) * FLOATER_DRAG;
    // Drag counters the floater's speed
    if (this->floaterDy > 0.0) {
      dragDdy = -dragDdy;
    }
  } else {
    dragDdy = 0.0;
  }

  this->floaterDy += gravityDdy + dragDdy;
  
  // Move the floater vertically
  this->floaterY += this->floaterDy * dt;
}

static void addNourishment(bubblemon_private_t *this,
			   weed_t *weed, int percentage)
{
  float heightLimit;
  assert(this != NULL);
  assert(weed != NULL);
  assert(percentage >= 0);
  assert(percentage <= 100);
  
  // How high do the weeds get at most?
  heightLimit = (this->height * WEED_HEIGHT) / 100.0;
  
  weed->nourishment += (heightLimit * percentage) / 100.0;
  
  if (weed->nourishment + weed->height > heightLimit) {
    weed->nourishment = heightLimit - weed->height;
  }
}

static void updateGrowth(bubblemon_private_t *this, int msecsDelta)
{
  int x;

  assert(this != NULL);
  assert(msecsDelta > 0);
  
  // If enough time has elapsed...
  this->msecsToNextWeedUpdate -= msecsDelta;
  while (this->msecsToNextWeedUpdate <= 0)
  {
    // ... update the nourishment level of our next weed
    this->lastUpdatedWeed--;
    if ((this->lastUpdatedWeed <= 0) ||
	(this->lastUpdatedWeed >= this->width))
    {
      this->lastUpdatedWeed = this->width - 1;
    }
    
    // Distribute the nourishment over several weeds
    if (this->lastUpdatedWeed > 0)
    {
      addNourishment(this,
		     &(this->weeds[this->lastUpdatedWeed - 1]),
		     (this->growthIntensity * 8) / 10);
    }
    addNourishment(this,
		   &(this->weeds[this->lastUpdatedWeed]),
		   this->growthIntensity);
    if (this->lastUpdatedWeed < (this->width - 1))
    {
      addNourishment(this,
		     &(this->weeds[this->lastUpdatedWeed + 1]),
		     (this->growthIntensity * 8) / 10);
    }
    
    this->msecsToNextWeedUpdate += WEED_INTERVAL;
  }
  
  // For all weeds...
  for (x = 0; x < this->width; x++) {
    // ... grow / shrink them according to their nourishment level
    float speed;
    float delta;
    
    if (this->weeds[x].nourishment <= 0.0)
    {
      speed = -WEED_MINSPEED;
    }
    else
    {
      speed = this->weeds[x].nourishment * WEED_SPEEDFACTOR;
      if (speed > WEED_MAXSPEED)
      {
	speed = WEED_MAXSPEED;
      }
      else if (speed < WEED_MINSPEED)
      {
	speed = WEED_MINSPEED;
      }
    }
    
    delta = (speed * msecsDelta) / 1000.0;
    if (delta > this->weeds[x].nourishment)
    {
      delta = this->weeds[x].nourishment;
    }
    
    if (delta > 0.0)
    {
      this->weeds[x].nourishment -= delta;
    }
    this->weeds[x].height += delta;
    if (this->weeds[x].height < 0.0)
    {
      this->weeds[x].height = 0.0;
    }
  }
}

static void updateColorIntensity(bubblemon_private_t *this, int msecsDelta)
{
  float secsDelta = ((float)msecsDelta) / 1000.0f;
  // 100.0f is the number of color intensity levels
  float maxDShades = secsDelta / (COLOR_CHANGE_TIME / 100.0f);
  
  float dShades = fabsf(this->colorIntensity -
			this->colorIntensityGoal);
  
  if (dShades > maxDShades) {
    dShades = maxDShades;
  }
  if (this->colorIntensityGoal < this->colorIntensity) {
    dShades = -dShades;
  }
  
  this->colorIntensity += dShades;

  assert(this->colorIntensity >= 0.0f);
  assert(this->colorIntensity <= 100.0f);
}

static void updatePhysicsDelta(bubblemon_private_t *this, int msecsDelta)
{
  assert(this != NULL);
  assert(msecsDelta >= 0);
  
  if (msecsDelta == 0) {
    return;
  }

  updateColorIntensity(this, msecsDelta);
  updateWaterlevels(this, msecsDelta);
  updateBubbles(this, msecsDelta);
  updateFloater(this, msecsDelta);
  updateGrowth(this, msecsDelta);
}

// Move our universe around
static void updatePhysics(bubblemon_private_t *this)
{
  timestamp_t now;
  timestamp_t msecsSinceLastUpdate;
  assert(this != NULL);

  now = getCurrentTime();
  msecsSinceLastUpdate = now - this->lastUpdate;
  assert(msecsSinceLastUpdate >= 0);
  
  // If we need to catch up, don't catch up to more than
  // MINIMUM_FRAMERATE frames per second
  if (msecsSinceLastUpdate > 1000 / MINIMUM_FRAMERATE) {
    msecsSinceLastUpdate = 1000 / MINIMUM_FRAMERATE;
    this->lastUpdate = now - 1000 / MINIMUM_FRAMERATE;
  }
  
  // Update the physics at least as often as the visual frame rate
  while (msecsSinceLastUpdate > MSECS_PER_PHYSICS_FRAME) {
    updatePhysicsDelta(this, MSECS_PER_PHYSICS_FRAME);
    
    msecsSinceLastUpdate -= MSECS_PER_PHYSICS_FRAME;
    this->lastUpdate += MSECS_PER_PHYSICS_FRAME;
  }
  
  if (msecsSinceLastUpdate > 0) {
    updatePhysicsDelta(this, msecsSinceLastUpdate);
    this->lastUpdate = now;
  }
  
  assert(this->lastUpdate == now);
}

static void plotBubble(bubblemon_private_t *this,
		       bubble_t *bubble)
{
  int antialias;
  int x0, y0, y;
  antialias_t *pixel;
  
  y = roundf(bubble->y);
  x0 = bubble->x - 1;
  y0 = y + 1;
  
  pixel = &this->antialiasingMap[(this->width * (this->height - y0 - 1)) + x0];
  y = y0;
  
  antialias = (ANTIALIASING - 1) / 2;
  
  // Draw the top row of the bubble
  if (y >= 0 && y < this->height) {
    *(pixel++) -= antialias;
    *(pixel++) = 0;  // Air
    *(pixel++) -= antialias;
    pixel += this->width - 3;
  } else {
    pixel += this->width;
  }
  y--;
  
  // Draw the middle row of the bubble
  if (y >= 0 && y < this->height) {
    *(pixel++) = 0;  // Air
    *(pixel++) = 0;  // Air
    *(pixel++) = 0;  // Air
    pixel += this->width - 3;
  } else {
    pixel += this->width;
  }
  y--;
  
  if (y >= 0 && y < this->height) {
    *(pixel++) -= antialias;
    *(pixel++) = 0;  // Air
    *(pixel++) -= antialias;
  }
}

static void renderAirAndWater(bubblemon_private_t *this,
			      bubblemon_picture_t *picture,
			      layerType_t layer)
{
  int x;
  int i;
  int width, height;
  int nPixels;
  bubblemon_color_t colors[ANTIALIASING];
  bubblemon_color_t airColor;
  bubblemon_color_t waterColor;
  
  assert(this != NULL);
  assert(picture != NULL);

  width = this->width;
  height = this->height;
  
  // Render the air and the water
  for (x = 0; x < width; x++) {
    int y;
    int roundedWaterLevel;
    waterLevel_t *waterLevel = &this->waterLevels[x];
    antialias_t *pixel = &this->antialiasingMap[(width * (height - 1)) + x];
    
    // Render the water
    roundedWaterLevel = (int)waterLevel->y;
    for (y = 0; y < roundedWaterLevel; y++)
    {
      *pixel = ANTIALIASING - 1;  // Water
      pixel -= width;
    }
    
    if (y >= height) {
      continue;
    }
    
    // Calculate and render antialiasing
    *pixel =
      roundf((waterLevel->y - truncf(waterLevel->y)) *
	     (float)(ANTIALIASING - 1));
    assert(*pixel >= 0);
    assert(*pixel < ANTIALIASING);
    pixel -= width;
    y++;
    
    // Render the air
    for (; y < height; y++) {
      *pixel = 0;  // Air
      pixel -= width;
    }
  }
  
  // Render the bubbles
  for (i = 0; i < this->nBubbles; i++) {
    if (this->bubbles[i].layer == layer ||
	this->bubbles[i].layer == FOREGROUND)
    {
      plotBubble(this, &this->bubbles[i]);
    }
  }

  // Set up the real world rendering colors
  airColor = interpolateColor(this->minAirColor, this->maxAirColor,
			      (this->colorIntensity * 255) / 100);
  waterColor = interpolateColor(this->minWaterColor, this->maxWaterColor,
				(this->colorIntensity * 255) / 100);
  for (i = 0; i < ANTIALIASING; i++) {
    colors[i] = interpolateColor(airColor, waterColor,
				 (i * 255) / (ANTIALIASING - 1));
  }
  
  // Render
  nPixels = width * height;
  for (i = 0; i < nPixels; i++) {
    if (this->antialiasingMap[i] < 0) {
      this->antialiasingMap[i] = 0;
    } else if (this->antialiasingMap[i] >= ANTIALIASING) {
      this->antialiasingMap[i] = ANTIALIASING - 1;
    }
    
    if (layer == BACKGROUND) {
      // Over-write the current picture
      picture->pixels[i] = colors[this->antialiasingMap[i]];
    } else {
      // Draw on top of the current picture
      picture->pixels[i] =
	interpolateColor(picture->pixels[i], colors[this->antialiasingMap[i]],
			 colors[this->antialiasingMap[i]].components.a);
    }
  }
}

static void renderGrowth(bubblemon_private_t *this,
			 bubblemon_picture_t *picture)
{
  int x;
  
  assert(this != NULL);
  assert(picture != NULL);
  
  for (x = 0; x < this->width; x++)
  {
    int y;
    int highestWeedPixel =
      this->height - roundf(this->weeds[x].height);
    
    for (y = highestWeedPixel; y < this->height; y++) {
      picture->pixels[y * this->width + x] =
	interpolateColor(picture->pixels[y * this->width + x], this->weeds[x].color,
			 this->weeds[x].color.components.a);
    }
  }
}

static void renderFloater(bubblemon_private_t *this,
			  bubblemon_picture_t *picture)
{
  int floaterX;
  int floaterW, floaterH;
  int floaterYMin, floaterYMax;
  int pictureX0, pictureY0;

  assert(this != NULL);
  assert(picture != NULL);
  
  if (this->floaterState == GONE) {
    return;
  }
  
  floaterW = this->floaterImage.width;
  floaterH = this->floaterImage.height;
  
  // Ditch the floater if the image is too small
  if (this->width < (floaterW + 4)) {
    return;
  }
  
  // Center the floater horizontally
  pictureX0 = (this->width - floaterW) / 2;
  // Position the floater vertically
  pictureY0 = this->height - this->floaterY - floaterH / 2;
  
  // Clip the floater vertically to fit it into the picture
  floaterYMin = 0;
  if (pictureY0 < 0) {
    floaterYMin = -pictureY0;
  }
  floaterYMax = floaterH;
  if (pictureY0 + floaterH >= this->height) {
    floaterYMax = floaterH - (pictureY0 + floaterH - this->height);
  }
  
  // Iterate over the floater
  for (floaterX = 0; floaterX < floaterW; floaterX++) {
    int floaterY;
    
    for (floaterY = floaterYMin; floaterY < floaterYMax; floaterY++) {
      int pictureX = pictureX0 + floaterX;
      int pictureY = pictureY0 + floaterY;
      
      bubblemon_color_t floaterPixel;
      bubblemon_color_t *picturePixel;

      assert(pictureY >= 0);
      assert(pictureY < this->height);
      
      floaterPixel = (this->floaterImage.pixels)[floaterX + floaterY * floaterW];
      picturePixel = &(picture->pixels[pictureX + pictureY * this->width]);
      
      *picturePixel = interpolateColor(*picturePixel, floaterPixel,
				       floaterPixel.components.a);
    }
  }
}

// Render our physics into the provided picture
static void renderPhysics(bubblemon_private_t *this,
			  bubblemon_picture_t *picture)
{
  assert(this != NULL);
  assert(picture != NULL);
  assert(this->width == picture->width);
  assert(this->height == picture->height);
  
  renderAirAndWater(this, picture, BACKGROUND);
  renderGrowth(this, picture);
  renderFloater(this, picture);
  renderAirAndWater(this, picture, FOREGROUND);
}

// Render a frame.  Before calling this function, the caller is
// responsible for that picture->pixels is pointing to a memory area
// of at least picture->width * picture->height *
// sizeof(bubblemon_color_t) bytes.
//
// Returns:
// 0 on success
// ENOMEM (as defined in <errno.h>) on out of memory
int bubblemon_render(bubblemon_t bubblemon, bubblemon_picture_t *picture)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  int result;
  assert(this != NULL);
  
  result = setSize(this, picture->width, picture->height);
  if (result != 0) {
    return result;
  }
  
  updatePhysics(this);
  
  renderPhysics(this, picture);
  
  return 0;
}

// Free all resources of a bubblemon previously allocated by
// bubblemon_create().
void bubblemon_destroy(bubblemon_t bubblemon)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  if (this->antialiasingMap != NULL) {
    free(this->antialiasingMap);
    this->antialiasingMap = NULL;
  }
  if (this->waterLevels != NULL) {
    free(this->waterLevels);
    this->waterLevels = NULL;
  }
  if (this->bubbles != NULL) {
    free(this->bubbles);
    this->bubbles = NULL;
    this->nBubbles = 0;
    this->maxBubbles = 0;
    this->createNNewBubbles = 0;
  }
  if (this->weeds != NULL) {
    free(this->weeds);
    this->weeds = NULL;
    this->msecsToNextWeedUpdate = 0;
    this->lastUpdatedWeed = 0;
  }
  
  assert(this->antialiasingMap == NULL);
  assert(this->waterLevels == NULL);
  assert(this->bubbles == NULL);
  assert(this->weeds == NULL);
  
  free(this);
}


/***************************************************************
 *
 * Sensor input that will be visualized:
 */

// Set the water level to 0-100
void bubblemon_setWaterLevel(bubblemon_t bubblemon, int level)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  assert(level >= 0);
  assert(level <= 100);
  
  this->targetWaterlevel = level;
}

// Set the color intensity to 0-100
void bubblemon_setColorIntensity(bubblemon_t bubblemon, int intensity)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  assert(intensity >= 0);
  assert(intensity <= 100);
  
  this->colorIntensityGoal = intensity;
}

// Set the amount of bubbles to 0-100
void bubblemon_setBubbles(bubblemon_t bubblemon, int bubbles)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  assert(bubbles >= 0);
  assert(bubbles <= 100);
  
  this->bubblingIntensity = bubbles;
}

// Set the amount of growth to 0-100
void bubblemon_setGrowthIntensity(bubblemon_t bubblemon, int growth)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  assert(growth >= 0);
  assert(growth <= 100);
  
  this->growthIntensity = growth;
}

// Set whether the floater is visible (true / false)
void bubblemon_setFloaterVisible(bubblemon_t bubblemon, int visible)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  this->floaterVisible = visible;
}


/***************************************************************
 *
 * Settings
 */

// Get / set the water and air colors
bubblemon_color_t bubblemon_getMaxWaterColor(bubblemon_t bubblemon)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  return this->maxWaterColor;
}

void bubblemon_setMaxWaterColor(bubblemon_t bubblemon,
				bubblemon_color_t color)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  this->maxWaterColor = color;
}

bubblemon_color_t bubblemon_getMinWaterColor(bubblemon_t bubblemon)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  return this->minWaterColor;
}

void bubblemon_setMinWaterColor(bubblemon_t bubblemon,
				bubblemon_color_t color)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  this->minWaterColor = color;
}

bubblemon_color_t bubblemon_getMaxAirColor(bubblemon_t bubblemon)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  return this->maxAirColor;
}

void bubblemon_setMaxAirColor(bubblemon_t bubblemon,
			      bubblemon_color_t color)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  this->maxAirColor = color;
}


bubblemon_color_t bubblemon_getMinAirColor(bubblemon_t bubblemon)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  return this->minAirColor;
}

void bubblemon_setMinAirColor(bubblemon_t bubblemon,
			      bubblemon_color_t color)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  this->minAirColor = color;
}

// Change the color of the stuff growing from the bottom.  The growth
// color is randomized between these two values.
void bubblemon_setGrowthColor(bubblemon_t bubblemon,
			      bubblemon_color_t color1,
			      bubblemon_color_t color2)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
  
  this->growthColor1 = color1;
  this->growthColor2 = color2;
}


// Fetch the current color of the stuff growing from the bottom.  If
// any of the parameters is NULL, that parameter will be ignored.
void bubblemon_getGrowthColor(bubblemon_t bubblemon,
			      bubblemon_color_t *color1,
			      bubblemon_color_t *color2)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  assert(this != NULL);
    
  if (color1 != NULL) {
    *color1 = this->growthColor1;
  }

  if (color2 != NULL) {
    *color2 = this->growthColor2;
  }
}

// Change the floater image (default is a message-in-a-bottle).  The
// horizontal center of gravity is counted from the top down, so lower
// values are higher up.
//
// Returns:
// 0 on success
// ENOMEM (as defined in <errno.h>) on out of memory.  In this case,
//   the previous image is preserved.
int bubblemon_setFloaterImage(bubblemon_t bubblemon,
			      bubblemon_picture_t *picture,
			      int horizontalCenterOfGravity,
			      int verticalCenterOfGravity)
{
  bubblemon_private_t *this = (bubblemon_private_t*)bubblemon;
  bubblemon_color_t *newFloaterPixels;
  
  assert(this != NULL);
  assert(picture != NULL);
  
  // First, try allocating space for the new image.  Thus, we won't
  // break our previous image in case we run out of memory.
  newFloaterPixels =
    (bubblemon_color_t *)malloc(picture->width *
				picture->height *
				sizeof(bubblemon_color_t));
  if (newFloaterPixels == NULL) {
    return ENOMEM;
  }
  
  // Free the previous image (if any)
  if (this->floaterImage.pixels != NULL) {
    free(this->floaterImage.pixels);
    this->floaterImage.pixels = NULL;
  }
  assert(this->floaterImage.pixels == NULL);
  
  // Copy the new image
  this->floaterImage.pixels = newFloaterPixels;
  memcpy(this->floaterImage.pixels,
	 picture->pixels,
	 picture->width * picture->height * sizeof(bubblemon_color_t));
  
  this->floaterImage.width = picture->width;
  this->floaterImage.height = picture->height;
  this->floaterXCenter = horizontalCenterOfGravity;
  this->floaterYCenter = verticalCenterOfGravity;

  return 0;
}
