#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif

/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation.
 */

// This file contains definitions of exported functions
// No static functions or symbols should be in this file.


// Deprecated because flags cannot be specified. Use dbh_new() instead.
// Creates a DBH file:
// path to DBH file
// key_length: length of key in bytes. The keylength will define 
// the dimensionality of the data table.
//
DBHashTable * dbh_create (const char *path, unsigned char key_length) {
    DBG( "dbh_create() is deprecated (%s). Please use dbh_new() instead.\n", path);
    // Flag setting is not enabled with this function call.
    // Neither parallel safe nor thread safe mode can be 
    // specified.
    return sdbh_create(path, key_length, 0);
}

// Open an existing DBH file. 
DBHashTable * dbh_open (const char *path) {
    DBG( "dbh_open() is deprecated (%s). Please use dbh_new() instead.\n", path);
    // No parallel safe nor thread safe mode with deprecated functions.
    DBHashTable *dbh = sdbh_open_S (path, WRITE, 0);
    return (dbh);
}


// Open an existing DBH file in read-only mode. 
DBHashTable * dbh_open_ro (const char *path) {
    DBG( "dbh_open_ro() is deprecated (%s). Please use dbh_new() instead.\n", path);
    // No parallel safe nor thread safe mode with deprecated functions.
    DBHashTable *dbh = sdbh_open_S (path, READ, 0);
    return (dbh);
}

// Create or open an existing DBH file.
// read-only, parallel-safe or thread-safe mode can be specified
// with flags. If process or thread contention is not an issue,
// then it is best not to specify parallel-safe or thread-safe
// modes, as these involve wait/post and lock/unlock semaphores
// or mutexes, which will involve a performance hit.
DBHashTable * dbh_new (const char *path, unsigned char *key_length, int flags) {
    DBHashTable *dbh;
    
    if (flags & DBH_THREAD_SAFE) pthread_mutex_lock(&new_mutex);
    // Check for coherent path
    if (!path || strlen(path)==0) {
        errno=EINVAL;
        ERR( "dbh_new(): %s\n", strerror (errno));
	return NULL;
    }
    if (flags & DBH_CREATE){
	// Check for coherent key length
	if (!key_length || *key_length > 254){
            errno=EINVAL;
	    ERR( "Cannot create DBH table (%s) of key length %d.\n", 
		    path, (key_length)?((int)(*key_length)):-1);
	    return NULL;
	}
	// Clear any previous dbh files which may cause creation failure.
	if (unlink(path) < 0) {
            // This is OK, since final test is done with sdbh_create()
	}
	dbh = sdbh_create(path, *key_length, flags);
        if (!dbh){
	    ERR( "Cannot create DBH table (%s).\n", path);
            return NULL;
        }
    } else {
	if (flags & DBH_READ_ONLY){
	    dbh = sdbh_open_S (path, READ, flags);
	} else {
	    dbh = sdbh_open_S (path, WRITE, flags);
	}
    }

    if (!dbh) {
        if (flags & DBH_THREAD_SAFE) pthread_mutex_unlock(&new_mutex);
		DBG("%s cannot be opened...\n", path);
        return NULL;
    }


#ifndef PARALLEL_SAFE
    if (!(flags & DBH_READ_ONLY) && (flags & DBH_PARALLEL_SAFE )){
            ERR("\n\
\tDBH_PARALLEL_SAFE not currently implemented on this platform\n\
\tPlease use DBH_READ_ONLY flag for parallel read access\n\
\tor file locking if you require parallel writes.\n\
\tDBH_THREAD_SAFE allows parallel writes from threaded applications.\n");
	    return NULL;
    }
#endif
    
    if (key_length) *key_length = DBH_KEYLENGTH(dbh);

    if (flags & DBH_THREAD_SAFE){
         dbh->mutex = (pthread_mutex_t *)malloc(sizeof(pthread_mutex_t));
         if (dbh->mutex == NULL) {
             dbh_close(dbh);
             ERR( "malloc(%ld): %s\n", (long)sizeof(pthread_mutex_t), strerror(errno));
                 return NULL;
         }     
         pthread_mutexattr_t attr;
         pthread_mutexattr_init(&attr);
         pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);
        // pthread_mutex_init(dbh->mutex, NULL);
         pthread_mutex_init(dbh->mutex, &attr);	
         pthread_mutexattr_destroy(&attr);
    }
    dbh->protection_flags = flags & (DBH_THREAD_SAFE|DBH_PARALLEL_SAFE|DBH_READ_ONLY);
    if (flags & DBH_THREAD_SAFE) pthread_mutex_unlock(&new_mutex);
    return dbh;
}


// dbh_close: Closes an open dbh table and flushes to disk
int dbh_close (DBHashTable * dbh) {
        TRACE("dbh_close\n");
    if(dbh == NULL) {
        ERR( "dbh_close(dbh): dbh == NULL\n");
        return 0;
    }
    if (dbh->protection_flags & DBH_THREAD_SAFE)  pthread_mutex_lock(&new_mutex);
    // This is done whenever needed, not until last moment...
    if(dbh->head_info && dbh->head_info->writeOK){ sdbh_writeheader (dbh, 1); }

    TRACE("flushing to disk on dbh_close()\n");

    if (dbh->fd >= 0){
#if HAVE_FSYNC
        fsync(dbh->fd);
#endif
#if HAVE_WINDOWS_H
        FlushFileBuffers ((HANDLE) _get_osfhandle (dbh->fd));
#endif

        if(close (dbh->fd) < 0) {
            ERR( "close(%d): %s\n", dbh->fd, strerror (errno));
        }
    }

    if (dbh->data) free (dbh->data);
    if (dbh->newdata) free (dbh->newdata);
    if (dbh->branch) free (dbh->branch);
    if (dbh->newbranch) free (dbh->newbranch);
    if (dbh->key) free (dbh->key);
    if (dbh->newkey) free (dbh->newkey);
    if (dbh->head_info) free (dbh->head_info);
#ifdef PARALLEL_SAFE
    if (dbh->sem) {
        if (dbh->protection_flags & DBH_PARALLEL_SAFE) {
            if (dbh->lock_attempt_limit == 0) sem_wait(dbh->sem);
            else { 
                struct timespec timeout;
                timeout.tv_sec = time(NULL) + dbh->lock_attempt_limit;
                timeout.tv_nsec = 0;
                if (sem_timedwait(dbh->sem, &timeout) < 0){
                    ERR("DBH: dbh_close() unable to unlock semaphore for %s (%s), proceeding on timeout...\n", dbh->path, strerror(errno));
                }
            }
            
            destroy_shm_lock(dbh->path, dbh->lock_p);
            sem_post(dbh->sem);
            sem_close(dbh->sem);
            // Only a semaphore that has been initialized  by  sem_init(3)  should  be
            // destroyed using sem_destroy().
            // This is wrong: sem_destroy(dbh->sem);
            char *s_name=sem_name(dbh->path);
            if (s_name) {
                sem_unlink(s_name);
                free(s_name);
            }
        }
    }
#endif
    if (dbh->protection_flags & DBH_THREAD_SAFE) {
        if (dbh->mutex == NULL){
	    DBG( "This should happen in barre: dbh->mutex == NULL on dbh_close\n");
	} else {
            pthread_mutex_destroy(dbh->mutex);
	    free(dbh->mutex);   
	}
    }
    if (dbh->path) free (dbh->path);
    if (dbh->tmpdir) free (dbh->tmpdir);
    if (dbh->protection_flags & DBH_THREAD_SAFE) pthread_mutex_unlock(&new_mutex);
    free (dbh);
    return 1;
}

int dbh_destroy (DBHashTable * dbh) {
    char *file;
    if(!dbh) {
        ERR( "dbh_destroy(): %s\n", strerror (EBADF));

        return 0;
    }

    file = (char *)malloc (strlen (dbh->path) + 1);
    if (!file){
	DBG( "malloc (%ld): %s\n", (long)(strlen (dbh->path) + 1),strerror(errno));
        return 0;
    }
    strcpy (file, dbh->path);
    dbh_close (dbh);
    int retval = remove (file);

    if(retval < 0)
        retval = 0;
    else
        retval = 1;
    free (file);
    return 1;
}

int dbh_set_size (DBHashTable * dbh, FILE_POINTER record_length) {
    int s = sdbh_size (dbh, record_length);
    return (s);
}

void dbh_set_recordsize (DBHashTable * dbh, int size) {
    if(!dbh) {
        errno=EINVAL;
        DBG( "dbh_set_recordsize(): %s\n", strerror (EINVAL));
        return;
    }
    dbh->bytes_userdata = size;
    return;
}

void dbh_set_data (DBHashTable * dbh, void *data, FILE_POINTER n) {
    if(!dbh || !data) {
        errno=EINVAL;
        DBG( "DBH: invalid parameter in dbh_set_data()\n");
        return;
    }
    if(n > DBH_MAXIMUM_RECORD_SIZE(dbh)) {
        DBG( "DBH: redefining maximum record size to %lld\n", (long long)n);
        errno=EINVAL;
        dbh_set_size (dbh, n);
        return;
    }
    memcpy ((void *)dbh->data, (void *)data, n);
    dbh->bytes_userdata = n;
    return;
}

void dbh_set_key (DBHashTable * dbh, unsigned char *key) {
    if(!dbh || !key) {
        errno=EINVAL;
        DBG( "DBH: invalid parameter in dbh_set_key()\n");
        return;
    }
    memcpy ((void *)dbh->key, (void *)key, dbh->head_info->n_limit);
    return;
}

FILE_POINTER dbh_update (DBHashTable * dbh) {
    TRACE ("dbh_update\n");

    if(dbh == NULL){
        DBG( "dbh == NULL at dbh_update()\n"); 
        errno=EINVAL;
        return 0;
    }
    if(!(dbh->head_info->writeOK)){
	DBG( "dbh_update() is invalid in a read only DBH Table\n");
        errno=EINVAL;
	return 0;
    }
    FILE_POINTER fp[3];
    unsigned char j, caso;
    int i;

    dbh_lock_write (dbh);

    /* before updating, clean erased flag, so updated record will be 
     * automatically unerased this allows reusing the erased space, 
     * although it introduces error in the tabulated values of 
     * erased_space and total_space, but who cares, it's an 
     * approximation anyways. */
    SET_UNERASED;

    dbh->head_info->reservedC = 0;

    if(sdbh_locate (dbh, fp) == NULL){
        return 0;
    }

    if(CURRENTSEEK) {
        if(dbh->newbytes_userdata < dbh->bytes_userdata) {
            if(LASTSEEK)
                caso = PRESENTE_MENOR;
            else
                caso = PRESENTE_MENOR_BOF;
        } else
            caso = PRESENTE_MAYORIGUAL;
    } else {
        if(LASTSEEK)
            caso = NO_PRESENTE;
        else
            caso = ARCHIVO_VACIO;
    }
    dbh->flag = 0;
    int flush_dbh = 0;
    switch (caso) {
    case NO_PRESENTE:          /* not found */
            TRACE("dbh:record not found\n");
        for(i = 0; i < dbh->head_info->n_limit; i++)
            dbh->branch[i] = 0;
        CURRENTSEEK = place_eof (dbh);
            TRACE("dbh:CURRENTSEEK (eof) at %lld\n", (long long)CURRENTSEEK);
        if(CURRENTSEEK < 0){
	    dbh_unlock_write (dbh);
            return 0;
	}
        dbh->newbranches -= (unsigned char)CURR_BRANCH;
        dbh->head_info->data_space += dbh->bytes_userdata;
        dbh->head_info->total_space += (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
        if(!sdbh_write (OLD, dbh, WRITEBRANCHES)){
	    dbh_unlock_write (dbh);
            return 0;
	}
        if(!sdbh_readBranches (dbh, LASTSEEK)){
	    dbh_unlock_write (dbh);
            return 0;
	}
        dbh->newbranch[CURR_BRANCH] = CURRENTSEEK;
        sdbh_updateBranch (dbh, LASTSEEK);
        dbh->head_info->records++;
        break;  /*********************************/
    case PRESENTE_MAYORIGUAL:  /* use already assigned disk space */
        dbh->head_info->erased_space += (dbh->newbytes_userdata - dbh->bytes_userdata);
        dbh->head_info->data_space -= (dbh->newbytes_userdata - dbh->bytes_userdata);
        if(!place_fp_at (dbh, (off_t)CURRENTSEEK)){
            dbh_unlock_write (dbh);
            return 0;
	}
        if(!sdbh_write (OLD, dbh, DONTWRITEBRANCHES)) {
            dbh_unlock_write (dbh);
            return 0;
        }
        // Replacing an existing data item: flush to disk
        flush_dbh = 1;
        break;  /*********************************/
    case PRESENTE_MENOR_BOF:   /* does not fit in assigned disk space and 
                                   is the first record (bof) to boot */
        dbh->head_info->erased_space += (dbh->newbytes_userdata);
        dbh->head_info->data_space += (dbh->bytes_userdata - dbh->newbytes_userdata);
        dbh->head_info->total_space += (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
        FILE_POINTER eof = place_eof (dbh);
        if(eof < 0) {
            dbh_unlock_write (dbh);
            return 0;
        }
        dbh->head_info->bof = eof;
        if(!sdbh_write (NEW, dbh, WRITEBRANCHES)) {
            dbh_unlock_write (dbh);
            return 0;
        }
        flush_dbh = 1;
        break;  /*********************************/
    case PRESENTE_MENOR:       /* does not fit in assigned disk space */
        {
            unsigned char ramas;
            dbh->head_info->erased_space += (dbh->newbytes_userdata);
            dbh->head_info->data_space += (dbh->bytes_userdata - dbh->newbytes_userdata);
            dbh->head_info->total_space +=
                (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
            CURRENTSEEK = place_eof (dbh);
            if(CURRENTSEEK < 0) {
                dbh_unlock_write (dbh);
                return 0;
            }
            j = dbh->newbranches;
            if(!sdbh_write (NEW, dbh, WRITEBRANCHES)) {
                dbh_unlock_write (dbh);
                return 0;
            }
            ramas = sdbh_readBranches (dbh, LASTSEEK);
            if(!ramas) {
                dbh_unlock_write (dbh);
                return 0;
            }
            dbh->newbranch[ramas - j + CURR_BRANCH] = CURRENTSEEK;
            sdbh_updateBranch (dbh, LASTSEEK);
        }
        flush_dbh = 1;
        break;   /*********************************/
    case ARCHIVO_VACIO:        /* empty dbh file */
        for(i = 0; i < dbh->head_info->n_limit; i++)
            dbh->branch[i] = 0;
        CURRENTSEEK = dbh->head_info->bof;
        if(!place_fp_at (dbh, (off_t)CURRENTSEEK)) {
            dbh_unlock_write (dbh);
            return 0;
        }
        dbh->newbranches = dbh->head_info->n_limit;
        dbh->head_info->data_space += dbh->bytes_userdata;
        dbh->head_info->total_space += (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
        if(!sdbh_write (OLD, dbh, WRITEBRANCHES)) {
            dbh_unlock_write (dbh);
            return 0;
        }
        dbh->head_info->records++;
        flush_dbh = 1;
        break;  /*********************************/
    }
    FILE_POINTER p = CURRENTSEEK;
    sdbh_writeheader(dbh, flush_dbh);
    dbh_unlock_write (dbh);
    return p;
}

FILE_POINTER dbh_load (DBHashTable * dbh) {
    int i;
    int j;
    FILE_POINTER fp[3];
    unsigned char *tmp1, *tmp2;
    if(dbh == NULL){
        return 0;
    }
    dbh_lock_write (dbh);

    /* before loading, clean erased flag, so it will have a
     * valid value before function returns 0 */
    SET_UNERASED;


    if(sdbh_locate (dbh, fp) == NULL){
	dbh_unlock_write (dbh);
        return 0;
    }
    dbh->reservedB = CURRENTSEEK;
    if(!CURRENTSEEK){
	dbh_unlock_write (dbh);
        return 0;
    }
    /* don't toggle erased yet! Read the dbh anyways! */
    /*if (ERASED){TOGGLE_ERASE; return 0;} */

    dbh->bytes_userdata = dbh->newbytes_userdata;
    dbh->branches = dbh->newbranches;
    tmp1 = dbh->key;
    tmp2 = dbh->newkey;
    for(j = 0; j < dbh->head_info->n_limit; j++)
        tmp1[j] = tmp2[j];
    tmp1 = (unsigned char *)dbh->data;
    tmp2 = (unsigned char *)dbh->newdata;
    for(i = 0; i < dbh->newbytes_userdata; i++)
        tmp1[i] = tmp2[i];

    /* must look for CURRENTSEEK, (at dbh->current_seek) if this condition occurs: */
    if(ERASED) {
	dbh_unlock_write (dbh);
        return 0;
    }
    FILE_POINTER p = CURRENTSEEK;
    dbh_unlock_write (dbh);
    return p;
}

void dbh_regen_sweep (DBHashTable **dbh_p) {
    DBHashTable *new_dbh = sdbh_regen (*dbh_p, 1);
    if (new_dbh) {
	*dbh_p = new_dbh;
    }
    return;
}

void dbh_regen_fanout (DBHashTable **dbh_p) {
    DBHashTable *new_dbh = sdbh_regen (*dbh_p, 0);
    if (new_dbh) {
	*dbh_p = new_dbh;
    }
    return;
}

FILE_POINTER dbh_find (DBHashTable * dbh, int n) {
    FILE_POINTER fp[3];
    if(dbh == NULL)
        return 0;
    dbh_lock_write (dbh);

    if(sdbh_locateFind (dbh, fp, n) == NULL){
	dbh_unlock_write (dbh);
        return 0;
    }
    if(!CURRENTSEEK){
	dbh_unlock_write (dbh);
        return 0;
    }
    dbh->bytes_userdata = dbh->newbytes_userdata;
    dbh->branches = dbh->newbranches;
    FILE_POINTER p = CURRENTSEEK;
    dbh_unlock_write (dbh);
    return p;
}

unsigned char dbh_load_address (DBHashTable * dbh, FILE_POINTER currentseek) {
    TRACE ("dbh_load_address\n");
    unsigned char i;
    if(dbh == NULL)
        return 0;
    if(currentseek == 0){
        return 0;
    }
    dbh_lock_write (dbh);
    dbh->reservedB = currentseek;
    for(i = 1; i <= dbh->head_info->n_limit; i++)
        dbh->branch[i - 1] = 0;
    if(!place_fp_at (dbh, (off_t)currentseek)) {
        dbh_unlock_write (dbh);
        return 0;
    }
    if(!sdbh_read (OLD, dbh, 1)) {
        dbh_unlock_write (dbh);
        return 0;
    }
    unsigned char b = dbh->branches;
    dbh_unlock_write (dbh);
    return b;
}

FILE_POINTER dbh_load_parent (DBHashTable * dbh) {
    FILE_POINTER fp[3];
    if(dbh == NULL)
        return 0;
    dbh_lock_write (dbh);
    if(sdbh_locate (dbh, fp) == NULL){
	dbh_unlock_write (dbh);
        return 0;
    }
    if(!CURRENTSEEK || !LASTSEEK){
	dbh_unlock_write (dbh);
        return 0;
    }
    FILE_POINTER address = dbh_load_address (dbh, LASTSEEK);
    dbh_unlock_write (dbh);
    return address;
}

FILE_POINTER dbh_load_child (DBHashTable * dbh, unsigned char key_index) {
    FILE_POINTER fp[3], child_address;
    if(dbh == NULL)
        return 0;
    dbh_lock_write (dbh);
    if(sdbh_locate (dbh, fp) == NULL){
	dbh_unlock_write (dbh);
        return 0;
    }
    if(!CURRENTSEEK){
	dbh_unlock_write (dbh);
        return 0;
    }
    if(key_index >= dbh->newbranches){
	dbh_unlock_write (dbh);
        return 0;
    }
    child_address = *(dbh->newbranch + key_index);
    FILE_POINTER p = dbh_load_address (dbh, child_address);
    dbh_unlock_write (dbh);
    return p;
}

int dbh_erase (DBHashTable * dbh) {
    if(dbh == NULL) return 0;
    if(!(dbh->head_info->writeOK)){
	DBG( "dbh_erase() is invalid in a read only DBH Table\n");
        errno=EINVAL;
	return 0;
    }   
    FILE_POINTER currentseek;
    
    dbh_lock_write (dbh);
    currentseek = dbh_load (dbh);
    if(!currentseek) {           /* will return false if record is already ERASED */
        return 0;
    }
    TOGGLE_ERASE;               /* set the erased bit on */
    /* set file pointer at record flag byte */
    if(!place_fp_at (dbh, (off_t)currentseek + 1)) {
        dbh_unlock_write (dbh);
        return 0;
    }

    /* write the flag to the file */
    if(write (dbh->fd, &(dbh->flag), 1) != 1) {
        dbh_unlock_write (dbh);
        return 0;
    }
    /* update file header information */
    dbh->head_info->data_space -= dbh->bytes_userdata;
    dbh->head_info->erased_space += (dbh->bytes_userdata);
    sdbh_writeheader (dbh, 1);
    dbh_unlock_write (dbh);
    return 1;
}

int dbh_unerase (DBHashTable * dbh) {
    if(dbh == NULL) return 0;
    if(!(dbh->head_info->writeOK)){
	DBG( "dbh_unerase() is invalid in a read only DBH Table\n");
        errno=EINVAL;
	return 0;
    }
    FILE_POINTER currentseek, fp[3];
    dbh_lock_write (dbh);
    /* this will return TRUE if record is already ERASED: */
    if(sdbh_locate (dbh, fp) == NULL){
        dbh_unlock_write (dbh);
        return 0;
    }

    currentseek = fp[0];
    if(!currentseek){
        dbh_unlock_write (dbh);
        return 0;
    }
    /* got currentseek, now load it */
    dbh_load_address (dbh, currentseek);        /* found currentseek, now load it */
    if(!ERASED){
        dbh_unlock_write (dbh);
        return 0;     /* hey man, nothing to do */
    }
    TOGGLE_ERASE;
    /* set file pointer at record flag byte */
    if(!place_fp_at (dbh, (off_t)currentseek + 1)) {
        dbh_unlock_write (dbh);
        return 0;
    }
    /* write the flag to the file */
    if(write (dbh->fd, &(dbh->flag), 1) != 1) {
        dbh_unlock_write (dbh);
        return 0;
    }
    /* update file header information */
    dbh->head_info->data_space += dbh->bytes_userdata;
    dbh->head_info->erased_space -= (dbh->bytes_userdata);
    sdbh_writeheader (dbh, 1);
    dbh_unlock_write (dbh);
    return 1;
}

int dbh_prune (DBHashTable * dbh, unsigned char *key, unsigned char subtree_length) {
    if(!(dbh->head_info->writeOK)){
	DBG( "dbh_prune() is invalid in a read only DBH Table\n");
        errno=EINVAL;
	return 0;
    }
    int result;
    dbh_lock_write (dbh);
    dbh->head_info->sweep_erased = 1;
    result = dbh_sweep (dbh, prune_mark_erased, key, NULL, subtree_length);
    dbh->head_info->sweep_erased = 0;
    /* update file header information */
    sdbh_writeheader (dbh, 1);
    dbh_unlock_write (dbh);
    return result;
}

int dbh_unprune (DBHashTable * dbh, unsigned char *key, unsigned char subtree_length) {
    if(!(dbh->head_info->writeOK)){
	DBG( "dbh_unprune() is invalid in a read only DBH Table\n");
        errno=EINVAL;
	return 0;
    }
    int result;
    dbh_lock_write (dbh);
    dbh->head_info->sweep_erased = 1;
    result = dbh_sweep (dbh, prune_mark_unerased, key, NULL, subtree_length);
    dbh->head_info->sweep_erased = 0;
    /* update file header information */
    sdbh_writeheader (dbh, 1);
    dbh_unlock_write (dbh);
    return result;
}

int dbh_foreach (DBHashTable * dbh, DBHashFunc2 operate, void *data) {
    dbh_lock_write(dbh);
    dbh->head_info->dbh_exit = 0;
    int r = sdbh_newbarre2 (dbh, operate, data);
    dbh_unlock_write (dbh);
    return r;
}

int dbh_foreach_sweep (DBHashTable * dbh, DBHashFunc operate) {
    dbh_lock_write(dbh);
    dbh->head_info->dbh_exit = 0;
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newbarre (dbh, NULL, NULL, 0);
    dbh_unlock_write (dbh);
    return r;
}

int dbh_foreach_fanout (DBHashTable * dbh, DBHashFunc operate) {
    dbh_lock_write(dbh);
    dbh->head_info->dbh_exit = 0;
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newreversebarre (dbh, NULL, NULL, 0);
    dbh_unlock_write(dbh);
    return r;
}

int dbh_sweep (DBHashTable * dbh, DBHashFunc operate, unsigned char *key1, unsigned char *key2, unsigned char ignore_portion) {
    if(!dbh)
        return 0;
    dbh_lock_write(dbh);
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newbarre (dbh, key1, key2, ignore_portion);
    dbh_unlock_write(dbh);
    return r;
}

int dbh_fanout (DBHashTable * dbh, DBHashFunc operate, unsigned char *key1, unsigned char *key2, unsigned char ignore_portion) {
    if(!dbh)
        return 0;
    dbh_lock_write(dbh);
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newreversebarre (dbh, key1, key2, ignore_portion);
    dbh_unlock_write(dbh);
    return r;
}

void dbh_exit_sweep (DBHashTable * dbh) {
    if(!dbh) {
        DBG( "dbh_exitsweep(): %s\n", strerror (EINVAL));
        errno=EINVAL;

        return;
    }
    dbh->head_info->dbh_exit = 1;
    return;
}

void dbh_exit_fanout (DBHashTable * dbh) {
    dbh_exit_sweep (dbh);
    return;
}


int dbh_writeheader (DBHashTable * dbh) {
    return sdbh_writeheader(dbh, 1);
}
void dbh_orderkey (unsigned char *numero, unsigned char orden, unsigned int n, unsigned char base) {
    int divisor,
      i,
      t;
    double d,
      b,
      o;
    if(!n) {
        DBG( "dbh_genkey: value must be > 0\n");
        errno=EINVAL;
        return;
    }

    for(i = 0; i < orden; i++) {
        b = base;
        o = orden - 1 - i;
        d = pow (b, o);
        divisor = d;
/*printf("pow(%d,%d) orden=%d, divisor=%d\n",base,orden-1-i,orden,divisor);*/
        numero[i] = (unsigned char)(n / divisor);
        n = n % divisor;
    }
    for(i = 0; i < orden; i++) {
        t = numero[i];
        t += 65;
        numero[i] = (unsigned char)t;
    }
    for(i = 0; i < orden; i++) {
        if(numero[i] > 'Z') {
            t = numero[i];
            t += 6;
            numero[i] = (unsigned char)t;
        }
    }
}

void dbh_genkey (unsigned char *numero, unsigned char orden, unsigned int n) {
    unsigned char i;
    int t;
    if(!n) {
        DBG( "dbh_genkey: value must be > 0\n");
        errno=EINVAL;
        return;
    }
    sdbh_cuenta ((unsigned char *)numero, orden, n);
    for(i = 0; i < orden; i++) {
        t = numero[i];
        t += 48;
        numero[i] = (unsigned char)t;
    }
}

void dbh_genkey2 (unsigned char *numero, unsigned char orden, unsigned int n) {
    unsigned char i;
    int t;
    if(!n) {
        DBG( "dbh_genkey: value must be > 0\n");
        errno=EINVAL;
        return;
    }
    sdbh_cuenta ((unsigned char *)numero, orden, n);
    for(i = 0; i < orden; i++) {
        t = numero[i];
        t += 65;
        numero[i] = (unsigned char)t;
    }
/* for (i=0;i<orden;i++) numero[i] += 65;*/

    for(i = 0; i < orden; i++) {
        if(numero[i] > 'Z') {
            t = numero[i];
            t += 6;
            numero[i] = (unsigned char)t;
        }
    }
/* for (i=0;i<orden;i++) if (numero[i]>'Z') numero[i] += 6;*/
}

int dbh_settempdir (DBHashTable * dbh, char *dir) {
    if(!dir) {
        dbh->head_info->user_tmpdir = 0;
        if(dbh->tmpdir)
            free (dbh->tmpdir);
        dbh->tmpdir = NULL;
        return 0;
    }
    if(dbh->tmpdir)
        free (dbh->tmpdir);
    dbh->tmpdir = (char *)malloc (strlen (dir) + 1);
    if (!dbh->tmpdir){
	DBG( "malloc dbh->tmpdir: %s\n", strerror(errno));
        return 0;
    }
    strcpy (dbh->tmpdir, dir);
    dbh->head_info->user_tmpdir = 1;
    return 1;
}

int dbh_mutex_lock(DBHashTable * dbh){
    if (!dbh) return 0;
    if (!dbh->mutex){
        errno=EINVAL;
	DBG( "dbh_mutex_lock(): Mutex not enabled. Open table with DBH_THREAD_SAFE to enable mutex\n");
	return 0;
    }
    pthread_mutex_lock(dbh->mutex);
    return 1;
}

int dbh_mutex_unlock(DBHashTable * dbh){
    if (!dbh) return 0;
    if (!dbh->mutex){
        errno=EINVAL;
	DBG( "dbh_mutex_unlock(): Mutex not enabled. Open table with DBH_THREAD_SAFE to enable mutex\n");
	return 0;
    }
    pthread_mutex_unlock(dbh->mutex);
    return 1;
}

// deprecated
int dbh_set_parallel_lock_attempt_limit(DBHashTable * dbh, int limit){
    if (!dbh) return 0;
    dbh->lock_attempt_limit = limit;
    return 0;
}


int dbh_set_parallel_lock_timeout(DBHashTable * dbh, int seconds){
    if (!dbh) return 0;
    dbh->lock_attempt_limit = seconds;
    return 1;
}

int dbh_clear_locks(DBHashTable * dbh){
#ifndef PARALLEL_SAFE
    errno = ENOSYS;
    return 0;
#else
    if (!dbh || !dbh->path) {
        errno = EINVAL;
        return 0;
    }
    if (!(dbh->protection_flags & DBH_PARALLEL_SAFE)){
        return 0;
    }
    if (dbh->lock_attempt_limit == 0) sem_wait(dbh->sem);
    else { 
        struct timespec timeout;
        timeout.tv_sec = time(NULL) + dbh->lock_attempt_limit;
        timeout.tv_nsec = 0;
        if (sem_timedwait(dbh->sem, &timeout) < 0){
            ERR("DBH: dbh_clear_locks() unable to unlock semaphore for %s (%s), proceeding on timeout...\n", dbh->path, strerror(errno));
        }
    }
    dbh->lock_p->write_lock_count = 0;
    dbh->lock_p->write_lock = 0;
    dbh->lock_p->read_lock_count = 0;
    
    if(msync (dbh->lock_p,  sizeof(dbh_lock_t), MS_ASYNC|MS_INVALIDATE) < 0){
            ERR("Cannot msync shared memory item for %s: %s\n", 
                dbh->path, strerror (errno));
    }
    sem_post(dbh->sem);
#endif
    return 1;
}

int
dbh_lock_write (DBHashTable * dbh) {
#ifndef PARALLEL_SAFE
    errno = ENOSYS;
    return 0;
#else
    return sdbh_lock(dbh, 1);
#endif
}

int
dbh_lock_read (DBHashTable * dbh) {
#ifndef PARALLEL_SAFE
    errno = ENOSYS;
    return 0;
#else
    return sdbh_lock(dbh, 0);
#endif
}

int
dbh_unlock_read (DBHashTable * dbh) {
#ifndef PARALLEL_SAFE
    errno = ENOSYS;
    return 0;
#else
    return sdbh_unlock(dbh, 0);
#endif
}

int
dbh_unlock_write (DBHashTable * dbh) {
#ifndef PARALLEL_SAFE
    errno = ENOSYS;
    return 0;
#else
    return sdbh_unlock(dbh, 1);
#endif
}

int
dbh_set_lock_timeout(int seconds){
    open_timeout = seconds;
    return 1;
}
int 
dbh_get_lock_timeout(void){
    return open_timeout;
}



int dbh_info (DBHashTable * dbh) {
    if(!dbh) {
        errno = EINVAL;
        DBG( "dbh_info(): %s\n", strerror (errno));

        return 0;
    }
    dbh_lock_read (dbh);
    FILE_POINTER eof = place_eof (dbh);
    if(eof < 0){
	dbh_unlock_read (dbh);
        DBG( "dbh_info(): %s\n", strerror (errno));
        return 0;
    }

    fprintf (stdout, "\nEnd of DBHashTable = %lld\n", (long long)eof);
    fprintf (stdout, "\nDBHashTable dbh_header_t size = %ld", (long)sizeof (dbh_header_t));
    fprintf (stdout, "\ndbh_header:\n\
 version=%s\n\
 keylength=%d\n\
 first record position=%ld\n\
 maximum record length=%ld\n\
 records=%ld\n\
 total_space=%ld\n\
 data_space=%ld\n\
 erased_space=%ld\n\
 format_space=%ld\n\
", dbh->head_info->version, DBH_KEYLENGTH (dbh), (long)dbh->head_info->bof, (long)DBH_MAXIMUM_RECORD_SIZE (dbh), (long)DBH_RECORDS (dbh), (long)DBH_TOTAL_SPACE (dbh), (long)DBH_DATA_SPACE (dbh), (long)DBH_ERASED_SPACE (dbh), (long)DBH_FORMAT_SPACE (dbh));
    dbh_unlock_read (dbh);
    return 1;
}

void dbh_genkey0 (unsigned char *numero, unsigned char orden, unsigned int n) {
    if(!n) {
        DBG( "dbh_genkey: value must be > 0\n");
        errno=EINVAL;
        return;
    }
    sdbh_cuenta ((unsigned char *)numero, orden, n);
}

/****************************************************************/
