/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for transport of the Ogg format
 *
 * Copyright (C) 2006, 2007  Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file multiThreadedDemux.C                                               *
/*                                                                           */
/* multiThreadedDemux - multi-threaded demultiplexor                         */
/*                                                                           */
/*===========================================================================*/
#include	<Ogg/Ogg.H>
#include	<Debug.H>
#include	<Thread.H>

#include	<fstream>

extern "C" 
{
#include	<string.h>
}

namespace	Examples
{
  using namespace	Ogg;

  class Demuxer: public Transport, public Thread
  {
  public:
    Demuxer()
      : Transport(false)
    {}

    void
    run()
    {
      loop();
    }

    size_t
    recvCallback(
		 Page &	pg
		 )
    {
      size_t count = Transport::recvCallback(pg);

      if ( std::cin.eof()
	   ||
	   (count <= 0 )
	   )
	{
	  terminate();
	}

      return(count);
    }
  }
  ;

  class Muxer: public Transport, public Thread
  {
    std::ofstream	f;

  public:
    Muxer()
      : Transport(true)
    {}

    void
    run()
    {
      loop();
    }

    void
    sendCallback(
		 Page &	pg
		 )
    {
      if ( std::cout.write(static_cast<char *>(pg.data())
			   , pg.size()
			   )
	   )
	{
	  debug(true,
		"Serial No. ", pg.serialNo()
		," sent page ",pg.pageNo()
		," with granule position ", pg.granulePosition()
		);
	}
      else
	{
	  debug(true,
		"Serial No. ", pg.serialNo()
		," error occured sending page ",pg.pageNo()
		," with granule position ", pg.granulePosition()
		);
	}

      if ( pg.ending() )
	{
	  debug(true
		,"Serial No. ", pg.serialNo()
		," wrote ending page ",pg.pageNo()
		);
	  terminate();
	}
    }

    bool
    laterThan (
	       const Page &		p1
	       ,const Page &		p2
	       );
  }
  ;

  class Parser : public Logical, public virtual Thread
  {
  public:
    Parser(
	   Demuxer		demuxer
	   )
      : Logical(demuxer)
    {}

  };

  class Writer : public Logical, public virtual Thread
  {
  public:
    Writer(
	   Muxer &	muxer
	   ,long	serialNo
	   )
      : Logical(muxer
		,serialNo
		)
    {}
  };

  class Rewriter : public Writer, public Parser
  {
  protected:
    const char * 	codecName;

  public:
    Rewriter(
	     Muxer &	muxer
	     ,long	serialNo
	     ,Demuxer &	demuxer
	     ,const char *name
	     )
      : Examples::Writer(muxer
			 ,serialNo
			 )
	, Parser(demuxer)
	, codecName(name)
    {}

    const
    char *
    name() const
    {
      return(codecName);
    }

    virtual
    Position
    time(Position	granulePosition) const = 0;

    void
    run()
    {
      try
	{
	  Ogg::Logical::Writer & w = Examples::Writer::writer();
	  Parser::Reader & r = Parser::reader();
	  size_t total = r->size();

	  // print first header packet
	  debug(true, codecName,
		" : serial no ", Parser::serialNo()
		," got first header"
		);

	  w.dataSize(r->size());
	  memcpy(w->data()
		 ,r->data()
		 ,r->size()
		 );
	  debug(true, codecName,
		" : serial no ", Writer::serialNo()
		," writing first header, size = ",w.dataSize()
		);

	  w.granulePosition(r.granulePosition());

	  ++w;

	  do
	    {
	      w.beginData();
	      ++r;
	      total += r->size();
	      // print subsequent packet #, size
	      debug(false,
		    " : got ",r->packetNo(),"th packet, "
		    ,"granule position = ",r.granulePosition()
		    );

	      w.dataSize(r->size());
	      w.granulePosition(r.granulePosition());
	      memcpy(w->data()
		     ,r->data()
		     ,r->size()
		     );
	      debug(true, codecName,
		    " : writing ",w->packetNo(),"th packet, "
		    ,"granule position = ",w.granulePosition()
		    ," size = ",w.dataSize()
		    );
	      if ( !r.ending() )
		++w;
	      else
		{
		  // destruction of w will write out the last packet
		  delete &w;
		}
		
	    }
	  while ( !r.ending() );
	  debug(true, codecName,
		" : last packet no.=",r->packetNo()
		," total size=",total
		);

	}
      catch ( exception ex
	      )
	{
	  debug(true, codecName,
		ex.what()
		);
	  throw;
	}

      return;
    }
      
  }
  ;

  
  bool
  Muxer::laterThan (
		    const Page &		p1
		    ,const Page &		p2
		    )
  {
    Rewriter * rw1 = static_cast<Rewriter *>
      (static_cast<Writer *>(logical(p1.serialNo())));
    Rewriter * rw2 = static_cast<Rewriter *>
      (static_cast<Writer *>(logical(p2.serialNo())));
      
    Position time1 = rw1->time(p1.granulePosition());
    Position time2 = rw2->time(p2.granulePosition());

    debug(true,
	  rw1->name()," vs ",rw2->name()
	  ,", granulee positions "
	  ,p1.granulePosition()
	  ," vs "
	  ,p2.granulePosition()
	  ,", times "
	  ,time1
	  ," vs "
	  ,time2
	  );
      
    if ( p1.isHeader() != p2.isHeader() )
      return( !p1.isHeader() );

    if ( p1.isHeader() )
      {
	if ( p1.pageNo() != p2.pageNo() )
	  return( p1.pageNo() >= p2.pageNo() );
	else
	  return( p1.serialNo() < p2.serialNo() );
      }

    return( time1 >= time2 );
  }


  class TheoraRewriter : public Rewriter
  {
    Position		granuleNumer;
    Position		granuleDenom;
    unsigned int	granuleShift;

  public:
    TheoraRewriter(
		   Muxer &	muxer
		   ,long	serialNo
		   ,Demuxer &	demuxer
		   )
      : Rewriter(muxer
		 ,serialNo
		 ,demuxer
		 ,"theora"
		 )
    {}

    bool
    selectCallback(
		   Packet &	pkt
		   )
    { 
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 41)
	   ||
	   (uheader[0] != 0x80)
	   ||
	   (strncmp (header+1, "theora", 6))
	   )
	return(false);

      unsigned int pos = (uheader[22] |
			  (uheader[23]<<8) |
			  (uheader[24]<<16) |
			  (uheader[25]<<24)
			  );
      granuleNumer = Position(pos);
      pos = (uheader[26] |
	     (uheader[27]<<8) |
	     (uheader[28]<<16) |
	     (uheader[29]<<24)
	     );
      granuleDenom = Position(pos);

      granuleNumer = granuleDenom/granuleNumer;

      if ( (granuleNumer < 25)
	   ||
	   (granuleDenom > 31)
	   )
	{
	  pos = (uheader[25] |
		 (uheader[24]<<8) |
		 (uheader[23]<<16) |
		 (uheader[22]<<24)
		 );
	  granuleNumer = Position(pos);
	  pos = (uheader[29] |
		 (uheader[28]<<8) |
		 (uheader[27]<<16) |
		 (uheader[26]<<24)
		 );
	  granuleDenom = Position(pos);
	}
      
      char
      shift  = ((header[40] & 0x03) << 3);
      shift |= (header[41] & 0xe0) >> 5;
      granuleShift = Position(shift);

      debug(true,codecName,
	    " numerator = ", granuleNumer
	    ," denominator = ", granuleDenom
	    ," shift = ", granuleShift
	    );

      return(true);
    }


    Position
    time(Position	granulePosition) const
    {
      Position iframe = granulePosition >> granuleShift;
      Position pframe = granulePosition - (iframe << granuleShift);
      return( (iframe + pframe) * granuleDenom * 1000/ granuleNumer );
    }

  }
  ;
  
  class VorbisRewriter : public Rewriter
  {
    Position		granuleRate;

  public:
    VorbisRewriter(
		   Muxer &	muxer
		   ,long	serialNo
		   ,Demuxer &	demuxer
		   )
      : Rewriter(muxer
		 ,serialNo
		 ,demuxer
		 ,"vorbis"
		 )
    {}

    bool
    selectCallback(
		   Packet &	pkt
		   )
    {
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 30)
	   ||
	   (uheader[0] != 0x01)
	   ||
	   (strncmp (header+1, "vorbis", 6))
	   )
	return(false);

      unsigned int pos = (uheader[12] |
			  (uheader[13]<<8) |
			  (uheader[14]<<16) |
			  (uheader[15]<<24)
			  );
      granuleRate = Position(pos);

      debug(true,codecName,
	    " rate ", granuleRate
	    );
      return(true);
    }

    Position
    time(Position	granulePosition) const
    {
      return(granulePosition*1000/granuleRate);
    }
  };

}

int main(int, const char*)
{
  using namespace	Examples;
  using namespace	Ogg;

  Demuxer & demuxer = *(new Demuxer);
  Muxer & muxer	= *(new Muxer);
  VorbisRewriter & vorwriter = *(new VorbisRewriter(muxer
						    ,450116321
						    ,demuxer
						    ));
  TheoraRewriter& theowriter = *(new TheoraRewriter(muxer
						    ,658032240
						    ,demuxer
						    ));

  muxer.start();
  demuxer.start();
  vorwriter.start();
  theowriter.start();

  vorwriter.finish();
  theowriter.finish();
  demuxer.finish();
  muxer.finish();

  exit(0);
}
