/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for the Ogg transport format
 *
 * Copyright (C) 2006, 2007  Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file Exception.H                                                        *
 *                                                                           *
 *  Ogg::Exception class implementation                                      *
 *                                                                           */
/*===========================================================================*/
#ifdef __GNUG__
#pragma implementation	"Exception.H"
#pragma implementation	"ErrorImpl.H"
#endif

#include	<Ogg/Exception.H>
#include	<ErrorImpl.H>


namespace	Ogg
{
  //--Exception-----------------------------------------------------------------------------

  Exception::Exception(
		       const	char *	nm
		       ) throw()
    : exception()
      , ostringstream()
  {
    *this << "Exception: "<< nm;
  }

  //--Error---------------------------------------------------------------------------------

  std::map<Error::ErrorNo, string>	ErrorImpl::descriptions_;

  void
  ErrorImpl::initDescriptions()
  {
    descriptions_[None] 			= "No Error Detected";
    descriptions_[BeyondCurrentVersion]		= "Ogg Version Beyond Current";
    descriptions_[PacketNonContinuation]	= "Packet Missing Later Portion";
    descriptions_[PacketContinuation]		= "Packet Missing Earlier Portion";
    descriptions_[NonContinuity]		= "Missing Portions";
    descriptions_[BadGranulePosition]		= "Granule Position Not Monotonically Increasing";
    descriptions_[StreamBegin]			= "Improper Stream Beginning";
    descriptions_[StreamEnd]			= "Improper Stream Ending";
    descriptions_[PrematureEnd]			= "Premature End Of Stream";
  }

  ErrorImpl::ErrorImpl()
  {
    if ( descriptions_.empty() )
      initDescriptions();

    bits_.reset();	// begin with no error
  }

  void
  ErrorImpl::clear()
  {
    bits_.reset();
  }

  Error &
  Error::operator |= (ErrorNo errorNo)
  {
    ErrorImpl * impl = static_cast<ErrorImpl *>(this);

    impl->bits_[errorNo] = true;
    
    return(*this);
  }

  Error &
  Error::operator |= (Error & error)
  {
    ErrorImpl * impl1 = static_cast<ErrorImpl *>(this);
    ErrorImpl * impl2 = static_cast<ErrorImpl *>(&error);

    impl1->bits_ |= impl2->bits_;

    return(*this);
  }

  bool
  Error::operator [] (Error::ErrorNo errorNo)
  {
    ErrorImpl * impl = static_cast<ErrorImpl *>(this);

    return(impl->bits_[errorNo]);
  }

  const char *
  Error::operator () (Error::ErrorNo errorNo)
  {
    ErrorImpl * impl = static_cast<ErrorImpl *>(this);

    if ( impl->bits_[errorNo] )
      return(ErrorImpl::descriptions_[errorNo].c_str());
    else
      return(0);
  }

}

