/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for the Ogg transport format
 *
 * Copyright (C) 2006, 2007  Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Ogg::Transport implementation interface                                   */
/*                                                                           */
/*===========================================================================*/
#ifndef TransportImplInterface
#define	TransportImplInterface

#ifdef __GNUG__
#pragma interface
#endif

#include	<Ogg/Transport.H>
#include	<TransportToLogical.H>
#include	<LogicalToTransport.H>
#include	<ErrorImpl.H>

extern "C" 
{
# include	<pthread.h>
}

#include	<map>
#include	<queue>

///
///@file <src/lib/TransportImpl.H>
///
namespace	Ogg
{
  class TransportImpl;
  class TransportEncapser;
  class TransportDecapser;

  /// @brief Page implementation interface
  ///
  /// @ingroup transportImplementation
  ///
  class PageImpl
    : public	Page
  {
    friend class Page;
    friend class PageEncapser;
    friend class PageDecapser;
    friend class Transport;
    friend class TransportToLogical;
    friend class TransportImpl;
    friend class TransportEncapser;
    friend class TransportDecapser;

  private:
    static
    const size_t	allocedHeaderSize_ = 282;
    static
    const size_t	maxSegments_ = 255;
    static
    const size_t	maxSegmentSize_ = 255;
    static
    const size_t	allocedBodySize_ = maxSegments_ * maxSegmentSize_ ;
    static
    const size_t	maxPageSize_ = allocedHeaderSize_ + allocedBodySize_;
    static
    const size_t	workSize_ = 3 * maxPageSize_;
    static
    const size_t	defaultPageSize_ = 4096;

    static
    const unsigned int	crc_lookup[256];

  private:
    unsigned char *	data_;		///< size == allocedHeaderSize_ + allocedBodySize_

    /// @brief pointer to body in data_
    ///
    /// For writing, body_ = data_ + allocedHeaderSize_
    /// For reading, body_ is where data is read
    ///
    unsigned char *	body_; 		///< 
    size_t		bodySize_;	///< actual body size

    /// @brief pointer to header in data_
    ///
    /// For writing, header_ = body_ - headerSize_,
    /// after actual header size has been determined,
    /// so that header and body make a contiguous page
    /// which is presented by Page::data().
    /// For reading, header_ = data_ initially
    ///
    unsigned char *	header_;	///< 
    size_t		headerSize_;	///< actual header size

    size_t		segmentSizes_[maxSegments_];	///< work pad for segments
    size_t		numSegments_;		///< actual number of segments
    Position		granulePosition_;	///< of last packet ending in this page

    size_t		pageSize_;

    bool		streamBegun_;	///< b_o_s
    bool		streamEnded_;	///< e_o_s

    bool		isHeader_;	///< encapsulating a header packet
    long		headerNo_;	///< 0 for first

    LogicalToTransport* logical_;	///< of this page
    long		serialNo_;	///< of the logical stream
    long		pageNo_;	///< of this page in logical stream

    PacketToTransport *	lastPacket_;	///< last packet on this page
    PacketNo		packetNo_;	///< of last packet

    // debug bit stream sizes
    size_t		pagesSize_;
    size_t		headersSize_;

  public:
    PageImpl();

    virtual
    ~PageImpl();

    bool
    ended() const;

    // less than function for TransportImpl::orderedPages_
    bool
    operator < (
		const PageImpl &	p2
		) const;

    /// @brief Returns checksum of page
    ///
    /// Zeros out header_[22...25].
    unsigned int
    checksum();

    /// @brief Page no. implemented differently for encapsulation/decapsulation.
    virtual
    long
    pageNo() const = 0;

    /// @brief Clear page
    virtual
    void
    clear() = 0;

    /// @brief Encapsulate/Decapsulate the page
    virtual
    PageImpl *
    page() = 0;

    /// @brief Form the page header
    virtual
    void
    header() = 0;

    virtual
    void
    transportError(Error::ErrorNo) = 0;

    virtual
    Error
    transportError() = 0;
  };

  class PageEncapser : public PageImpl
  {
    friend class Page;
    friend class TransportToLogical;
    friend class TransportEncapser;

  private:
    bool		ready_;
    bool		gotLastPacket_;
    pthread_cond_t	condLast_;
    pthread_mutex_t	lockLast_;


    PageEncapser();

    ~PageEncapser();

    PageEncapser *
    page();

    void
    header();

    void
    transportError(
		   Error::ErrorNo
		   )
    {}

    Error
    transportError()
    {}

    /// @brief Returns pageNo_
    long
    pageNo() const;

    bool
    gotLastPacket();

    void
    gotLastPacket(
		   bool
		   );

    /// @brief Calls PageImpl::checksum, and sets checksum into header 
    void
    setChecksum();

    void
    clear();
  };

  class PageDecapser : public PageImpl
  {
    friend class Page;
    friend class TransportToLogical;
    friend class TransportDecapser;

  private:
    bool		seeking_;
    Position		seekPosition_;
    PacketNo		seekNo_;
    pthread_cond_t	condSeek_;
    pthread_mutex_t	lockSeek_;

    bool		gotFirstPacket_;
    pthread_cond_t	condFirst_;
    pthread_mutex_t	lockFirst_;

    ErrorImpl		transportError_;


    PageDecapser();

    PageDecapser(
		 const PageDecapser &
		 );

    void
    skip();

    void
    header();

    PageDecapser &
    operator=(const PageDecapser &);

    bool
    seeking();

    void
    seeking(
	    bool
	    );

    void *
    seek(
	 size_t *
	 );


    PageDecapser *
    page();

    void
    formPacket(
	       void *
	       ,size_t
	       );


    void
    handlePrematureEnd();

    bool
    gotFirstPacket();

    void
    gotFirstPacket(
		   bool
		   );

    /// @brief Verify page checksum
    ///
    /// Returns xor of page-inset checksum to newly computed checksum.
    bool
    checksumOK();

    void
    clear();

    unsigned char
    version();

    bool
    packetContinued();

    void
    packetContinued(
		    bool
		    );

    bool
    streamBeginning();

    bool
    streamEnding();

    Position
    granulePosition();

    /// @brief Returns page no. set into page header.
    long
    pageNo() const;

    void
    transportError(
		   Error::ErrorNo
		   );

    Error
    transportError();
  };

  ///
  /// @brief Implementation Class for Transport
  ///
  /// Deques packets from each of its Logical streams.
  /// Encapsulates packet(s) into the corresponding page.
  ///
  class TransportImpl : protected TransportToLogical
  {
    friend class	Transport;
    friend class	TransportToLogical;
    friend class	TransportEncapser;
    friend class	TransportDecapser;

  protected:
    Transport *				intf_;
    bool				encapsulating_;
    bool				terminate_;
    std::map<long, PageImpl *>		pages_;		///< Keyed by serial no.
    typedef std::map<long, PageImpl *>::iterator iPages;
    PageImpl *				workPage_;
    pthread_cond_t			condWork_;
    bool				gotAllStreams_;
    pthread_cond_t			condAll_;
    pthread_mutex_t			lockPages_;


    class PageOrder
    {
    public:
      bool
      operator()(const PageImpl * p1, const PageImpl * p2) const
      {
	    return(*p1 < *p2);
      }
    };

    std::priority_queue<PageImpl *
			, std::vector<PageImpl *>
			,PageOrder
			>		orderedPages_;


    TransportImpl(
		  bool	encapsulating
		  );

    Logical *
    logical(long	serialNo);

    /// @brief Removes all pages, and set page count to 0
    virtual
    void
    reset();

    virtual
    long
    serialNo(long) = 0;

    virtual
    void
    loop() = 0;

    virtual
    size_t
    logicalCount();

  };

  class TransportEncapser : public TransportImpl
  {
    friend class	Transport;
    friend class	TransportToLogical;

  private:
    size_t		nToMux_;


    TransportEncapser()
      : TransportImpl(true)
    {
      workPage_ = new PageEncapser();
      nToMux_ = 0;
    }

    long
    serialNo(long);

    void
    loop();

    size_t
    logicalCount();

    void
    encapPages();

    void
    writePages(
	       long	///< Number of pages
	       );
  };

  class TransportDecapser : public TransportImpl
  {
    friend class	Transport;
    friend class	TransportToLogical;

  private:
    size_t		nToSelect_;
    size_t		nDoneSelect_;
    pthread_mutex_t	lockSelect_;
    pthread_cond_t	condSelect_;


    TransportDecapser()
      : TransportImpl(false)
	, nToSelect_(0)
	, nDoneSelect_(0)
    {
      pthread_mutex_init(&lockSelect_, 0);
      pthread_cond_init(&condSelect_, 0);
      workPage_ = new PageDecapser();
     }

    void
    reset();

    long
    serialNo(long);

    void
    loop();

    PageDecapser *
    readPage();

    void
    decapPage(
	      PageDecapser &
	      );

    void
    handlePrematureEnd();

  };

}
#endif
