/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for the Ogg transport format
 *
 * Copyright (C) 2006, 2007, 2008 Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Ogg::Logical to Transport interface                                       */
/*                                                                           */
/*===========================================================================*/
#ifndef LogicalImplInterface
#define	LogicalImplInterface

#ifdef __GNUG__
#pragma interface
#endif

#include	<Ogg/Logical.H>
#include	<LogicalToTransport.H>
#include	<TransportToLogical.H>
#include	<ErrorImpl.H>

#include	<queue>
#include	<exception>

///
///@file <src/lib/LogicalToTransport.H>
///
namespace	Ogg
{
  class	LogicalImpl;
  ///
  /// @defgroup logicalImplementation Logical Implementation Interface
  ///
  /// @{

  /// @brief Packet implementation interface
  class PacketImpl
    : public PacketToTransport
  {
    friend class Packet;
    friend class PacketToTransport;
    friend class LogicalImpl;
    friend class Logical;
    friend class PageImpl;

  private:
    LogicalImpl *	logical_;	///< logical stream of this packet
    unsigned char *	data_;		///< type irrelevant to Logical
    size_t		size_;		///< of allocated data
    size_t		done_;		///< size encapped/decapped

    size_t		pageSize_;	///< for the page within which this packet begins
    bool		flush_;		///< flush any previous packets
      
    bool		isHeader_;	///< true for stream headers
    long		headerNo_;	///< 0 for first

    PacketNo		packetNo_;	///< within logical stream
    Position		granulePosition_;///< independent of any metric
    bool		streamEnding_;	///< last packet indicator
    ErrorImpl		transportError_;///< transport error

    PacketImpl(
	       LogicalImpl *
	       ,size_t 
	       );

    PacketImpl(
	       PacketImpl &
	       );

    ~PacketImpl();

  };

  /// @brief Implementation interface for Logical
  ///
  /// Enques packets for encapsulation. Each Logical has its own queue.
  ///
  class LogicalImpl : private LogicalToTransport
  {
    friend class Logical;
    friend class LogicalToTransport;
    friend class PacketToTransport;

  private:
    static
    const size_t	defaultQueueSize_ = 65536;

    class WriterImpl
      : public Logical::Writer
    {
      friend class	Logical;
      friend class	LogicalImpl;

    private:
      static
      const size_t	defaultDataSize_ = 1024;

    private:
      WriterImpl(
		 LogicalImpl &
		 );

      ~WriterImpl()
      {}

    private:
      LogicalImpl &		logical_;
      PacketImpl *		packet_;
      size_t			dataSize_;		// of next packets
    };
    
    class ReaderImpl
      : public Logical::Reader
    {
      friend class	Logical;
      friend class	LogicalImpl;
      friend class	PacketImpl;

    private:
      ReaderImpl(
		 LogicalImpl &
		 );

      ~ReaderImpl()
      {}

    private:
      LogicalImpl &		logical_;
      Position			granulePosition_;	// last known granulePosition
      PacketImpl *		packet_;

      void
      seek(
	   PacketNo
	   ,Position
	   );
    };

    class PageWriterImpl
      : public Logical::PageWriter
    {
      friend class	Logical;
      friend class	LogicalImpl;

    private:
      PageWriterImpl(
		     LogicalImpl &
		     );

      ~PageWriterImpl()
      {}

    private:
      LogicalImpl &		logical_;
      Page *			page_;
    };
    
    class PageReaderImpl
      : public Logical::PageReader
    {
      friend class	Logical;
      friend class	LogicalImpl;

    private:
      PageReaderImpl(
		     LogicalImpl &
		     );

      ~PageReaderImpl()
      {}

    private:
      LogicalImpl &		logical_;
      PacketImpl *		page_;

    };

  private:
    Logical *			intf_;
    TransportToLogical *	transport_;
    long			serialNo_;

    std::queue<PacketImpl *>	packets_;
    pthread_cond_t		condEmpty_;
    pthread_cond_t		condFull_;
    pthread_mutex_t		lockPackets_;
    WriterImpl *		writer_;
    ReaderImpl *		reader_;

    std::queue<Page *>		pages_;
    pthread_cond_t		condPagesEmpty_;
    pthread_cond_t		condPagesFull_;
    pthread_mutex_t		lockPages_;
    PageWriterImpl *		pageWriter_;
    PageReaderImpl *		pageReader_;
    bool			selected_;

    enum Services
      {
	NONE = 0
	,DECAP = 1
	,READPAGE = 2
	,DECAPPAGE = 3
	,ENCAP = 4
	,WRITPAGE = 8
	,ENCAPPAGE = 12
      }
      ;

    Services			services_;
    
  private:
    ///@brief For sending a Logical stream
    LogicalImpl(
		Transport &
		,long	serialNo
		);

    ///@brief Withdraw from transport
    ~LogicalImpl();

    void
    select();
    
  };
  /// @} logical
};
#endif
