/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Examples::Que - implementation                                            */
/*                                                                           */
/*===========================================================================*/
#ifdef __GNUG__
#pragma implementation
#endif

#include	<Que.H>
#include	<persist/GetPutDelTemplates.H>
#include	<persist/RelocateTemplates.H>

namespace	Persistence
{
  using namespace Examples;

  //
  // instantiate template functions
  //
  template
  Que *	get(
	    const char *	name
	    );

  template
  Que *	get(
	    const Id
	    );
  template
  Status	put(
		    Que &
		    );
  template
  Status	del(
		    Que &
		    );
  template
  Que *	first();

  template
  Que *	next(const Que	&);

  template
  Que *	relocate(
		 Que &
		 );

  template
  void		swizzle(
			Que &
			);

}


namespace	Examples
{
  using namespace Persistence;

  Id		Que::ClassId(ExampleQue);
  Name		Que::ClassName("Que");
  LongName	Que::ClassDecl(QueClassDecl);
  LongName	Que::ClassImpl("Que.C $Id$ keyword substitution from version control");

  void
  Que::initReadWrite()
  {
    pthread_mutex_init(&writeLock, 0);
    pthread_cond_init(&writeCond, 0 );
    pthread_mutex_init(&readLock, 0);
    pthread_cond_init(&readCond, 0 );

    writeInc = 0;
    writeArea = area;
    writeOffset = 0;
    writeEnded = false;

    readInc = 0;
    readArea = 0;
    readOffset = 0;

  }

  Status
  Que::start()
  {
    Status	s;

    if ( outputFilter == 0 )
      {
	s = Status("Que"
		   ,"start"
		   ,cname()
		   );
	s << " has no output filter!" << std::endl;
	return(s);
      }

    if ( !outputFilter->started() )
      {
	s &= outputFilter->start();
      }

    return(s);
  }

  Status
  Que::finish()
  {
    Status	s;

    if ( outputFilter == 0 )
      {
	s = Status("Que"
		   ,"finish"
		   ,cname()
		   );
	s << " has no output filter!" << std::endl;
	return(s);
      }

    if ( inputFilter == 0 )
      {
	s = Status("Que"
		   ,"finish"
		   ,cname()
		   );
	s << " has no output filter!" << std::endl;
	return(s);
      }

    if ( !outputFilter->finished() )
      {
	s &= outputFilter->finish();
      }

    if ( !inputFilter->finished() )
      {
	s &= inputFilter->finish();
      }

    return(s);
  }

  Que::Que(
	   const char *	nm
	   ,size_t		width
	   ,size_t		len
	   )
    :Obj(nm, classId()) 
    ,inputFilter(0,0)
    ,outputFilter(0)
    ,length(len)
    ,frameWidth(width)
    ,area(len*width)
    ,rframes(0)
  {
    init(
	 ClassId		// Class ClassId
	 ,ClassName.cstr()
	 ,ClassDecl.cstr()
	 ,ClassImpl.cstr()	// Class Class Impl
	 ,0			// don't keep track of Que objects
	 );
  }
      
  Que::iterwriter
  Que::beginWrite()
  {
    if ( !rframes )
      {
	rframes = new Array(area);
      }

    // wait for everything written to be raed
    pthread_mutex_lock(&writeLock);
    {
      while ( area > writeArea )
	pthread_cond_wait(&writeCond, &writeLock);
    }
    pthread_mutex_unlock(&writeLock);

    write();

    return(*this);
  }

  void
  Que::wrote()
  {
    if ( 0 < writeInc )
      {
	pthread_mutex_lock(&readLock);
	{
	  readArea += writeInc;
	  pthread_cond_signal(&readCond);
	}
	pthread_mutex_unlock(&readLock);
	
	writeOffset += writeInc;
	if ( area <= writeOffset )
	  writeOffset -= area;
      }
  }

  size_t
  Que::write()
  {
    if ( writeEnded )
      return(writeOffset);

    writeInc = frameWidth;

    pthread_mutex_lock(&writeLock);
    {
      while ( writeInc > writeArea )
	pthread_cond_wait(&writeCond, &writeLock);
    
      writeArea -= writeInc;
    }
    pthread_mutex_unlock(&writeLock);

    return(writeOffset);
  }

  void
  Que::endWrite()
  {
    writeInc = 0;

    pthread_mutex_lock(&readLock);
    {
      writeEnded = true;
      pthread_cond_signal(&readCond);
    }
    pthread_mutex_unlock(&readLock);
  }

  Que::itereader
  Que::beginRead()
  {
    // no different from normal iteration
    raed();
    read();
    return(*this);
  }

  void
  Que::raed()
  {
    if ( 0 < readInc )
      {
    	pthread_mutex_lock(&writeLock);
	{
	  writeArea += readInc;
	  pthread_cond_signal(&writeCond);
	}
	pthread_mutex_unlock(&writeLock);

	readOffset += readInc;
	if ( area <= readOffset )
	  readOffset -= area;
      }
  }

  size_t
  Que::read()
  {
    readInc = frameWidth;

    pthread_mutex_lock(&readLock);
    {
      while ( !writeEnded
	      && (readArea < readInc)
	      )
	{
	  pthread_cond_wait(&readCond, &readLock);
	}
      if ( readInc <= readArea )
	{
	  readArea -= readInc;
	}
      else if ( writeEnded )
	{
	  readInc = 0;
	}
    }
    pthread_mutex_unlock(&readLock);

    return(readOffset);
    // if writeEnded, incrementing an itereader
    // would cause it to equal endRead()
  }

  Que::itereader
  Que::endRead()
  {
    bool ended;

    pthread_mutex_lock(&readLock);
    {
      ended = writeEnded;
    }
    pthread_mutex_unlock(&readLock);

    if ( ended
	 &&
	 (0 == readInc)
	 )
      return(*this);
    else
      return(itereader(*this,area));
  }

  //------------------------------------------------------------------------------
  Status
  Que::put()
  {
    Status s(Persistence::put<Que>(*this));

    if ( s.succeeded() )
      return(s);
    else
      {
	Status sta("Que"
		   ,"put"
		   ,cname()
		   );
	sta << " failed" << std::endl;
	sta &= s;
	return(sta);
      }
  }

  Status
  Que::del()
  {
    Status s(Persistence::del<Que>(*this));
    
    if ( s.succeeded() )
      return(s);
    else
      {
	Status sta("Que"
		   ,"del"
		   ,cname()
		   );
	sta << " failed" << std::endl;
	sta &= s;
	return(sta);
      }
  }

  Que *
  Que::relocate()
  {
    return(Persistence::relocate<Que>(*this));
  }

  //------------------------------------------------------------------------------
  void	Que::dump(
		   ostream &	out
		   ) const
  {
    this->Object::dump(out);

    if ( input() )
      out << "Input  filter :" << input()->cname()  << std::endl;

    if ( output() )
      out << "Output filter :" << output()->cname() << std::endl;

  }
}
