/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* basic Persistence:: types                                                 */
/*                                                                           */
/*===========================================================================*/
#ifndef	PersistBasic
#define	PersistBasic

#ifdef __GNUG__
#pragma interface
#endif

#include	<cstring>
#include	<string>
#include	<sstream>
#include	<persist/Exception.H>

namespace Persistence
{
  typedef unsigned int	Id;		// makes this compatible with std::key_t

  //
  // for embedded names
  //
  template<size_t capacity>	class Naming
  {
  public:
    char	s[capacity];

    Naming() throw()
    {
    }

    Naming(const char * cstr) throw()
    {
      strncpy(s,cstr,capacity);
      s[capacity-1]='\0';
    }

    Naming(const Naming & nm) throw()
    {
      strncpy(s,nm.s,capacity);
      s[capacity-1]='\0';
    }

    const char * cstr() const throw()
    {
      return(s);
    }

    size_t size() throw()
    {
      return(strlen(s));
    }

    Naming & operator=(const char * cstr) throw()
    {
      strncpy(s,cstr,capacity);
      s[capacity-1]='\0';
      return(*this);
    }

    Naming & operator+=(const char * cstr) throw()
    {
      strncpy(s+size(), cstr, capacity-size());
      s[capacity-1]='\0';
      return(*this);
    }

    Naming operator+(const char * cstr) const throw()
    {
      Naming	conc(s);
      
      conc += cstr;
      return(conc);
    }

    Naming operator+(const Naming & nm) const throw()
    {
      return(operator+(nm.cstr()));
    }

    bool operator==(const char * cs) const throw()
    {
      return(std::string(cstr()) == std::string(cs));
    }

    bool operator!=(const char * cs) const throw()
    {
      return(std::string(cstr()) != std::string(cs));
    }

    Naming & operator=(const Naming & name) throw()
    {
      *this = name.cstr();
      return(*this);
    }
    bool operator==(const Naming & name) const throw()
    {
      return(*this == name.cstr());
    }

    bool operator!=(const Naming & name) const throw()
    {
      return(!(operator==(name.cstr())));
    }
  }
  ;

  template<size_t capacity>
  inline
  std::ostream &
  operator<<(
	     std::ostream &		out
	     ,const Naming<capacity> &	nm
	     )
  {
    out << nm.cstr();
    return(out);
  }

  typedef Naming<48> 	Name;
  typedef Naming<64>	LongName;
  typedef Naming<256>	Path;

  template
  inline
  std::ostream &
  operator<<(
	     std::ostream &	out
	     , const Name &	nm
	     );
  template
  inline
  std::ostream &
  operator<<(
	     std::ostream &	out
	     , const LongName &	nm
	     );
  template
  inline
  std::ostream &
  operator<<(
	     std::ostream &	out
	     , const Path &	nm
	     );
  

  
  class Status : public Exception
  {
  public:
    enum Return {
      SUCCESS = 0
      , FAILED = -1
      , STOPPED = -2
    }
      ;

  private:
    Return			return_;

  public:
    Status() throw()
    : Exception()
      , return_(SUCCESS)
    {}

    Status(
	   const char *		nm
	   ,const char *	clasnm
	   ,const char *	funnm
	   ,const char *	objnm = ""
	   ,const Return	ret = FAILED
	   ) throw()
      : Exception(nm, clasnm, funnm, objnm)
      , return_(ret)
    {}

    Status(
	   const bool suc
	   ) throw()
      : Exception()
      , return_(suc? SUCCESS:FAILED)
    {}

    Status(
	   const Status &	st
	   ) throw()
      : Exception(st)
      , return_(st.return_)
    {}

    virtual
    ~Status() throw ()
    {}

    void
    succeeded(bool suc) throw()
    {
      return_ = suc?SUCCESS:FAILED;
    }

    bool
    succeeded() throw()
    {
      return(return_ == SUCCESS);
    }

    bool
    failed() throw()
    {
      return(!succeeded());
    }
    
    Status&
    operator=(const Status & s);

    Status&
    operator&=(const Status & s);
  }
  ;
  
  inline
  std::ostream &
  operator<<(
	     std::ostream &	out
	     ,const Status &	s
	     )
  {
    out << s.what();
    return(out);
  }
    
}
#endif
