/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::List template class interface                                */
/*                                                                           */
/*===========================================================================*/
#ifndef	PersistList
#define	PersistList	1

#ifdef __GNUG__
#pragma interface
#endif

#include	<persist/Object.H>
#include	<persist/DtorCommand.H>
#include	<persist/GetPutDel.H>
#include	<persist/Relocate.H>
#include	<persist/Exception.H>

namespace	Persistence
{
  class OutOfRange : public Exception
  {
  private:
    Naming<1024>		message; 
  public:
    OutOfRange(
	       const		char *	funnm
	       ,const		char *	objnm
	       ) throw()
      ;

    OutOfRange(
	       OutOfRange & ex
	       ) throw()
      : Exception(ex)
    {}

    ~OutOfRange() throw()
    {}

  }
  ;

  // Entry should not be a virtual based object.
  // Entry must have the following constructor for a cleared Entry
  // Entry	zero(bool);
  // Entry must also have assignment operator
  // *(i-1) = zero;
  // and equality operator
  // *i == e;

  template<typename Entry>
  class List : public Object, public DtorCommand
  {
  public:
    struct iterator
    {
      Entry *	pEntry;

      iterator() throw()
      {}

      iterator(Entry * pe) throw()
	: pEntry(pe)
      {}

      iterator(const Entry * pe) throw()
	: pEntry(const_cast<Entry *>(pe))
      {}

      iterator(Entry *& pe) throw()
	: pEntry(pe)
      {}

      iterator(const Entry *& pe) throw()
	: pEntry(const_cast<Entry *>(pe))
      {}

      iterator(Entry & e) throw()
	: pEntry(&e)
      {}

      iterator(const Entry & e) throw()
	: pEntry(&e)
      {}

      iterator(iterator& i) throw()
	: pEntry(i.pEntry)
      {}

      iterator(const iterator& i) throw()
	: pEntry(i.pEntry)
      {}

      Entry &
      operator*() const
      {
	return (*pEntry);
      }

      Entry *
      operator->() const throw()
      {
	return(pEntry);
      }

      bool
      operator==(const iterator& i) const throw()
      {
	return( pEntry == i.pEntry );
      }

      bool
      operator!=(const iterator& i) const throw()
      {
	return( pEntry != i.pEntry );
      }
      

      iterator &
      operator++() throw()
      {
	pEntry++;
	return(*this);
      }

      iterator
      operator+=(
		 int	n
		 ) throw()
      {
	pEntry+=n;
	return(*this);
      }


      iterator
      operator+(
		const int	n
		) throw()
      {
	iterator it = iterator(pEntry + n);
	return(it);
      }

      iterator &
      operator--() throw()
      {
	pEntry--;
	return(*this);
      }

      iterator
      operator-(
		const int	n
		) throw()
      {
	iterator it = iterator(pEntry - n);
	return(it);
      }

      iterator
      operator-=(
		 const int	n
		 ) throw()
      {
	pEntry-=n;
	return(*this);
      }

      size_t
      operator-(
		const iterator &	it
		) const throw()
      {
	return(size_t(pEntry - it.pEntry));
      }

      bool
      operator<(
		const iterator &	it
		) const throw()
      {
	return( 0 > (pEntry - it.pEntry) );
      }

      bool
      operator<=(
		 const iterator &	it
		 ) const throw()
      {
	return( 0 >= (pEntry - it.pEntry) );
      }

      bool
      operator>(
		const iterator &	it
		) const throw()
      {
	return( 0 < (pEntry - it.pEntry) );
      }

      bool
      operator>=(
		 const iterator &	it
		 ) const throw()
      {
	return( 0 <= (pEntry - it.pEntry) );
      }

    }
    ;

  private:
    static
    const Id		ClassId = PersistList;

  public:
    static
    Id    		classId() throw()
    {
      return(ClassId);
    }
    
    //
    // object data
    //
  private:
    static
    const size_t	DefaultNumEntries = 7;
    size_t		numEntries;
    size_t		endOff;
    Entry		ents[DefaultNumEntries];


    Entry *
    entries() const throw()
    {
      /*
      List<Entry> * beyond = const_cast<List<Entry> *>(this+1);
      return(reinterpret_cast<Entry *>(beyond));
      */
      return(const_cast<Entry *>(ents));
    }

    /*
  public:
    static
    void* operator new(size_t size
		       ,size_t nEntries
		       )
    {
      return(::operator new(size + nEntries*sizeof(Entry)));
    }
    */

  public:
    List()
      : Obj()
	, DtorCommand()
      // List<Entry> not a full Class, has only ClassId
    {
      if ( 0 >= numEntries )
	{
	  // new, not reconstruction
	  numEntries = DefaultNumEntries;
	  endOff = 0;
	}

      //
      // automatic reconstruction does not know bout
      // entries added with dynamic resize
      // so we strenuously reconstruct them here
      //
      if ( DefaultNumEntries < numEntries )
	{
	  iterator i;
	  size_t j;
	  for ( i = begin()+DefaultNumEntries
		  , j = DefaultNumEntries
		 ;
	       j < numEntries
		 ;
	       ++i
		 , ++j
	       )
	    {
	      new(i.operator->()) Entry;
	    }
	}
    }

    List(
	 const char *	nam
	 ) throw()
      : Obj(nam, ClassId)
	, DtorCommand()
	, numEntries(DefaultNumEntries)
	, endOff(0)
    {}

    List(
	 const char *	nam
	 ,Id	id
	 ) throw()
      : Obj(nam, ClassId, id)
	, DtorCommand()
	, numEntries(DefaultNumEntries)
	, endOff(0)
    {}

    ~List()
    {
      if ( (dtorCommand() == DtorCommand::DEL)
	   ||
	   (dtorCommand() == DtorCommand::DELETE)
	   )
	{
	  // directly delete what is owned
	  iterator i = end();
	  while ( i != begin() )
	    {
	      // ~Entry() should remove itself from this list
	      iterator j = i-1;
	      j->~Entry();
	      if ( j != end() )
		{
		  // No, ~Entry did not.
		  // *i should still behave,
		  // even though it has been destructed
		  // it should still have operator =
		  Entry	zero(true);
		  *j = zero;
		  endOff--;
		}
	      i = j;
	    }
	  // No entries are left
	  return;
	}

      if ( DefaultNumEntries < endOff )
	{
	  iterator it = begin() + DefaultNumEntries-1;
	  for (iterator i = end()-1
		 ;
	       i != it
		 ;
	       --i
	       )
	    {
	      // With Entry e defined in this scope
	      // on exit from this scope
	      // e.~Entry() will automatically be called
	      // but this may be optimized away
	      // so the listed objects must be explicitly processed
	      i->~Entry();
	    }
	}
    }

    List *
    resize(
	   size_t nes = 0
	   )			// note returned list may be relocated
    {
      if ( 0 == nes )
	nes = numEntries + (numEntries>>1);

      if ( nes <= numEntries )
	return(this);
      
      // re-allocate
      // this may throw(std::bad_alloc)
      List<Entry> * newlist =
	static_cast<List<Entry> *>(Obj::operator new(
						     this->objSize
						     + (nes-numEntries)*sizeof(Entry)
						     )
				   );

      // reconstruct
      // this may throw whatever OMan::registr throws
      newlist = new(newlist) List<Entry>(objName.cstr(), objId);
      newlist->numEntries = nes;
      // reconstruct the newlist entries above DefaultNumEntries
      // just so that we can copy to it from the old
      iterator i;
      size_t j;
      for ( i = newlist->begin()+DefaultNumEntries
	      ,j = DefaultNumEntries
	      ;
	    j < nes
	      ;
	    ++i
	      ,++j
	    )
	{
	  new(i.operator->()) Entry;
	}

      *newlist = *this;
      // reconstruct the entries again
      // because they may contain refs to the List
      // whose address has just changed,
      // as in trefs!
      for ( i = newlist->begin()
	      ;
	    i != newlist->end()
	      ;
	    ++i
	    )
	{
	  new(i.operator->()) Entry;
	}

      // newlist has already replaced this in terms of id and name
      // but still in OMan::byAddress
      // Will just have to leave it there
      ::operator delete(static_cast<void *>(this));	// can you do this and return alive? YES!
      return(newlist);

    }

    List &
    operator=(const List & l) throw()
    {
      objClassId = l.objClassId;
      updateState = l.updateState;
      com = l.com;
      sta = l.sta;

      iterator isrc, idest;

      for(
	  idest = begin(),
	    isrc = l.begin()
	    ;
	  ( (isrc != l.end())
	    &&
	    !full()			// may not copy all
	    )
	    ;
	  ++isrc, ++idest
	  )
	{
	  *idest = *isrc;		// operator= must be defined for Entry
	  ++endOff;
	}

      return(*this);
    }

    iterator
    begin() const throw()
    {
      return(entries());
    }

    iterator
    end() const	 throw()		// highest written element
    {
      return(entries()+endOff);		
    }

    bool
    empty() const throw()
    {
      return(0 == endOff);
    }

    bool
    full() const throw()
    {
      return(endOff == numEntries);
    }

    size_t
    size() const throw()
    {
      return(end() - begin());
    }

    size_t
    capacity() const throw()
    {
      return( numEntries );
    }


    Entry &
    operator[](
	       size_t	i
	       ) throw(OutOfRange)
    {
      if ( endOff <= i )
	{
	  OutOfRange ex("operator[]", cname());
	  ex << "readonly index " << i << " greater than last index " << endOff << std::endl;
	  throw ex;
	}

      return(entries()[i]);
	  
    }
    
    iterator
    find(
	 const Entry &	e
	 ) const throw()
    {
      iterator i;

      for ( i = begin();
	    i != end();
	    ++i
	    )
	{
	  if ( *i == e )
	    return(i);
	}
      return(i);
    }

    size_t
    which(
	  const iterator	it
	  ) const
    {
      return(it - begin());
    }
    
    void
    insert(
	   const Entry &	e
	   ,const iterator	it
	   ) throw(OutOfRange)
    {
      if ( full() )
	{
	  OutOfRange ex("insert", cname());
	  ex << "attempt to insert beyond capacity of " << numEntries << std::endl;
	  ex << "caller should check List::full() before inserting";
	  throw ex;
	}


      if ( (it < begin())
	   ||
	   (end() < it)
	   )
	{
	  OutOfRange ex("insert", cname());
	  ex << "attempt to insert at index " << which(it) << std::endl;
	  throw ex;
	}
      else
	{
	  // push up to open slot
	  for(
	      iterator i = end()-1;
	      it < i;
	      --i
	      )
	    {
	      *i = *(i-1);
	    }

	  *it = e;
	  endOff++;

	  update();
	}
    }

    void
    insert(
	   const Entry &	e
	   ) throw(OutOfRange)
    {
      insert(e, end());
    }

    void
    remove(
	   const Entry &	e
	   )
    {
      iterator it = find(e);
      if ( it >= end() )
	return;

      iterator i;

      // push down to close slot
      for(
	  i = it+1;
	  i < end();
	  ++i
	  )
	{
	  *(i-1) = *i;
	}
      // Any Entry must have the following constructor for a cleared Entry
      Entry	zero(true);
      // Entry must also have assignment operator
      *(i-1) = zero;
      endOff--;
      update();
    }

    Status	put()
    {
      return(Persistence::put<List>(*this));
    }

    Status	del()
    {
      return(Persistence::del<List>(*this));
    }
    
    List *	relocate()
    {
      return(Persistence::relocate<List>(*this));
    }

    void	swizzle()
    {
      Persistence::swizzle<List>(*this);
    }
  };

}
#endif
