/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::Object base class interface                                  */
/*                                                                           */
/*===========================================================================*/
#ifndef PersistObject
#define	PersistObject

#ifdef __GNUG__
#pragma interface
#endif

#include	<persist/Basic.H>

namespace	Persistence
{
  class		Object;
  typedef	Object		Obj;
}
#include	<iostream>

namespace	Persistence
{

  using std::ostream;

  class Object
  {
    //
    // object attributes
    //
  protected:
    Id			objId;
    Name		objName;
    Id			objClassId;
    Id			reconClassId;
    ptrdiff_t		headDiff_;
    size_t		objSize;
    bool		updateState;
    					    
    //
    // class methods
    //
  public:
    static
    Id		classId()
    {
      return(0);
    }

    static
    Obj *		fullyReconstruct(Obj *);

    static
    void *		operator new(			// for all derived classes
				     size_t
				     ) throw(std::bad_alloc);
    static
    void *		operator new(			// placement for reconstruct
				     size_t
				     ,void *
				     ) throw();
    static
    void		operator delete(		// for all derived classes
					void *
					);
  
    static
    Obj *		base(void * 		head
			     , ptrdiff_t	headDiff);

    //
    // object methods
    //
  public:
    Object();

    explicit
    Object(
	   const char *	nam
	   ,Id		cid
	   ,Id		i		// for constructing Class objects
	   );

    explicit
    Object(
	   const char *	nam
	   ,Id		cid		// for ordinary objects
	   ,void *	head = 0	// head of virtual Object base
	   ,size_t	fullSize = 0	// fully derived size for virtual Object base
	   );
    
    virtual
    ~Object()
    {
      // nefariously do not unregistr here
      // in reverse to Object( ... )
      // because we are using ~X() for the nefarious purpose
      // of cascading other ~Y() via ~tref()
      // instead Obj::delete is used to unregistr
    }

    const Id	id() const throw()
    {
      return(objId);
    }

    const Name& name() const throw()
    {
      return(objName);
    }

    const char*	cname() const throw()
    {
      return(objName.cstr());
    }

    const Id	clid() const throw()
    {
      return(objClassId);
    }

    const Id	reclid() const throw()
      // reconstructed class id
    {
      return(reconClassId);
    }

    void	reclid(const Id i) throw()
    {
      reconClassId = i;
    }

    void * 	head() throw()
    {
      return(reinterpret_cast<char *>(this) - headDiff_);
    }

    ptrdiff_t	virtualBase() throw()
    {
      return( headDiff_ );
    }

    size_t 	putSize() const throw()
    {
      return(objSize);
    }

    bool	operator==(const Id i) const throw()
    {
      return( objId == i);
    }
    bool	operator==(const Obj & o) const throw()
    {
      return( objId == o.objId);
    }
    
    void	update() throw()
    {
      updateState = true;
    }

    void	dontUpdate() throw()
    {
      updateState = false;
    }

    bool	toBeUpdated() const throw()
    {
      return(updateState);
    }


    virtual
    Status	put();
    
    virtual
    Status	del();

    virtual
    Obj *	relocate();

    virtual
    void	swizzle()
    {}

    virtual
    void	dump(
		     ostream &	out
		     ) const;
  }
  ;

  inline
  ostream&
  operator<<(ostream & os, Obj & obj)
  {
    obj.dump(os);
    return(os);
  }

}

#endif
