/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::mref multiple reference template class interface             */
/*                                                                           */
/*===========================================================================*/
#ifndef	PersistMReference
#define	PersistMReference

#ifdef __GNUG__
#pragma interface
#endif

#include	<persist/tref.H>
#include	<persist/List.H>
#include	<persist/DtorCommand.H>

namespace	Persistence
{
  class DtorObject : public Object, public DtorCommand
  {
  public:
    ~DtorObject()
    {}
  }
  ;
  typedef	DtorObject	DObj;

  template<class Owner, class Cla>
  class mref : public tref<Owner, List< tref<DObj, Cla> > >
  {
  public:
    typedef	tref<DObj, Cla>			Tref;
    typedef	List<Tref>			ListRef;
    typedef	tref<Owner,ListRef>		refList;
    typedef	tref<ListRef, Cla>		Ref;


    typedef	typename ListRef::iterator	iRef;
    struct	iterator : public iRef
    {
      iterator() throw()
      {}
      
      iterator(iRef i) throw()
	: iRef(i)
      {}

      Cla *
      operator->() const
      {
	// first dereference to Ref
	const Tref & r = iRef::operator*();
	// then get the pointer in r
	Cla * o = dynamic_cast<Cla *>(r.operator->());

	return(o);
      }


      Cla &
      operator*() const
      {
	// now doubly dereference to Cla
	Cla & o = *(operator->());
	return(o);
      }
    }
    ;

  public:
    mref()
      : refList()/* for reconstruction
		    and for deferring List creation
		    when mrefs are embedded in classes
		 */
    {}

    mref(
	 Id	own
	 ,Id	nee
	 )
      : refList(own, nee)
      // used for zeroing
    {}
      

    mref(
	 Owner &	own
	 ,Name		nm
	 ,size_t 	size = 0
	 )
      : refList(0, 0)
    {
      ListRef * lst = new ListRef(nm.cstr());		// default List size

      if ( lst->capacity() < size )
	{
	  lst = lst->resize(size);
	}

      refList r(own, *lst);

      this->refList::operator =(r);
    }

    ~mref()
    {}

    mref &
    operator = (Id	i)
    {
      ref<ListRef *>::operator = (i);
      return(*this);
    }

    bool
    operator == (Id i) const throw()
    {
      return(ref<ListRef *>::operator == (i));
    }

    bool
    operator != (Id i) const throw()
    {
      return( ! operator == (i) );
    }

    Owner *
    owner()
    {
      return(refList::owner());
    }

    iterator
    begin() throw()
    {
      return((*this)->begin());
    }

    const iterator
    begin() const throw()
    {
      return(iterator((*this)->begin()));
    }

    iterator
    end() throw()		// highest written element
    {
      return((*this)->end());		
    }

    const iterator
    end() const throw()
    {
      return((*this)->end());
    }

    bool
    empty() const throw()
    {
      return((*this)->empty());
    }

    bool
    full() const throw()
    {
      return((*this)->full());
    }

    size_t
    size() const throw()
    {
      return((*this)->size());
    }

    size_t
    capacity() const throw()
    {
      return((*this)->capacity());
    }


    Cla &
    operator[](
	       size_t	i
	       )
    {
      ListRef & l = *(*this);
      Tref & r = l[i];
      Cla & o = *(dynamic_cast<Cla *>(r.operator->()));
      return(o);
    }
    
    iterator
    find(
	 Cla &	o
	 )
    {
      DObj * l = reinterpret_cast<DObj *>((*this).operator->());
      Tref r(*l, o);

      iterator i = (*this)->find(r);
      // don't want ~Tref() to destroy anything
      r = Tref(0, 0); 
      return(i);
    }

    size_t
    which(
	  const iterator &	it
	  ) throw()
    {
      return((*this)->which(it));
    }

    virtual
    void
    insert(
	   Ref &		r
	   ,const iterator &	it = end()
	   )
    {
      iterator i = it;

      if ( (*this)->full() )
	{
	  // argment list, but remember offset
	  // because it won't be good after resize!
	  size_t offset = it - begin();
	  ListRef * l = (*this)->resize();
	  Owner * onr = (*this).owner();
	  
	  this->refList::operator = (refList(
					     *onr
					     ,*l
					     )
				     );
	  
	  i = (l->begin()) + offset;
	}

      
      /*iRef * pi =
	static_cast<iRef *>(&i);
      */
      iRef ir = i;
      
      Tref & tr = *(reinterpret_cast<Tref *>(&r));

      (*this)->insert(tr
		      ,ir
		      );
    }

    /*
    void
    insert(
	   Cla *		o
	   ,const iterator &	it = end()
	   )
    {
      ListRef * l = (*this).operator->();

      Ref r(*l, o);

      insert(r
	     ,it
	     );
    }
    */

    virtual
    void
    insert(
	   Cla &		o
	   ,const iterator &	it
	   )
    {
      ListRef * l = (*this).operator->();

      Ref r(*l, o);

      insert(r
	     ,it
	     );
    }

    virtual
    void
    insert(
	   Cla &		o
	   )
    {
      insert(o, end());
    }

    virtual
    void
    remove(
	   Cla &		o
	   )
    {
      DObj * l = reinterpret_cast<DObj *>((*this).operator->());
      Tref r(*l, o);

      (*this)->remove(r);
      // don't let ~Tref() destroy anything
      r = Tref(0,0);
    }

    void
    clear()
    {
      for ( iterator it = begin()
	      ;
	    it != end()
	      ;
	    it = begin()
	    )
	{
	  remove(*it);
	}
    }

    bool found(Cla & o)
    {
      return( find(o) < end() );
    }

  }
  ;

  template<class Owner, class Cla>
  class Owns_objs : public DtorCommand
  {
  public:
    typedef mref<Owner, Cla>			mref_objs;
    typedef typename mref_objs::iterator	i_objs;

  protected:
    mref_objs	objs_;

    mref_objs &
    objs()
    {
      return(objs_);
    }

  public:
    Owns_objs()
      : DtorCommand()
	, objs_()
    {}

    Owns_objs(
	      size_t	nclas		// to distinguish it from Owns_objs()
	      )
      : DtorCommand()
	, objs_(0,0)			// zeroes objs_
    {}

    virtual
    ~Owns_objs()
    {}

    bool
    has_objs() const
    {
      return(objs_ != Id(0));
    }

    i_objs
    begin_objs() const
    {
      return(objs_.begin());
    }

    i_objs
    end_objs() const
    {
      return(objs_.end());		
    }

    bool
    empty_objs() const
    {
      return(objs_.empty());
    }

    size_t
    size_objs() const
    {
      return(objs_.size());
    }
    
    Cla &
    operator[](
	       size_t	i
	       )
    {
      return(objs_[i]);

    }
    
    i_objs
    find_obj(
	 Cla &	o
	 )
    {
      return(objs_.find(o));
    }

    void
    insert_obj(
	   Cla &		o
	   ,const i_objs &	it
	   )
    {
      objs_.insert(o, it);
    }

    void
    insert_obj(
	   Cla &	o
	   )
    {
      insert_obj(o
	     ,objs_.end()
	     );
    }

    void
    remove_obj(
	   Cla &		o
	   )
    {
      objs_.remove(o);
    }


    bool found_obj(Cla & o)
    {
      return( objs_.found(o) );
    }

    virtual
    void	dump_objs(
		     ostream &	out
		     ) const
    {
      if ( has_objs() )
      {
	short j;
	i_objs i;
	for ( i = begin_objs()
		,j = 0
		;
	      i != end_objs()
		;
	      ++i
		,++j
	      )
	  {
	    out << "part " << j << std::endl;
	    i->Obj::dump(out);
	  }
      }
    }

  }
  ;

}

#define	mDefineMrefOwner(margOwnerClassName, margOwnedClassName, margOwnedObjectName)			\
  class Owns_##margOwnedObjectName##s : public virtual DtorCommand					\
  {													\
  public:												\
    typedef mref<margOwnerClassName, margOwnedClassName>	mref_##margOwnedObjectName##s;		\
    typedef mref_##margOwnedObjectName##s::iterator		i_##margOwnedObjectName##s;		\
													\
  protected:												\
    mref_##margOwnedObjectName##s	margOwnedObjectName##s_;					\
													\
    mref_##margOwnedObjectName##s &									\
    margOwnedObjectName##s()										\
    {													\
      return(margOwnedObjectName##s_);									\
    }													\
													\
  public:												\
    Owns_##margOwnedObjectName##s()									\
    : DtorCommand()											\
      , margOwnedObjectName##s_()									\
    {}													\
													\
    Owns_##margOwnedObjectName##s(									\
				  size_t	n)							\
    : DtorCommand()											\
      , margOwnedObjectName##s_(0,0)									\
    {}													\
													\
    virtual												\
    ~Owns_##margOwnedObjectName##s()									\
    {}													\
													\
    bool												\
    has_##margOwnedObjectName##s() const								\
    {													\
      return(margOwnedObjectName##s_ != Id(0));								\
    }													\
													\
    i_##margOwnedObjectName##s										\
    begin_##margOwnedObjectName##s() const								\
    {													\
      return(margOwnedObjectName##s_.begin());								\
    }													\
													\
    i_##margOwnedObjectName##s										\
    end_##margOwnedObjectName##s() const								\
    {													\
      return(margOwnedObjectName##s_.end());								\
    }													\
													\
    bool												\
    empty_##margOwnedObjectName##s() const								\
    {													\
      return(margOwnedObjectName##s_.empty());								\
    }													\
													\
    size_t												\
    size_##margOwnedObjectName##s() const								\
    {													\
      return(margOwnedObjectName##s_.size());								\
    }													\
    													\
    margOwnedClassName &										\
    operator[](												\
	       size_t	i)										\
    {													\
      return(margOwnedObjectName##s_[i]);								\
    }													\
    													\
    i_##margOwnedObjectName##s										\
    find_##margOwnedObjectName(										\
			      margOwnedClassName &	o)						\
    {													\
      return(margOwnedObjectName##s_.find(o));								\
    }													\
													\
    void												\
    insert_##margOwnedObjectName(									\
				margOwnedClassName &			o				\
				,const i_##margOwnedObjectName##s &	it)				\
    {													\
      margOwnedObjectName##s_.insert(o, it);								\
    }													\
													\
    void												\
    insert_##margOwnedObjectName(									\
				margOwnedClassName &	o)						\
    {													\
      margOwnedObjectName##s_.insert(o									\
				     ,margOwnedObjectName##s_.end()					\
				     );									\
    }													\
													\
    void												\
    remove_##margOwnedObjectName(									\
				margOwnedClassName &		o)					\
    {													\
      margOwnedObjectName##s_.remove(o);								\
    }													\
													\
													\
    bool												\
    found_##margOwnedObjectName(margOwnedClassName & o)							\
    {													\
      return( margOwnedObjectName##s_.found(o) );							\
    }													\
													\
    virtual												\
    void	dump_##margOwnedObjectName##s(								\
					      ostream &	out) const					\
    {													\
      if ( has_##margOwnedObjectName##s() )								\
	{												\
	  short j;											\
	  i_##margOwnedObjectName##s i;									\
	  for ( i = margOwnedObjectName##s_.begin()							\
		  ,j = 0										\
		  ;											\
		i != margOwnedObjectName##s_.end()							\
		  ;											\
		++i											\
		  ,++j											\
		)											\
	    {												\
	      out << #margOwnedObjectName << " " << j << " ; ";						\
	      i->Obj::dump(out);									\
	    }												\
	}												\
    }													\
													\
  }													\
    ;
#endif
