/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::ref 1-1 reference template class interface                   */
/*                                                                           */
/*===========================================================================*/
#ifndef	PersistReference
#define	PersistReference

#include	<iostream>

#include	<persist/Basic.H>
#include	<persist/GetPutDel.H>
#include	<persist/Object.H>

namespace	Persistence
{
  using	std::ostream;

  template<class V>
  class ref
  {
  protected:
    V		object;

  public:
    ref()
    {}

    ~ref()
    {}
  }
    ;

  template<class V> class ref<V *>
  {
  protected:
    Id		id_;
    V *		object_;
   
  public:
    ref()			// called during object reconstruction
    {
      if ( id_ == 0 )
	{
	  object_ = 0;
	}
      else
	{
	  object_ = get<V>(id_);
	  if ( !object_ )
	    {
	      // couldn't get it
	      Object * obj = get<Object>(id_);
	      if ( !obj )
		// actually object not found
		id_ = 0;
	      // Otherwise, some problem with interaction of cyclic refs and duplicate inheritance.
	      // As follows: a SS object contains a ref<T *>, which in turn has a ref<S *>.
	      // SS inherits from S twice. If a SS object is being reconstructed, it will
	      // lead to the reconstruction of the T object, which will in turn lead to the
	      // reconstruction of the S part of SS, which may lead to another call to
	      // reconstruct SS, if S::fullyReconstruct is defined. This infinite looping is stopped
	      // in get<S> on this call, because SS::reclid() == SS::clid() (set by reconstruct<SS>
	      // in the first call), which works for objects without duplicate inheritance.
	      // However, the dynamic_cast<S *> in get<S> fails because the SS object is
	      // not yet fully constructed, i.e., the duplicate inheritance has not been reconstructed.
	      //
	      // Save id_ for another try later, when the cyclicity may have been overcome
	      // precisely because this swizzling has failed, leading to the objects
	      // in the cyclic referencing to be fully reconstructed.
	      // In the mean time, this ref will remain unswizzled.
	      // We choose not to compensate for this failure here by
	      // adding a check and re-swizzling to operator->().
	      // Maybe we should throw an exception here as a prompt to the coder.
	    }
	}
    }

    explicit
    ref(Id i)
      : id_(i)
    {
      // make use of same code
      new(this) ref;
    }

    ref(
	const V & 	o
	) throw()
      : id_(o.id())
      , object_(const_cast<V *>(&o))
    {}
    
    ref(const ref & 	r)
      : id_(r.id_)
      // may be used to check r before assignment or test :
      // r1 = ref(r2)
      // r1 == ref(r2)
    {
      // check to see whether r really points to anything
      new(this) ref;
    }

    virtual
    ~ref()
    {
    }

    virtual
    ref &
    operator = (const ref &	r) throw()
    {
      id_ = r.id_;
      object_ = r.object_;

      return(*this);
    }
    virtual
    bool
    operator == (const ref & r) const throw()
    {
      return( (id_ == r.id_)
	      &&
	      (object_ == r.object_)
	      );
    }
    virtual
    bool operator != (const ref & r) const throw()
    {
      return( ! operator == (r) );
    }

    virtual
    ref &
    operator = (Id	i)
    {
      *this = ref(i);
      return(*this);
    }
    virtual
    bool
    operator == (Id i) const throw()
    {
      return(id_ == i);
    }
    virtual
    bool
    operator != (Id i) const throw()
    {
      return(id_ != i);
    }

    virtual
    ref &
    operator = (const V & o) throw()
    {
      *this = ref(o);
      return(*this);
    }

    virtual
    bool
    operator == (const V & o) const throw()
    {
      if ( *this == o.id() )
	{
	  return( object_ == &o );
	}
      return(false);
    }
    virtual
    bool operator != (const V & o) const throw()
    {
      return( ! (operator == (o)) );
    }
    
    virtual
    V * operator->() const throw()
    {
      return(object_);
    }
    
    virtual
    V & operator*() const
    {
      return(*(operator->()));
    }

  }
  ;

}

#endif
