/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::GoBase OBase-derived, based on gdbm, interface file          */
/*                                                                           */
/*===========================================================================*/
#ifndef	PersistGdbmObjectBase
#define	PersistGdbmObjectBase

#ifdef __GNUG__
#pragma interface
#endif

#include	<persist/OBase.H>
#include	<persist/Exception.H>

#include	<list>
#include	<map>

namespace	Persistence
{
  using std::list;
  using	std::map;

  class BasesClosed : public Exception
  {
  private:
    int		signum;

  public:
    BasesClosed(
		int	sn
	       ) throw()
      ;

    BasesClosed(
		const BasesClosed & ex
		) throw()
      : Exception(ex)
	, signum(ex.signum)
    {}

    ~BasesClosed() throw()
    {}

    int
    signalNumber()
    {
      return(signum);
    }
  }
  ;

  class OpenFailed : public Exception
  {
  public:
    OpenFailed(
	       const char *	path
	       ) throw()
      ;

    OpenFailed(
		const OpenFailed & ex
		) throw()
      : Exception(ex)
    {}

    ~OpenFailed() throw()
    {}
  }
  ;

  class		GdbmObjectBase;
  typedef	GdbmObjectBase		GoBase;

  typedef	list<GoBase *>		listBases;
  typedef	listBases::iterator	iBases;

  class GdbmObjectBase : public ObjectBase
  {
  private:

    struct	tally
    {
      size_t	count;
      size_t	size;
    }
    ;
    
    //
    // class attributes
    //
  private:
    static const
    size_t		gdbmBlocksize = 512;
    static volatile
    sig_atomic_t	HandlingSignal;		// suppress interrupt within handler
    static volatile
    sig_atomic_t	ClosingBases;		// mutually exclusive of HandlingSignal
    static
    int			Interrupt;		// interrupting signal number
    static
    listBases *		OpenBases;		/* list of
						  open Base objects to be closed
						  upon caught signals
						*/
    static
    pthread_mutex_t &	InterruptLock;		// guard on Interrupt, OpenBases

    static
    void		closeBases();
    static
    void		signalHandler(int);
    static
    void		handleInterrupts(
					 // sets up signal handler
					 );

    //
    // class methods
    //
  public:
    static
    void	doNotInterrupt();	/* Should be called by any thread except
					   the thread that instantiated the first GoBase
					   which will handle the interrupts,
					   so that only on thread is handling the interrupts.
					*/
    static
    int		whatInterrupted();
    static
    void	throwIfInterrupted();	/* Not absolutely safe if called by
					   thread that is not also handling the interrupts
					*/


    //
    // object attributes
    //
  private:
    Path	root;			/* root directory of this database */
    Name	name;			/* name of this database */
    Path	path;			/* path of this database */
    GDBM_FILE	gdbmByName;
    GDBM_FILE	gdbmById;
    size_t	nobjs;			/* # of objects */
    size_t	nnames;			/* # of name entries */
    int		access;			/* access mode */
    unsigned int	timeout;
    size_t	maxtrys;
    
  private:
    Id		getId(
		      const char *	nam
		      ) ;

    void	open();
    void	close();

  public:
    //
    // object methods
    //
    GdbmObjectBase(				// amounts to opening the base
		   const char * 	nam
		   ,const char *	roo
		   ,const int		rw = GDBM_WRCREAT
		   ,const size_t	timout = 2
		   ,const size_t	maxtris = 60
		   );
    GdbmObjectBase(				/* amounts to reopening */
		   GoBase &	ob
		   );
    ~GdbmObjectBase();			// amounts to closing
    

    Obj *	get(
		    const Id		id
		    ) ;
      
    Obj *	get(
		    const char *	nam
		    ) ;

    Status	put(
		    Obj &		obj
		    ) ;

    Status	del(
		    Obj &		obj
		    ) ;


    Obj	*	first() ;
    Obj *	next(
		     const Obj &	obj
		     ) ;
    Status	check();
    Status	reorg();
    GoBase &	operator=(
			  GoBase &		src
			  );
  };

}
#endif
