/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::Object base class implementation                             */
/*                                                                           */
/*===========================================================================*/
#ifdef __GNUG__
#pragma implementation
#endif

#include	<persist/Object.H>
#include	<persist/OMan.H>
#include	<persist/OBase.H>

#include	<new>

extern "C"
{
#include <strings.h>
}

namespace	Persistence
{
  //------------------------------------------------------------------------------
  Obj *
  Obj::fullyReconstruct(Obj * o)
  {
    /*
    std::cerr << "static Obj::fullyReconstruct called - Obj "
	      << o->cname() << " not fully reconstructed" << std::endl;
    */

    return(o);
  }

  Obj *
  Obj::base(void *		head
	    , std::ptrdiff_t	headDiff
	    )
  {
    Obj * o = new(reinterpret_cast<char *>(head) + headDiff) Obj;
    if ( o->virtualBase() != headDiff )
      throw Status("Not valid head or headDiff"
		   , "Obj", "base(void * head, ptrdiff_t headDiff)"
		   );
    o->reclid(0);
    return(o);
  }
  //------------------------------------------------------------------------------
  void *
  Obj::operator new(			// for all derived classes
		    std::size_t	siz
		    ) throw(std::bad_alloc)
  {
    Obj * o = reinterpret_cast<Obj *>(::operator new(siz));
    bzero(o, siz);

    o->objSize = siz;
    return(o);
  }
  //------------------------------------------------------------------------------
  void *
  Obj::operator new(
		    size_t	siz
		    ,void *	o
		    ) throw()
  {
    return(o);
  }
  //------------------------------------------------------------------------------
  void
  Obj::operator delete(
		       void *	o
		       )
  {
    // unregister here instead of in ~Object
    OMan * om = OMan::obj();
    if ( om && o )
      {
	om->unregistr(o);

	::operator delete(o);
      }
  }
  //------------------------------------------------------------------------------
  Object::Object()			// for reconstruction
  {
    if ( 0 != id() )
      {
	OMan * om = OMan::obj();
	if ( om )
	  om->registr(*this);
      }
  }

  Object::Object(
		 const char *	nam
		 ,Id		cid
		 ,Id		i	// for constructing Class objects
		 )
    : objId(i)
    , objName(const_cast<char *>(nam))
    , objClassId(cid)
    , reconClassId(cid)
    , headDiff_(0)
    , updateState(true)
  {
    OMan * om = OMan::obj();
    if ( om )
      om->registr(*this);
  }

  Obj::Object(
	      const char *	nam
	      , Id		cid	// for ordinary objects
	      , void *		head
	      , size_t		fullSize
	      )
    : objId(OMan::obj()->idGen())
    , objName(const_cast<char *>(nam))
    , objClassId(cid)
    , reconClassId(cid)
    , headDiff_(0)
    , updateState(true)
  {
    if ( head )
      headDiff_ = reinterpret_cast<char *>(this) - reinterpret_cast<char *>(head);
    if ( fullSize )
      // for virtual Object base
      objSize = fullSize;
    // othewise objSize is as assigned in new

    OMan * om = OMan::obj();
    if ( om )
      om->registr(*this);
  }

  Status
  Object::put()
  {
    OBase * ob = OBase::defaultOBase();
    Status sta;
    bool priorUpdateState = toBeUpdated();

    if (ob && toBeUpdated() )
      {
	updateState = false;
	sta = ob->put(*this);
	if ( sta.succeeded() )
	  {
	    return(sta);
	  }
      }

    updateState = priorUpdateState;
    sta.succeeded(false);
    sta.os() << "object " << cname() << " not saved" << std::endl;

    return(sta);
  }

  Status
  Object::del()
  {
    OBase * ob = OBase::defaultOBase();
    Status sta;
    if ( ob )
      sta = ob->del(*this);
    else
      sta.succeeded(false);

    if ( sta.failed() )
      {
	sta.os() << "object " << cname() << " not deleted" << std::endl;
      }

    Object::operator delete(head());
      
    return(sta);
  }

  //------------------------------------------------------------------------------
  Obj *
  Object::relocate()
  {
    // this could be on another memory node
    void * v = Obj::operator new(objSize);

    memcpy(v
	   ,head()
	   ,objSize
	   );
    Obj * o = Obj::base(v, virtualBase());
    //std::cerr << "Object object " << o->cname() << " @" << this << ", head @" << head() << " relocated to " << v << std::endl;

    // delete without side effect
    // but this object remains an entry in OMan::byAddress
    ::operator delete(head());
    return(o);
  }

  //------------------------------------------------------------------------------
  void		Obj::dump(
			  ostream &	out
			  ) const
  {
    out << "\"" << objName << "\" of ClassId " << objClassId << std::endl;
    /*
    out << "{\n";
    out << "Obj address = " << this << "\n";
    out << "Obj size    = " << sizeof(*this) << "\n";
    out << "Obj id       @ " << &Obj::objId << " = " << objId << "\n";
    out << "Obj name     @ " << &Obj::objName << " = " << objName.cstr() << "\n";
    out << "Obj class id @ " << &Obj::objClassId << " = " << objClassId << "\n";
    out << "Obj size     @ " << &Obj::objSize << " = " << objSize << "\n";
    out << "Obj update   @ " << &Obj::updateState << " = " << updateState << "\n";
    out << "}\n";
    */
  }
  //------------------------------------------------------------------------------
}
