/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence:: get, put, del template function definitions                 */
/*                                                                           */
/*===========================================================================*/
#ifndef PersistGetPutDelTemplates
#define	PersistGetPutDelTemplates

#include	<persist/OMan.H>
#include	<persist/OBase.H>
#include	<persist/DtorCommand.H>

#include	<iostream>

namespace Persistence
{
  //------------------------------------------------------------------------------
  template<class T>
  T *	reconstruct(
		    Obj * 	obj
		    )
    //------------------------------------------------------------------------------
    //  Software-theoretically speaking, reconstruct
    //  is the tear in C++'s OO fabric to accomodate
    //  persistence.
    //
    //  reconstruct<T> will be called by T::get
    //  to reconstruct a T object.
    //
    //
    //------------------------------------------------------------------------------
  {
	// now we should be able to use the Object object
	if ( !T::classId()	// not a final class object
	     || (obj->clid() == T::classId() )
	     )
	  {
	    // set the reclid now
	    // so as to prevent cyclicity
	    // in later attempts to reconstruct more fully
	    obj->reclid(T::classId());
	    if ( obj->clid() == T::classId()		// fully derived
		 ||
		 !obj->virtualBase()			// no virtual Object base
		 )
	      {
		T * r = new(obj->head()) T;		// reserve T() for reconstruction!!
		//std::cerr << "reconstructed " << r->cname() << " as type " << r->reclid() << std::endl;
		return(r);
	      }
	  }
	return( dynamic_cast<T *>(T::fullyReconstruct(obj)) );
  }    

  template<class T>
  T *		get(
		    const char *	name
		    )
  {
    OMan * oman = OMan::obj();
    Obj * obj = 0;

    if ( !(obj = oman->get(name)) )
      {
	OBase * obase = OBase::defaultOBase();
	
	if ( obase &&
	     (obj = obase->get(name))
	     )
	  {
	    return ( reconstruct<T>(obj) );
	  }
      }

    if ( obj )
      {
	 if ( obj->reclid() != obj->clid() )
	   {
	     // A prior reference to partially constructed
	     // has partially reconstructed this object
	     // now a reference to fully constructed
	     // fully reconstructs this object.
	     // If T is not the fully derived class
	     // then this may fail
	     return( reconstruct<T>(obj) );
	   }
	 else
	   {
	     // already fully reconstructed
	     T * tobj;
	     if ( obj->virtualBase() )
	       tobj = dynamic_cast<T *>(obj);
	     else
	       tobj = reinterpret_cast<T *>(obj);
	     return(tobj);
	   }

      }

    return(0);
  }


  template<class T> 
  T *		get(
		    const Id	id
		    )
  {
    OMan * oman = OMan::obj();
    Obj * obj = 0;

    if ( !(obj = oman->get(id)) )
      {
	OBase * obase = OBase::defaultOBase();

	if (obase &&
	    (obj = obase->get(id))
	    )
	  {
	    return ( reconstruct<T>(obj) );
	  }
      }

    if ( obj )
      {
	 if (obj->reclid() != obj->clid())
	   {
	     // A prior reference to partially constructed
	     // has partially reconstructed this object
	     // now a reference to fully constructed
	     // fully reconstructs this object.
	     // If T is not the fully derived class
	     // then this may fail
	     return( reconstruct<T>(obj) );
	   }
	 else
	   {
	     // already fully reconstructed
	     T * tobj;
	     if ( obj->virtualBase() )
	       tobj = dynamic_cast<T *>(obj);
	     else
	       tobj = reinterpret_cast<T *>(obj);
	     return(tobj);
	   }
      }

    return(0);
  }

  template<class T>
  Status	put(
		    T &		o
		    )
  {
    if ( o.dtorCommand() == DtorCommand::PUT )
      {
	// necessary to break cyclicity
	// been here before
	return(false);
      }
    
    Status s(o.Obj::put());

    o.setDtorStatus(true);
    o.setDtorCommand(DtorCommand::PUT);
    Obj * obj = static_cast<Obj *>(&o);
    o.~T();				// recurse via ~tref()
    reconstruct<T>(obj);
    o.resetDtorCommand();

    s &= o.dtorStatus();
    return(s);
  }

  template<class T>
  Status	del(
		    T &		o
		    )
  {
    if ( o.dtorCommand() == DtorCommand::DEL )
      {
	// been here before
	return(false);
      }

    o.setDtorStatus(true);
    o.setDtorCommand(DtorCommand::DEL);
    Obj * obj = static_cast<Obj *>(&o);
    o.~T();				// recurse via ~tref() first
    // need to reconstruct,
    // even though we will annihilate it
    reconstruct<T>(obj);
    o.resetDtorCommand();

    Status s(o.dtorStatus());
    s &= o.Obj::del();

    return(s);
  }

  template<class T>
  T *		first()
  {
    OMan * oman = OMan::obj();

    return(oman->template first<T>());
  }

  template<class T>
  T *		next(
		     const T &	obj
		     )
  {
    OMan * oman = OMan::obj();

    return(oman->template next<T>(obj));
  }
}

#endif
