/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::OMan in memory object manager class interface                */
/*                                                                           */
/*===========================================================================*/
#ifndef	PersistObjectManager
#define	PersistObjectManager

#include	<ext/hash_map>
#include	<map>
#include	<ctime>

#ifdef __GNUG__
#pragma interface
#endif

namespace	Persistence
{
  class		ObjectManager;
  typedef	ObjectManager	OMan;
}


#include	<persist/Basic.H>
#include	<persist/Object.H>
#include	<persist/GetPutDel.H>
#include	<persist/ReadWriteLock.H>

namespace	Persistence
{
  using __gnu_cxx::hash_map;
  using std::map;
    

  class	ObjectManager : private ReadWriteLock
  {
  public:
    class IdGen
    {
    private:
      //            Ids < 100000 used by this system
      static
      const Id	baseUserId = 100000;
      Id		state;
      Id		seed;

    public:
      IdGen()
      {

	time_t	t = std::time(0);

	seed = Id(t);
	std::srand(seed);
      }


      Id operator()()
      {
	OMan *om= OMan::obj();

	Id id	= 0;
	while ( id < baseUserId
		|| Persistence::get<Obj>(id)
		)
	  {
	    om->startWrite();		// guarding state
	    id = Id(rand_r(&state));
	    om->endWrite();
	  }

	return(id);
      }
    };

  private:
    static
    OMan *		oman;

    map<void *, Obj *>				byAddress;
    typedef map<void *, Obj *>::iterator	iByAddress;

    map<std::string, Obj *>			byNames;
    typedef map<std::string, Obj *>::iterator	iByNames;

    hash_map<Id, Obj *>				byIds;
    typedef hash_map<Id, Obj *>::iterator	iByIds;
    
  public:
    //
    // only one in shared memory space
    //
    static
    OMan *	obj();

    IdGen		idGen;

    ObjectManager();
    ~ObjectManager()
    {}

    void	registr(
			Obj &		obj
			);
    void	unregistr(
			  void *	o
			  );
    
    template<class V>
    V *		first()
    {
      startRead();

      for (iByIds i = byIds.begin();
	   i != byIds.end();
	   ++i
	   )
	{
	  Obj * o = i->second;
	  if ( o->clid() == V::classId() )
	    {
	      endRead();
	      return(reconstruct<V>(o));
	    }
	}

      endRead();
      return(0);
    }
      
    template<class V>
    V *		next(
		     const V &		obj
		     )
    {
      startRead();

      iByIds i = byIds.find(obj.id());
      if ( i != byIds.end() )
	{
	  ++i;
	}

      for (
	   ;
	   i != byIds.end()
	     ;
	   ++i
	   )
	{
	  Obj * o = i->second;
	  if ( o->clid() == V::classId() )
	    {
	      endRead();
	      return(reconstruct<V>(o));
	    }
	}

      endRead();
      return(0);
    }
    
    Obj *	get(
		    const char *	nam
		    )
    {
      startRead();

      iByNames i = byNames.find(nam);
      if ( i != byNames.end() )
	{
	  Obj * o = i->second;

	  if ( o &&
	       (std::string(o->cname()) == nam)
	       )
	    {
	      endRead();
	      return(o);
	    }
	}
      endRead();
      return(0);
    }

    Obj *	get(
		    const Id		id
		    )
    {
      startRead();

      iByIds i = byIds.find(id);
      if ( i != byIds.end() )
	{
	  Obj * o = i->second;

	  if ( o && ( *o == id )  )
	    {
	      endRead();
	      return(o);
	    }
	}
      endRead();
      return(0);
    }
    
  }
  ;

}

#endif
