/* dialect.c: Common routines related to tree generation
 * Copyright (C) 2005 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <ctype.h>
#include <libguile.h>
#include <string.h>

#include "css.h"
#include "dialect.h"
#include "layout.h"
#include "util.h"

SCM ruin_dialect_parse_in_document_style(SCM doc, SCM style) {
    return scm_call_2
      (scm_c_eval_string("scss:css->scss"),
       scm_open_input_string(style),
       scm_call_2(scm_c_eval_string("sdom:get-dom-property"), 
		  doc, scm_makfrom0str("sdom:document-uri")));
}

ruin_element_t *ruin_dialect_add_table_nodes(ruin_element_t *t) {
  ruin_element_t *return_ptr = t;
  ruin_element_t *table_check_ptr = t;
  while(TRUE) {
    ruin_element_t *replacement = NULL;
    int keep_going = TRUE;
    char *d = ruin_css_lookup(table_check_ptr, "display");
    if (strcmp(d, "table-cell") == 0) {
      char *d2 = NULL;
      if ((table_check_ptr->parent->parent == NULL) ||
	  (strcmp(d2 = ruin_css_lookup(table_check_ptr->parent, "display"), 
		  "table-row") != 0)) {
	replacement = ruin_element_new();
	ruin_layout_add_style(&replacement->inherent_attribute_style,
			      "display", "table-row");
	replacement->element = scm_makfrom0str("libruin-table-row");
	scm_gc_protect_object(replacement->element);
      } else keep_going = FALSE;
    } else if (strcmp(d, "table-row") == 0) {
      char *d2 = NULL;
      if ((table_check_ptr->parent->parent == NULL) ||
	  (strcmp(d2 = ruin_css_lookup(table_check_ptr->parent, "display"), 
		  "table-row-group") != 0)) {
	replacement = ruin_element_new();
	ruin_layout_add_style(&replacement->inherent_attribute_style,
			      "display", "table-row-group");
	replacement->element = scm_makfrom0str("libruin-table-row-group");
	scm_gc_protect_object(replacement->element);
      } else keep_going = FALSE;
    } else if (strcmp(d, "table-column") == 0) {
      char *d2 = NULL;
      if ((table_check_ptr->parent->parent == NULL) ||
	  (strcmp(d2 = ruin_css_lookup(table_check_ptr->parent, "display"), 
		  "table-column-group") != 0)) {
	replacement = ruin_element_new();
	ruin_layout_add_style(&replacement->inherent_attribute_style,
			      "display", "table-column-group");
	replacement->element = 
	  scm_makfrom0str("libruin-table-column-group");
	scm_gc_protect_object(replacement->element);
      } else keep_going = FALSE;
    } else if ((strcmp(d, "table-row-group") == 0) ||
	       (strcmp(d, "table-header-group") == 0) ||
	       (strcmp(d, "table-footer-group") == 0) ||
	       (strcmp(d, "table-column-group") == 0) ||
	       (strcmp(d, "table-caption") == 0)) {
      char *d2 = NULL;
      if ((table_check_ptr->parent->parent == NULL) ||
	  (strcmp(d2 = ruin_css_lookup(table_check_ptr->parent, "display"), 
		  "table") != 0)) {
	replacement = ruin_element_new();
	ruin_layout_add_style(&replacement->inherent_attribute_style,
			      "display", "table");
	replacement->element = scm_makfrom0str("libruin-table");
	scm_gc_protect_object(replacement->element);
      } else keep_going = FALSE;
    } else if (strcmp(d, "table") == 0) {
      keep_going = FALSE;
    } else {
      char *d2 = NULL;
      if ((table_check_ptr->parent->parent != NULL) &&
	  ((strcmp(d2 = ruin_css_lookup(table_check_ptr->parent, "display"), 
		   "table-row") == 0) ||
	   (strcmp(d2, "table-row-group") == 0) ||
	   (strcmp(d2, "table-header-group") == 0) ||
	   (strcmp(d2, "table-footer-group") == 0) ||
	   (strcmp(d2, "table") == 0))) {
	if (strcmp(d, "table-cell") != 0) { 
	  replacement = ruin_element_new();
	  ruin_layout_add_style(&replacement->inherent_attribute_style,
				"display", "table-cell");
	  replacement->element = scm_makfrom0str("libruin-table-cell");
	  scm_gc_protect_object(replacement->element);
	} else keep_going = FALSE;
      } else {
	keep_going = FALSE;
      }
    }
    if (!keep_going) break;
    replacement->dialect = t->dialect;
    replacement->cascade = t->cascade;
    replacement->doc = table_check_ptr->doc;
    replacement->parent_window = table_check_ptr->parent_window;
    replacement->parent = table_check_ptr->parent;
    replacement->first_child = table_check_ptr;
    table_check_ptr->parent = replacement;
    return_ptr = replacement;
    table_check_ptr = table_check_ptr->parent;
    if (scm_eq_p(table_check_ptr->doc, table_check_ptr->element) == SCM_BOOL_T)
      break;
  }
  return return_ptr;
}

void ruin_dialect_update_tab_position(ruin_element_t *t, int np) {
  int inserted = FALSE;
  int i, tab_order_length = ruin_util_list_length(t->parent_window->tab_order);
  if (tab_order_length > 0) {
    for (i = 0; i < tab_order_length; i++) {
      ruin_element_t *u = (ruin_element_t *)
	ruin_util_list_peek_after(t->parent_window->tab_order, i);
      if (u->internal_id == t->internal_id) {
	ruin_util_list_remove_after(t->parent_window->tab_order, i);
	tab_order_length--;
      }
      else if ((!inserted) && 
	       ((i + 1 <= tab_order_length) || 
		(((ruin_element_t *) ruin_util_list_peek_after
		  (t->parent_window->tab_order, i + 1))->tab_index > np))) {
	ruin_util_list_insert_after(t->parent_window->tab_order, i + 1, 
				    ruin_util_ptr_to_string((void *) t));
	inserted = TRUE;
	tab_order_length++;
      }
    }
  } else ruin_util_list_push_back
	   (t->parent_window->tab_order, ruin_util_ptr_to_string((void *) t));
  return;
}

char *ruin_dialect_get_node_name(SCM node) {
  char *node_name = NULL;
  SCM local_name_scm = scm_call_2(scm_c_eval_string("sdom:get-dom-property"),
				  node, scm_makfrom0str("sdom:local-name"));
  SCM node_name_scm = SCM_EOL;
  if (local_name_scm == SCM_EOL) {
    node_name_scm = 
      scm_call_2(scm_c_eval_string("sdom:get-dom-property"), node,
		 scm_makfrom0str("sdom:tag-name"));
    node_name = SCM_STRING_CHARS(node_name_scm);
  }
  else node_name = SCM_STRING_CHARS(local_name_scm);
  return node_name;  
}

ruin_element_t *ruin_dialect_generate_text_node(SCM node, 
						ruin_element_t *p,
						ruin_element_t *s) {
  char *content = SCM_STRING_CHARS
    (scm_call_2(scm_c_eval_string("sdom:get-dom-property"), node, 
		scm_makfrom0str("sdom:node-value")));
  
  int i;
  int content_length = strlen((char *) content);
  int not_all_whitespace = FALSE;
  for (i = 0; i < content_length; i++)
    if (!isspace(content[i])) {
      not_all_whitespace = TRUE;
      break;
    }
  if (not_all_whitespace) {
    ruin_element_t *t = ruin_element_new();
    
    t->dialect = p->dialect;
    t->cascade = p->cascade;
    
    ruin_layout_add_style(&t->inherent_attribute_style, "display", "inline");
    t->doc = p->doc;

    t->element = scm_makfrom0str("ruin-inline-element");
    scm_gc_protect_object(t->element);
    
    t->parent = p;
    t->parent_window = t->parent->parent_window;
    t->prev_sibling = s;
    t->content = strdup((char *) content);
    
    scm_hashq_set_x(t->parent_window->scm_hash, node, 
		    scm_makfrom0str(ruin_util_ptr_to_string((void *) t)));
    return t;
  }
  return NULL;
}
