/* input.c: Input management routines for libRUIN
 * Copyright (C) 2006 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <curses.h>
#include <libguile.h>

#include "input.h"
#include "window.h"

enum ruin_key_identifier _ruin_input_char_to_key_identifier(int ch) {
  switch(ch) {
  case ' ': return RUIN_KEY_U_0020;
  case '!': return RUIN_KEY_U_0021;
  case '"': return RUIN_KEY_U_0022;
  case '#': return RUIN_KEY_U_0023;
  case '$': return RUIN_KEY_U_0024;
  case '&': return RUIN_KEY_U_0025;
  case '\'': return RUIN_KEY_U_0026;
  case '(': return RUIN_KEY_U_0028;
  case ')': return RUIN_KEY_U_0029;
  case '*': return RUIN_KEY_U_002A;
  case '+': return RUIN_KEY_U_002B;
  case ',': return RUIN_KEY_U_002C;
  case '-': return RUIN_KEY_U_002D;
  case '.': return RUIN_KEY_U_002E;
  case '/': return RUIN_KEY_U_002F;
  case ':': return RUIN_KEY_U_003A;
  case ';': return RUIN_KEY_U_003B;
  case '<': return RUIN_KEY_U_003C;
  case '=': return RUIN_KEY_U_003D;
  case '>': return RUIN_KEY_U_003E;
  case '?': return RUIN_KEY_U_003F;
  case '@': return RUIN_KEY_U_0040;
  case '[': return RUIN_KEY_U_005B;
  case '\\': return RUIN_KEY_U_005C;
  case ']': return RUIN_KEY_U_005D;
  case '^': return RUIN_KEY_U_005E;
  case '_': return RUIN_KEY_U_005F;
  case '`': return RUIN_KEY_U_0060;
  case '{': return RUIN_KEY_U_007B;
  case '|': return RUIN_KEY_U_007C;
  case '}': return RUIN_KEY_U_007D;

  case '0': return RUIN_KEY_U_0030;
  case '1': return RUIN_KEY_U_0031;
  case '2': return RUIN_KEY_U_0032;
  case '3': return RUIN_KEY_U_0033;
  case '4': return RUIN_KEY_U_0034;
  case '5': return RUIN_KEY_U_0035;
  case '6': return RUIN_KEY_U_0036; 
  case '7': return RUIN_KEY_U_0037;
  case '8': return RUIN_KEY_U_0038;
  case '9': return RUIN_KEY_U_0039;

  case 'A': return RUIN_KEY_U_0041;
  case 'B': return RUIN_KEY_U_0042;
  case 'C': return RUIN_KEY_U_0043;
  case 'D': return RUIN_KEY_U_0044;
  case 'E': return RUIN_KEY_U_0045;
  case 'F': return RUIN_KEY_U_0046;
  case 'G': return RUIN_KEY_U_0047;
  case 'H': return RUIN_KEY_U_0048;
  case 'I': return RUIN_KEY_U_0049;
  case 'J': return RUIN_KEY_U_004A;
  case 'K': return RUIN_KEY_U_004B;
  case 'L': return RUIN_KEY_U_004C;
  case 'M': return RUIN_KEY_U_004D;
  case 'N': return RUIN_KEY_U_004E;
  case 'O': return RUIN_KEY_U_004F;
  case 'P': return RUIN_KEY_U_0050;
  case 'Q': return RUIN_KEY_U_0051;
  case 'R': return RUIN_KEY_U_0052;
  case 'S': return RUIN_KEY_U_0053;
  case 'T': return RUIN_KEY_U_0054;
  case 'U': return RUIN_KEY_U_0055;
  case 'V': return RUIN_KEY_U_0056;
  case 'W': return RUIN_KEY_U_0057;
  case 'X': return RUIN_KEY_U_0058;
  case 'Y': return RUIN_KEY_U_0059;
  case 'Z': return RUIN_KEY_U_005A;

  case '\t': return RUIN_KEY_U_0009;

  case KEY_DOWN: return RUIN_KEY_DOWN;
  case KEY_UP: return RUIN_KEY_UP;
  case KEY_LEFT: return RUIN_KEY_LEFT;
  case KEY_RIGHT: return RUIN_KEY_RIGHT;
  case KEY_HOME: return RUIN_KEY_HOME;
  case KEY_BACKSPACE:

  case KEY_F(1): return RUIN_KEY_F1;
  case KEY_F(2): return RUIN_KEY_F2;
  case KEY_F(3): return RUIN_KEY_F3;
  case KEY_F(4): return RUIN_KEY_F4;
  case KEY_F(5): return RUIN_KEY_F5;
  case KEY_F(6): return RUIN_KEY_F6;
  case KEY_F(7): return RUIN_KEY_F7;
  case KEY_F(8): return RUIN_KEY_F8;
  case KEY_F(9): return RUIN_KEY_F9;
  case KEY_F(10): return RUIN_KEY_F10;
  case KEY_F(11): return RUIN_KEY_F11;
  case KEY_F(12): return RUIN_KEY_F12;
  case KEY_F(13): return RUIN_KEY_F13;
  case KEY_F(14): return RUIN_KEY_F14;
  case KEY_F(15): return RUIN_KEY_F15;
  case KEY_F(16): return RUIN_KEY_F16;
  case KEY_F(17): return RUIN_KEY_F17;
  case KEY_F(18): return RUIN_KEY_F18;
  case KEY_F(19): return RUIN_KEY_F19;
  case KEY_F(20): return RUIN_KEY_F20;
  case KEY_F(21): return RUIN_KEY_F21;
  case KEY_F(22): return RUIN_KEY_F22;
  case KEY_F(23): return RUIN_KEY_F23;
  case KEY_F(24): return RUIN_KEY_F24;

  case KEY_ENTER: return RUIN_KEY_ENTER;

  case KEY_LL: 
  case KEY_A1:
  case KEY_A3:
  case KEY_B2:
  case KEY_C1:
  case KEY_C3:
  case KEY_END: return RUIN_KEY_END;

  case KEY_FIND: return RUIN_KEY_FIND;
  case KEY_HELP: return RUIN_KEY_HELP;
  }
  return RUIN_KEY_UNDEFINED + ch;
}

SCM _ruin_key_identifier_to_string(enum ruin_key_identifier k) {
  switch(k) {
  case RUIN_KEY_UNDEFINED: break;
  case RUIN_KEY_ACCEPT: return scm_makfrom0str("sdom:key-accept"); 
  case RUIN_KEY_AGAIN: return scm_makfrom0str("sdom:key-again"); 
  case RUIN_KEY_ALL_CANDIDATES: 
    return scm_makfrom0str("sdom:key-all-candidates");
  case RUIN_KEY_ALPHANUMERIC: return scm_makfrom0str("sdom:key-alphanumeric");
  case RUIN_KEY_ALT: return scm_makfrom0str("sdom:key-alt"); 
  case RUIN_KEY_ALT_GRAPH: return scm_makfrom0str("sdom:key-alt-graph"); 
  case RUIN_KEY_APPS: return scm_makfrom0str("sdom:key-apps"); 
  case RUIN_KEY_ATTN: return scm_makfrom0str("sdom:key-attn"); 
  case RUIN_KEY_BROWSER_BACK: return scm_makfrom0str("sdom:key-browser-back"); 
  case RUIN_KEY_BROWSER_FAVORITES: 
    return scm_makfrom0str("sdom:key-browser-favorites"); 
  case RUIN_KEY_BROWSER_FORWARD: 
    return scm_makfrom0str("sdom:key-browser-forward"); 
  case RUIN_KEY_BROWSER_HOME: return scm_makfrom0str("sdom:key-browser-home"); 
  case RUIN_KEY_BROWSER_REFRESH: 
    return scm_makfrom0str("sdom:key-browser-refresh"); 
  case RUIN_KEY_BROWSER_SEARCH: 
    return scm_makfrom0str("sdom:key-browser-search");
  case RUIN_KEY_BROWSER_STOP: return scm_makfrom0str("sdom:key-browser-stop"); 
  case RUIN_KEY_CAPS_LOCK: return scm_makfrom0str("sdom:key-caps-lock"); 
  case RUIN_KEY_CLEAR: return scm_makfrom0str("sdom:key-clear"); 
  case RUIN_KEY_CODE_INPUT: return scm_makfrom0str("sdom:key-code-input"); 
  case RUIN_KEY_COMPOSE: return scm_makfrom0str("sdom:key-compose"); 
  case RUIN_KEY_CRSEL: return scm_makfrom0str("sdom:key-crsel"); 
  case RUIN_KEY_CONVERT: return scm_makfrom0str("sdom:key-convert"); 
  case RUIN_KEY_COPY: return scm_makfrom0str("sdom:key-copy"); 
  case RUIN_KEY_CUT: return scm_makfrom0str("sdom:key-cut"); 
  case RUIN_KEY_DOWN: return scm_makfrom0str("sdom:key-down"); 
  case RUIN_KEY_END: return scm_makfrom0str("sdom:key-end"); 
  case RUIN_KEY_ENTER: return scm_makfrom0str("sdom:key-enter"); 
  case RUIN_KEY_ERASE_EOF: return scm_makfrom0str("sdom:key-erase-eof"); 
  case RUIN_KEY_EXECUTE: return scm_makfrom0str("sdom:key-execute"); 
  case RUIN_KEY_EXSEL: return scm_makfrom0str("sdom:key-exsel"); 
  case RUIN_KEY_F1: return scm_makfrom0str("sdom:key-f1"); 
  case RUIN_KEY_F2: return scm_makfrom0str("sdom:key-f2"); 
  case RUIN_KEY_F3: return scm_makfrom0str("sdom:key-f3"); 
  case RUIN_KEY_F4: return scm_makfrom0str("sdom:key-f4"); 
  case RUIN_KEY_F5: return scm_makfrom0str("sdom:key-f5"); 
  case RUIN_KEY_F6: return scm_makfrom0str("sdom:key-f6"); 
  case RUIN_KEY_F7: return scm_makfrom0str("sdom:key-f7"); 
  case RUIN_KEY_F8: return scm_makfrom0str("sdom:key-f8"); 
  case RUIN_KEY_F9: return scm_makfrom0str("sdom:key-f9"); 
  case RUIN_KEY_F10: return scm_makfrom0str("sdom:key-f10"); 
  case RUIN_KEY_F11: return scm_makfrom0str("sdom:key-f11"); 
  case RUIN_KEY_F12: return scm_makfrom0str("sdom:key-f12"); 
  case RUIN_KEY_F13: return scm_makfrom0str("sdom:key-f13"); 
  case RUIN_KEY_F14: return scm_makfrom0str("sdom:key-f14"); 
  case RUIN_KEY_F15: return scm_makfrom0str("sdom:key-f15"); 
  case RUIN_KEY_F16: return scm_makfrom0str("sdom:key-f16"); 
  case RUIN_KEY_F17: return scm_makfrom0str("sdom:key-f17"); 
  case RUIN_KEY_F18: return scm_makfrom0str("sdom:key-f18"); 
  case RUIN_KEY_F19: return scm_makfrom0str("sdom:key-f19"); 
  case RUIN_KEY_F20: return scm_makfrom0str("sdom:key-f20"); 
  case RUIN_KEY_F21: return scm_makfrom0str("sdom:key-f21"); 
  case RUIN_KEY_F22: return scm_makfrom0str("sdom:key-f22"); 
  case RUIN_KEY_F23: return scm_makfrom0str("sdom:key-f23"); 
  case RUIN_KEY_F24: return scm_makfrom0str("sdom:key-f24"); 
  case RUIN_KEY_FINAL_MODE: return scm_makfrom0str("sdom:key-final-mode"); 
  case RUIN_KEY_FIND: return scm_makfrom0str("sdom:key-find"); 
  case RUIN_KEY_FULL_WIDTH: return scm_makfrom0str("sdom:key-full-width"); 
  case RUIN_KEY_HALF_WIDTH: return scm_makfrom0str("sdom:key-half-width"); 
  case RUIN_KEY_HANGUL_MODE: return scm_makfrom0str("sdom:key-hangul-mode"); 
  case RUIN_KEY_HANJA_MODE: return scm_makfrom0str("sdom:key-hanja-mode"); 
  case RUIN_KEY_HELP: return scm_makfrom0str("sdom:key-help"); 
  case RUIN_KEY_HIRAGANA: return scm_makfrom0str("sdom:key-hiragana"); 
  case RUIN_KEY_HOME: return scm_makfrom0str("sdom:key-home");
  case RUIN_KEY_INSERT: return scm_makfrom0str("sdom:key-insert"); 
  case RUIN_KEY_JAPANESE_HIRAGANA: 
    return scm_makfrom0str("sdom:key-japenese-hiragana"); 
  case RUIN_KEY_JAPANESE_KATAKANA: 
    return scm_makfrom0str("sdom:key-japanese-katakana"); 
  case RUIN_KEY_JAPANESE_ROMAJI: 
    return scm_makfrom0str("sdom:key-japanese-romaji"); 
  case RUIN_KEY_JUNJA_MODE: return scm_makfrom0str("sdom:key-junja-mode"); 
  case RUIN_KEY_KANA_MODE: return scm_makfrom0str("sdom:key-kana-mode"); 
  case RUIN_KEY_KANJI_MODE: return scm_makfrom0str("sdom:key-kanji-mode"); 
  case RUIN_KEY_KATAKANA: return scm_makfrom0str("sdom:key-katakana"); 
  case RUIN_KEY_LAUNCH_APPLICATION_1: 
    return scm_makfrom0str("sdom:key-launch-application-1"); 
  case RUIN_KEY_LAUNCH_APPLICATION_2: 
    return scm_makfrom0str("sdom:key-launch-application-2"); 
  case RUIN_KEY_LAUNCH_MAIL: return scm_makfrom0str("sdom:key-launch-mail"); 
  case RUIN_KEY_LEFT: return scm_makfrom0str("sdom:key-left"); 
  case RUIN_KEY_META: return scm_makfrom0str("sdom:key-meta"); 
  case RUIN_KEY_MEDIA_NEXT_TRACK: 
    return scm_makfrom0str("sdom:key-media-next-track"); 
  case RUIN_KEY_MEDIA_PLAY_PAUSE: 
    return scm_makfrom0str("sdom:key-media-play-pause"); 
  case RUIN_KEY_MEDIA_PREVIOUS_TRACK: 
    return scm_makfrom0str("sdom:key-media-previous-track"); 
  case RUIN_KEY_MEDIA_STOP: return scm_makfrom0str("sdom:key-media-stop"); 
  case RUIN_KEY_MODE_CHANGE: return scm_makfrom0str("sdom:key-mode-change"); 
  case RUIN_KEY_NONCONVERT: return scm_makfrom0str("sdom:key-nonconvert"); 
  case RUIN_KEY_NUM_LOCK: return scm_makfrom0str("sdom:key-num-lock"); 
  case RUIN_KEY_PAGE_DOWN: return scm_makfrom0str("sdom:key-page-down"); 
  case RUIN_KEY_PAGE_UP: return scm_makfrom0str("sdom:key-page-up"); 
  case RUIN_KEY_PASTE: return scm_makfrom0str("sdom:key-paste"); 
  case RUIN_KEY_PAUSE: return scm_makfrom0str("sdom:key-pause"); 
  case RUIN_KEY_PLAY: return scm_makfrom0str("sdom:key-play"); 
  case RUIN_KEY_PREVIOUS_CANDIDATE: 
    return scm_makfrom0str("sdom:key-previous-candidate"); 
  case RUIN_KEY_PRINT_SCREEN: return scm_makfrom0str("sdom:key-print-screen"); 
  case RUIN_KEY_PROCESS: return scm_makfrom0str("sdom:key-process"); 
  case RUIN_KEY_PROPS: return scm_makfrom0str("sdom:key-props"); 
  case RUIN_KEY_RIGHT: return scm_makfrom0str("sdom:key-right");
  case RUIN_KEY_ROMAN_CHARACTERS: 
    return scm_makfrom0str("sdom:key-roman-characters"); 
  case RUIN_KEY_SCROLL: return scm_makfrom0str("sdom:key-scroll"); 
  case RUIN_KEY_SELECT: return scm_makfrom0str("sdom:key-select"); 
  case RUIN_KEY_SELECT_MEDIA: return scm_makfrom0str("sdom:key-select-media"); 
  case RUIN_KEY_SHIFT: return scm_makfrom0str("sdom:key-shift"); 
  case RUIN_KEY_STOP: return scm_makfrom0str("sdom:key-stop"); 
  case RUIN_KEY_UP: return scm_makfrom0str("sdom:key-up"); 
  case RUIN_KEY_UNDO: return scm_makfrom0str("sdom:key-undo"); 
  case RUIN_KEY_VOLUME_DOWN: return scm_makfrom0str("sdom:key-volume-down"); 
  case RUIN_KEY_VOLUME_MUTE: return scm_makfrom0str("sdom:key-volume-mute"); 
  case RUIN_KEY_VOLUME_UP: return scm_makfrom0str("sdom:key-volume-up"); 
  case RUIN_KEY_WIN: return scm_makfrom0str("sdom:key-win"); 
  case RUIN_KEY_ZOOM: return scm_makfrom0str("sdom:key-zoom"); 

  case RUIN_KEY_U_0008: return scm_makfrom0str("sdom:key-u+0008"); 
  case RUIN_KEY_U_0009: return scm_makfrom0str("sdom:key-u+0009"); 
  case RUIN_KEY_U_0018: return scm_makfrom0str("sdom:key-u+0018"); 
  case RUIN_KEY_U_001B: return scm_makfrom0str("sdom:key-u+001b"); 
  case RUIN_KEY_U_0020: return scm_makfrom0str("sdom:key-u+0020"); 
  case RUIN_KEY_U_0021: return scm_makfrom0str("sdom:key-u+0021"); 
  case RUIN_KEY_U_0022: return scm_makfrom0str("sdom:key-u+0022"); 
  case RUIN_KEY_U_0023: return scm_makfrom0str("sdom:key-u+0023"); 
  case RUIN_KEY_U_0024: return scm_makfrom0str("sdom:key-u+0024"); 
  case RUIN_KEY_U_0025: return scm_makfrom0str("sdom:key-u+0025"); 
  case RUIN_KEY_U_0026: return scm_makfrom0str("sdom:key-u+0026"); 
  case RUIN_KEY_U_0027: return scm_makfrom0str("sdom:key-u+0027"); 
  case RUIN_KEY_U_0028: return scm_makfrom0str("sdom:key-u+0028"); 
  case RUIN_KEY_U_0029: return scm_makfrom0str("sdom:key-u+0029"); 
  case RUIN_KEY_U_002A: return scm_makfrom0str("sdom:key-u+002a"); 
  case RUIN_KEY_U_002B: return scm_makfrom0str("sdom:key-u+002b"); 
  case RUIN_KEY_U_002C: return scm_makfrom0str("sdom:key-u+002c"); 
  case RUIN_KEY_U_002D: return scm_makfrom0str("sdom:key-u+002d"); 
  case RUIN_KEY_U_002E: return scm_makfrom0str("sdom:key-u+002e"); 
  case RUIN_KEY_U_002F: return scm_makfrom0str("sdom:key-u+002f"); 
  case RUIN_KEY_U_0030: return scm_makfrom0str("sdom:key-u+0030"); 
  case RUIN_KEY_U_0031: return scm_makfrom0str("sdom:key-u+0031"); 
  case RUIN_KEY_U_0032: return scm_makfrom0str("sdom:key-u+0032"); 
  case RUIN_KEY_U_0033: return scm_makfrom0str("sdom:key-u+0033"); 
  case RUIN_KEY_U_0034: return scm_makfrom0str("sdom:key-u+0034");  
  case RUIN_KEY_U_0035: return scm_makfrom0str("sdom:key-u+0035"); 
  case RUIN_KEY_U_0036: return scm_makfrom0str("sdom:key-u+0036"); 
  case RUIN_KEY_U_0037: return scm_makfrom0str("sdom:key-u+0037"); 
  case RUIN_KEY_U_0038: return scm_makfrom0str("sdom:key-u+0038"); 
  case RUIN_KEY_U_0039: return scm_makfrom0str("sdom:key-u+0039"); 
  case RUIN_KEY_U_003A: return scm_makfrom0str("sdom:key-u+003a"); 
  case RUIN_KEY_U_003B: return scm_makfrom0str("sdom:key-u+003b"); 
  case RUIN_KEY_U_003C: return scm_makfrom0str("sdom:key-u+003c"); 
  case RUIN_KEY_U_003D: return scm_makfrom0str("sdom:key-u+003d"); 
  case RUIN_KEY_U_003E: return scm_makfrom0str("sdom:key-u+003e"); 
  case RUIN_KEY_U_003F: return scm_makfrom0str("sdom:key-u+003f"); 
  case RUIN_KEY_U_0040: return scm_makfrom0str("sdom:key-u+0040"); 
  case RUIN_KEY_U_0041: return scm_makfrom0str("sdom:key-u+0041"); 
  case RUIN_KEY_U_0042: return scm_makfrom0str("sdom:key-u+0042"); 
  case RUIN_KEY_U_0043: return scm_makfrom0str("sdom:key-u+0043"); 
  case RUIN_KEY_U_0044: return scm_makfrom0str("sdom:key-u+0044"); 
  case RUIN_KEY_U_0045: return scm_makfrom0str("sdom:key-u+0045"); 
  case RUIN_KEY_U_0046: return scm_makfrom0str("sdom:key-u+0046"); 
  case RUIN_KEY_U_0047: return scm_makfrom0str("sdom:key-u+0047"); 
  case RUIN_KEY_U_0048: return scm_makfrom0str("sdom:key-u+0048"); 
  case RUIN_KEY_U_0049: return scm_makfrom0str("sdom:key-u+0049");  
  case RUIN_KEY_U_004A: return scm_makfrom0str("sdom:key-u+004a"); 
  case RUIN_KEY_U_004B: return scm_makfrom0str("sdom:key-u+004b"); 
  case RUIN_KEY_U_004C: return scm_makfrom0str("sdom:key-u+004c"); 
  case RUIN_KEY_U_004D: return scm_makfrom0str("sdom:key-u+004d"); 
  case RUIN_KEY_U_004E: return scm_makfrom0str("sdom:key-u+004e"); 
  case RUIN_KEY_U_004F: return scm_makfrom0str("sdom:key-u+004f"); 
  case RUIN_KEY_U_0050: return scm_makfrom0str("sdom:key-u+0050"); 
  case RUIN_KEY_U_0051: return scm_makfrom0str("sdom:key-u+0051"); 
  case RUIN_KEY_U_0052: return scm_makfrom0str("sdom:key-u+0052"); 
  case RUIN_KEY_U_0053: return scm_makfrom0str("sdom:key-u+0053"); 
  case RUIN_KEY_U_0054: return scm_makfrom0str("sdom:key-u+0054"); 
  case RUIN_KEY_U_0055: return scm_makfrom0str("sdom:key-u+0055"); 
  case RUIN_KEY_U_0056: return scm_makfrom0str("sdom:key-u+0056"); 
  case RUIN_KEY_U_0057: return scm_makfrom0str("sdom:key-u+0057"); 
  case RUIN_KEY_U_0058: return scm_makfrom0str("sdom:key-u+0058"); 
  case RUIN_KEY_U_0059: return scm_makfrom0str("sdom:key-u+0059"); 
  case RUIN_KEY_U_005A: return scm_makfrom0str("sdom:key-u+005a"); 
  case RUIN_KEY_U_005B: return scm_makfrom0str("sdom:key-u+005b"); 
  case RUIN_KEY_U_005C: return scm_makfrom0str("sdom:key-u+005c"); 
  case RUIN_KEY_U_005D: return scm_makfrom0str("sdom:key-u+005d"); 
  case RUIN_KEY_U_005E: return scm_makfrom0str("sdom:key-u+005e"); 
  case RUIN_KEY_U_005F: return scm_makfrom0str("sdom:key-u+005f"); 
  case RUIN_KEY_U_0060: return scm_makfrom0str("sdom:key-u+0060"); 
  case RUIN_KEY_U_007B: return scm_makfrom0str("sdom:key-u+007b"); 
  case RUIN_KEY_U_007C: return scm_makfrom0str("sdom:key-u+007c"); 
  case RUIN_KEY_U_007D: return scm_makfrom0str("sdom:key-u+007d"); 
  case RUIN_KEY_U_007F: return scm_makfrom0str("sdom:key-u+007f"); 
  case RUIN_KEY_U_00A1: return scm_makfrom0str("sdom:key-u+00a1"); 
  case RUIN_KEY_U_0300: return scm_makfrom0str("sdom:key-u+0300"); 
  case RUIN_KEY_U_0301: return scm_makfrom0str("sdom:key-u+0301"); 
  case RUIN_KEY_U_0302: return scm_makfrom0str("sdom:key-u+0302"); 
  case RUIN_KEY_U_0303: return scm_makfrom0str("sdom:key-u+0303"); 
  case RUIN_KEY_U_0304: return scm_makfrom0str("sdom:key-u+0304"); 
  case RUIN_KEY_U_0305: return scm_makfrom0str("sdom:key-u+0305"); 
  case RUIN_KEY_U_0306: return scm_makfrom0str("sdom:key-u+0306"); 
  case RUIN_KEY_U_0307: return scm_makfrom0str("sdom:key-u+0307"); 
  case RUIN_KEY_U_0308: return scm_makfrom0str("sdom:key-u+0308"); 
  case RUIN_KEY_U_030A: return scm_makfrom0str("sdom:key-u+030a"); 
  case RUIN_KEY_U_030B: return scm_makfrom0str("sdom:key-u+030b"); 
  case RUIN_KEY_U_030C: return scm_makfrom0str("sdom:key-u+030c"); 
  case RUIN_KEY_U_0327: return scm_makfrom0str("sdom:key-u+0327"); 
  case RUIN_KEY_U_0328: return scm_makfrom0str("sdom:key-u+0328"); 
  case RUIN_KEY_U_0345: return scm_makfrom0str("sdom:key-u+0345"); 
  case RUIN_KEY_U_20AC: return scm_makfrom0str("sdom:key-u+20ac"); 
  case RUIN_KEY_U_3099: return scm_makfrom0str("sdom:key-u+3099"); 
  case RUIN_KEY_U_309A: return scm_makfrom0str("sdom:key-u+309a"); 
  }
  { char v[2]; v[0] = RUIN_KEY_UNDEFINED - k; v[1] = 0;
    return scm_makfrom0str(v);
  }
}

void ruin_input_send(ruin_window_t *w, int ch) {
  SCM key_str = _ruin_key_identifier_to_string
    (_ruin_input_char_to_key_identifier(ch));

  SCM key_loc = scm_c_eval_string("sdom:dom-key-location-standard");

  ruin_util_log(w, "dispatching event sdom:event-keydown for key '%s'\n",
		SCM_STRING_CHARS(key_str));
  
  scm_apply_0
    (scm_c_eval_string("sdom:dispatch-event"), 
     scm_list_n(w->focused->element, scm_str2symbol("sdom:event-keydown"), 
		SCM_EOL, SCM_EOL, key_str, key_loc, SCM_BOOL_F, SCM_BOOL_F, 
		SCM_BOOL_F, SCM_BOOL_F, SCM_UNDEFINED));

  ruin_util_log(w, "dispatching event sdom:event-keyup for key '%s'\n",
		SCM_STRING_CHARS(key_str));

  scm_apply_0
    (scm_c_eval_string("sdom:dispatch-event"), 
     scm_list_n(w->focused->element, scm_str2symbol("sdom:event-keyup"), 
		SCM_EOL, SCM_EOL, key_str, key_loc, SCM_BOOL_F, SCM_BOOL_F, 
		SCM_BOOL_F, SCM_BOOL_F, SCM_UNDEFINED));

  return;
}


