/* node-fns.c: Node-management functions for the libRUIN Scheme API
 * Copyright (C) 2006 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "node-fns.h"

#include <libguile.h>
#include <string.h>

#include "../css.h"
#include "../libruin.h"
#include "../scheme.h"
#include "../window.h"

SCM ruin_scm_api_node_pseudo_element_handler(SCM pe, SCM doc, SCM node) {

  /* Here's how we're going to do pseudo-elements: The inline renderer will set
     a flag on the currently rendering ruin_element_t to indicate that a
     particular pseudo-element match should evaluate to true. When this handler
     is called, we find the first immediate inline child (if we're a block 
     element) or the first nested inline child (if we're an inline element),
     and look at the pseudo-element flag set for that element_t. */

  char *pelt_name = scm_to_locale_string(pe);
  int i = 0, len = 0;

  ruin_element_t *t = ruin_window_lookup_scm(node);
  
  if (t == NULL)
    return SCM_BOOL_F;

  len = ruin_util_list_length(t->current_pseudo_elements);
  for (i = 0; i < len; i++) {
    char *e = ruin_util_list_peek_after(t->current_pseudo_elements, i);
    if (strcmp(e, pelt_name) == 0)
      return SCM_BOOL_T;
  }
  
  return SCM_BOOL_F;   
}

static int is_link(ruin_element_t *t) {
  if (t->dialect == RUIN_XML_DIALECT_XHTML) {
    if (strcmp(ruin_scheme_sdom_get_dom_property(t->element, "sdom:node-name"),
	       "a") == 0 &&
	ruin_scheme_sdom_get_attribute(t->element, "href") != NULL)
      return TRUE;
  }
  return FALSE;
}

static int has_visited_target(ruin_element_t *t) {
  if (t->dialect == RUIN_XML_DIALECT_XHTML) {
    char *href = ruin_scheme_sdom_get_attribute(t->element, "href");
    if (strcmp(href, "") == 0)
      return TRUE;
  }
  return FALSE;
}

SCM ruin_scm_api_node_pseudo_class_handler(SCM str, SCM doc, SCM node) {
  char *class = SCM_STRING_CHARS(str);
  ruin_element_t *t = ruin_window_lookup_scm(node);

  int link = is_link(t);
  int visited = link ? has_visited_target(t) : FALSE;

  if (strcmp(class, "visited") == 0 && visited) {
    return SCM_BOOL_T;
  } else if (strcmp(class, "link") == 0 && link && !visited) {
    return SCM_BOOL_T;
  } else if (strcmp(class, "focus") == 0) {
    if (t == t->parent_window->focused)
      return SCM_BOOL_T;
  }
  return SCM_BOOL_F;
}
