/* util.c: Utility functions for libRUIN
 * Copyright (C) 2006 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <ctype.h>
#include <libguile.h>
#include <math.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/time.h>
#include <time.h>

#include "util.h"
#include "window.h"

pthread_mutex_t _ruin_util_id_lock;

int ruin_util_hash_hash (char *k, int table_size) {
  unsigned int a, b, c, len, length;

  len = strlen (k);
  length = len;
  a = b = 0x9e3779b9;
  c = 0;
  
  while (len >= 12) {
    a += (k[0] + ((unsigned int) k[1] << 8) + ((unsigned int) k[2] << 16) +
	  ((unsigned int) k[3] << 24));
    b += (k[4] + ((unsigned int) k[5] << 8) + ((unsigned int) k[6] << 16) +
	  ((unsigned int) k[7] << 24));
    c += (k[8] + ((unsigned int) k[9] << 8) + ((unsigned int) k[10] << 16) +
	  ((unsigned int) k[11] << 24));
    RUIN_UTIL_HASH_MIX (a, b, c);
    k += 12; 
    len -= 12;
  }
  
  c += length;
  switch (len) {              /* all the case statements fall through */
  case 11: c += ((unsigned int) k[10] << 24);
  case 10: c += ((unsigned int) k[9] << 16);
  case 9: c += ((unsigned int) k[8] << 8);
  case 8: b += ((unsigned int) k[7] << 24);
  case 7: b += ((unsigned int) k[6] << 16);
  case 6: b += ((unsigned int) k[5] << 8);
  case 5: b += k[4];
  case 4: a += ((unsigned int) k[3] << 24);
  case 3: a += ((unsigned int) k[2] << 16);
  case 2: a += ((unsigned int) k[1] << 8);
  case 1: a += k[0];
  default:
    break;
  }

  RUIN_UTIL_HASH_MIX (a, b, c);

  return (c & RUIN_UTIL_HASH_MASK ((int) (log ((double) table_size))));
}

void ruin_util_hash_insertion_helper(ruin_util_hash *hash, int index, 
				     char *key, char *value) {
  hash->keys = realloc(hash->keys, sizeof(char *) * ++hash->num_keys);
  hash->values = realloc(hash->values, sizeof(char *) * hash->num_keys);
  hash->keys[hash->num_keys - 1] = strdup(key);
  hash->values[hash->num_keys - 1] = strdup(value);
  hash->map[index] = hash->num_keys - 1;
  return;
}

void ruin_util_hash_insert(ruin_util_hash *hash, char *key, char *value) {
  int hashed_index, mapped_index, i, new_table_size = 0;
  int *new_map;
  if (((hash == NULL) || (key == NULL)) || (value == NULL))
    return;
  pthread_mutex_lock(hash->lock);
  hashed_index = ruin_util_hash_hash(key, hash->table_size);
  if (hash->map[hashed_index] == -1) {
    ruin_util_hash_insertion_helper(hash, hashed_index, key, value);
    pthread_mutex_unlock(hash->lock);
    return;
  }
  else {
    mapped_index = hash->map[hashed_index];
    if (strcmp(hash->keys[mapped_index], key) == 0) {
      free(hash->values[mapped_index]);
      hash->values[mapped_index] = strdup(value);
      pthread_mutex_unlock(hash->lock);
      return;
    }
    for (i = hashed_index + 1; i < hash->table_size; i++) {
      if (hash->map[i] == -1) {
	ruin_util_hash_insertion_helper(hash, i, key, value);
	pthread_mutex_unlock(hash->lock);
	return;
      }
      else if(strcmp(hash->keys[hash->map[i]], key) == 0) {
	free(hash->values[hash->map[i]]);
	hash->values[hash->map[i]] = strdup(value);
	pthread_mutex_unlock(hash->lock);
	return;
      }
    }
    for (i = 0; i < hashed_index; i++) {
      if (hash->map[i] == -1) {
	ruin_util_hash_insertion_helper(hash, i, key, value);
	pthread_mutex_unlock(hash->lock);
	return;
      }
      else if(strcmp(hash->keys[hash->map[i]], key) == 0) {
	free(hash->values[hash->map[i]]);
	hash->values[hash->map[i]] = strdup(value);
	pthread_mutex_unlock(hash->lock);
	return;
      }
    }

    /* Uh-oh... time to grow the hash... */
    new_table_size = hash->table_size * RUIN_UTIL_HASH_GROWTH_FACTOR;
    new_map = malloc(sizeof(int) * new_table_size);
    for (i = 0; i < new_table_size; i++)
      new_map[i] = -1;
    for (i = 0; i < hash->num_keys; i++) {
      int new_hashed_index = 
	ruin_util_hash_hash(hash->keys[i], new_table_size);
      if (new_map[new_hashed_index] == -1)
	new_map[new_hashed_index] = i;
      else {
	int j, found_null = FALSE;
	for (j = new_hashed_index + 1; j < new_table_size; j++)
	  if (new_map[j] == -1) {
	    new_map[j] = i;
	    found_null = TRUE;
	    break;
	  }
	if (!found_null) {
	  for (j = 0; j < new_hashed_index; j++)
	    if (new_map[j] == -1) {
	      new_map[j] = i;
	      break;
	    }
	}
      }
    }
    free(hash->map);
    hash->map = new_map;
    hash->table_size = new_table_size;
    pthread_mutex_unlock(hash->lock);
    ruin_util_hash_insert(hash, key, value);
    return;
  }  
}

char *ruin_util_hash_retrieve(ruin_util_hash *hash, char *key) {
  int hashed_index, mapped_index, i, found_null;
  if ((hash == NULL) || (key == NULL))
    return NULL;
  hashed_index = ruin_util_hash_hash(key, hash->table_size);
  pthread_mutex_lock(hash->lock);
  if (hash->map[hashed_index] == -1) {
    pthread_mutex_unlock(hash->lock);
    return NULL;
  }
  mapped_index = hash->map[hashed_index];
  if (strcmp(hash->keys[mapped_index], key) == 0) {
    pthread_mutex_unlock(hash->lock);
    return hash->values[mapped_index];
  }
  found_null = FALSE;
  for (i = hashed_index + 1; i < (hash->table_size - 1); i++) {
    if (hash->map[i] == -1) {
      found_null = TRUE;
      break;
    }
    mapped_index = hash->map[i];
    if (strcmp(hash->keys[mapped_index], key) == 0) {
      pthread_mutex_unlock(hash->lock);
      return hash->values[mapped_index];
    }
  }
  if (found_null) {
    pthread_mutex_unlock(hash->lock);
    return NULL;
  }
  for (i = 0; i < hashed_index; i++) {
    if (hash->map[i] == -1) {
      pthread_mutex_unlock(hash->lock);
      return NULL;
    }
    mapped_index = hash->map[i];
    if (strcmp(hash->keys[mapped_index], key) == 0) {
      pthread_mutex_unlock(hash->lock);
      return hash->values[mapped_index];
    }
  }
  pthread_mutex_unlock(hash->lock);
  return NULL;
}

void _ruin_util_hash_remove(ruin_util_hash *hash, char *key) {
  int i;
  for (i = 0; i < hash->num_keys; i++) {
    if (strcmp(hash->keys[i], key) == 0) {
      int j;
      free(hash->keys[i]);
      free(hash->values[i]);
      hash->keys[i] = NULL;
      hash->values[i] = NULL;
      if ((hash->num_keys == 1) || (i == hash->num_keys - 1)) {
	for (j = 0; j < hash->table_size; j++)
	  if (hash->map[j] == i) {
	    hash->map[j] = -1;
	    break;
	  }
	hash->num_keys--;
      }
      else {
	int old_map_index = -1;
	for (j = 0; j < hash->table_size; j++)
	  if (hash->map[j] == hash->num_keys - 1) {
	    old_map_index = j;
	    break;
	  }	
	hash->map[old_map_index] = -1;
	hash->keys[i] = hash->keys[hash->num_keys - 1];
	hash->keys[hash->num_keys - 1] = NULL;
	hash->values[i] = hash->values[hash->num_keys - 1];
	hash->values[hash->num_keys - 1] = NULL;
	hash->num_keys--;
      }
      break;
    }
  }
}

void ruin_util_hash_remove(ruin_util_hash *hash, char *key) {
  if ((hash == NULL) || (key == NULL))
    return;
  pthread_mutex_lock(hash->lock);
  _ruin_util_hash_remove(hash, key);
  pthread_mutex_unlock(hash->lock);
  return;
}

void ruin_util_hash_clear(ruin_util_hash *hash) {
  int i;
  if (hash == NULL)
    return;
  pthread_mutex_lock(hash->lock);
  for (i = 0; i < hash->num_keys; i++) {
    if (hash->keys[i] == NULL)
      continue;
    else {
      _ruin_util_hash_remove(hash, hash->keys[i]);
      i = 0;
    }
  }
  pthread_mutex_unlock(hash->lock);
}

char **ruin_util_hash_get_keys(ruin_util_hash *hash, int *key_count) {
  int i;
  char **key_list;
  if (hash == NULL)
    return NULL;
  pthread_mutex_lock(hash->lock);
  if (key_count != NULL)
    *key_count = hash->num_keys;
  key_list = malloc(sizeof(char *) * hash->num_keys);
  for (i = 0; i < hash->num_keys; i++)
    key_list[i] = strdup(hash->keys[i]);
  pthread_mutex_unlock(hash->lock);
  return key_list;
}

ruin_util_hash *ruin_util_hash_new() {
  int i;
  ruin_util_hash *hash = calloc(1, sizeof(ruin_util_hash));
  hash->lock = malloc(sizeof(pthread_mutex_t));
  pthread_mutex_init(hash->lock, NULL);
  hash->table_size = RUIN_UTIL_HASH_DEFAULT_SIZE;
  hash->num_keys = 0;
  hash->map = malloc(sizeof(int) * hash->table_size);
  for (i = 0; i < hash->table_size; i++)
    hash->map[i] = -1;
  hash->keys = NULL;
  hash->values = NULL;
  return hash;
}

void ruin_util_hash_free(ruin_util_hash *hash) {
  int i;
  pthread_mutex_lock(hash->lock);
  free(hash->map);
  for (i = 0; i < hash->num_keys; i++) {
    free(hash->keys[i]);
    free(hash->values[i]);
  }
  pthread_mutex_unlock(hash->lock);
  pthread_mutex_destroy(hash->lock);
  return;
}

/* Here are the list functions. */

ruin_util_list *ruin_util_list_new() {
  ruin_util_list *foo = calloc(1, sizeof(ruin_util_list));
  pthread_mutex_init(&(foo->lock), NULL);
  return foo;
}

void ruin_util_list_free(ruin_util_list *list) {
  struct _ruin_util_list_item *the_item;
  if (list == NULL)
    return;
  pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock, 
		       (void *) &(list->lock));
  pthread_mutex_lock(&(list->lock));
  the_item = list->head;
  if (the_item != NULL)
    while(the_item->next != NULL) {
      struct _ruin_util_list_item *next_item = the_item->next;
      free(the_item->value);
      free(the_item);
      the_item = next_item;
    }
  pthread_mutex_unlock(&(list->lock));

  pthread_cleanup_pop(0);
  free(list);
  return;
}

int ruin_util_list_length(ruin_util_list *list) {
  if (list != NULL)
    return list->num_items;
  return -1;
}

void ruin_util_list_push_back(ruin_util_list *list, char *value) {
  struct _ruin_util_list_item *new_item;
  if ((list == NULL) || (value == NULL))
    return;
  pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock, 
		       (void *) &(list->lock));
  pthread_mutex_lock(&(list->lock));
  new_item = calloc(1, sizeof(struct _ruin_util_list_item));
  new_item->value = strdup(value);
  if (list->head == NULL) {
    list->head = new_item;
    list->tail = new_item;
  }
  else {
    new_item->prev = list->tail;
    list->tail->next = new_item;
    list->tail = new_item;
  }
  list->num_items++;
  pthread_mutex_unlock(&(list->lock));
  pthread_cleanup_pop(0);
  return;
}

void ruin_util_list_push_front(ruin_util_list *list, char *value) {
  struct _ruin_util_list_item *new_item;
  if ((list == NULL) || (value == NULL))
    return;
  pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock, 
		       (void *) &(list->lock));
  pthread_mutex_lock(&(list->lock));
  new_item = calloc(1, sizeof(struct _ruin_util_list_item));
  new_item->value = strdup(value);
  if (list->head == NULL) {
    list->head = new_item;
    list->tail = new_item;
  }
  else {
    new_item->next = list->head;
    list->head->prev = new_item;
    list->head = new_item;
  }
  list->num_items++;
  pthread_mutex_unlock(&(list->lock));
  pthread_cleanup_pop(0);
  return;
}

char *ruin_util_list_peek_after(ruin_util_list *list, int index) {
  char *return_val;
  if ((list == NULL) || (index < 0))
    return NULL;
  pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock, 
		       (void *) &(list->lock));
  pthread_mutex_lock(&(list->lock));
  if (list->head == NULL)
    return_val = NULL;
  else {
    int i = 0;
    struct _ruin_util_list_item *item_ptr = list->head;
    while ((i < index) && (item_ptr->next != NULL)) {
      item_ptr = item_ptr->next;
      i++;
    }
    return_val = item_ptr->value;
  }
  pthread_mutex_unlock(&(list->lock));
  pthread_cleanup_pop(0);
  return return_val;
}

void ruin_util_list_remove_after(ruin_util_list *list, int index) {
  if ((list == NULL) || (index < 0))
    return;
  pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock, 
		       (void *) &(list->lock));
  pthread_mutex_lock(&(list->lock));
  if (list->head != NULL) {
    int i = 0;
    struct _ruin_util_list_item *item_ptr = list->head;
    while ((i < index) && (item_ptr->next != NULL)) {
      item_ptr = item_ptr->next;
      i++;
    }
    if(item_ptr->next != NULL)
      item_ptr->next->prev = item_ptr->prev;
    if(item_ptr->prev != NULL)
      item_ptr->prev->next = item_ptr->next;
    if(list->head == item_ptr)
      list->head = item_ptr->prev;
    if(list->tail == item_ptr)
      list->tail = item_ptr->next;
    free(item_ptr);
  }
  list->num_items--;
  pthread_mutex_unlock(&(list->lock));
  pthread_cleanup_pop(0);
  return;
}

void ruin_util_list_insert_after(ruin_util_list *list, int index, 
				 char *value) {
  struct _ruin_util_list_item *new_item;
  if ((list == NULL) || (value == NULL))
    return;
  pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock, 
		       (void *) &(list->lock));
  pthread_mutex_lock(&(list->lock));
  new_item = calloc(1, sizeof(struct _ruin_util_list_item));
  new_item->value = strdup(value);
  if (list->head == NULL) {
    list->head = new_item;
    list->tail = new_item;
  }
  else {
    int i = 0;
    struct _ruin_util_list_item *item_ptr = list->head;
    while ((i < index) && (item_ptr->next != NULL)) {
      item_ptr = item_ptr->next;
      i++;
    }
    if (item_ptr->next != NULL) {
      new_item->next = item_ptr->next;
      item_ptr->next->prev = new_item;
    }
    else list->tail = new_item;
    item_ptr->next = new_item;
    new_item->prev = item_ptr;
  }
  list->num_items++;
  pthread_mutex_unlock(&(list->lock));
  pthread_cleanup_pop(0);
  return;  
}

char *_ruin_util_list_pop_front_helper(ruin_util_list *list) {
  struct _ruin_util_list_item *the_item;
  char *value;
  if (list->head == NULL)
    return NULL;
  the_item = list->head;
  value = the_item->value;
  if (list->head == list->tail) {
    list->head = NULL;
    list->tail = NULL;
  }
  else {
    list->head = list->head->next;
    list->head->prev = NULL;
  }
  the_item->next = NULL;
  list->num_items--;
  free(the_item);
  return value;
}

char *ruin_util_list_pop_front(ruin_util_list *list) {
  char *value;
  if (list == NULL)
    return NULL;  

  pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock, 
		      (void *) &(list->lock));
  pthread_mutex_lock(&(list->lock));

  value = _ruin_util_list_pop_front_helper(list);

  pthread_mutex_unlock(&(list->lock));
  pthread_cleanup_pop(0);
  return value;
}

/* And some string conversion functions. */

char *ruin_util_int_to_string (int fd) {
  char *out = NULL;
  int num_digits = 0;
  int i;
  if (fd < 0)
    return NULL;
  if (fd == 0)
    num_digits = 1;
  else for (i = 1;; i *= 10) {
    if (fd >= i)
      num_digits += 1;
    else break;
  }
  out = calloc(1, sizeof(char) * (num_digits + 1));
  (void) snprintf(out, num_digits + 1, "%d", fd);
  out = realloc(out, sizeof(char) * (strlen(out) + 1));
  return out;
}

char *ruin_util_ptr_to_string(void *ptr) {
  int num_digits = (sizeof(void *) * 2) + 2;
  char *out = calloc(1, sizeof(char) * (num_digits + 1));
  (void) snprintf(out, num_digits + 1, "%p", ptr);
  return out;
}

char *ruin_util_long_to_string(long value) {
  char *out = NULL;
  int num_digits = 0;
  int i;
  if (value < 0)
    return NULL;
  if (value == 0)
    num_digits = 1;
  else for (i = 1;; i *= 10) {
    if (value >= i)
      num_digits += 1;
    else break;
  }
  out = malloc(sizeof(char) * (num_digits + 1));
  (void) snprintf(out, num_digits + 1, "%ld", value);
  return out;
}

void *ruin_util_string_to_ptr(char *ptr) {
  void *result;
  if((ptr != NULL) && (sscanf(ptr, "%p", &result) == 1))
    return result;
  return NULL;
}

long ruin_util_generate_id() {
  long return_val;
  static long next_id = 0;
  pthread_mutex_lock(&_ruin_util_id_lock);
  return_val = next_id;
  next_id++;
  pthread_mutex_unlock(&_ruin_util_id_lock);
  return return_val;
}

void ruin_util_log(ruin_window_t *w, char *msg, ...) {
  va_list args;
  va_start(args, msg);
  if (w->log != NULL) {
    fprintf(w->log, "window %ld: ", w->internal_id);
    vfprintf(w->log, msg, args);
    fflush(w->log); 
  }
  va_end(args);
}

/* Somewhat inefficient; refactor to minimize use of realloc()? */

char *ruin_util_arabic_to_roman(int arabic, int use_upper) {
  int i = 0, len = 0;
  char *ret = NULL;
  static int digs[] = { 1000, 900, 500, 400, 100, 90, 50, 40, 10, 9, 5, 4, 1 };
  static char *strsu[] = 
    { "M", "CM", "D", "CD", "C", "XC", "L", "XL", "X", "IX", "V", "IV", "I" };
  static char *strsl[] =
    { "m", "cm", "d", "cd", "c", "xc", "l", "xl", "x", "ix", "v", "iv", "i" };
  if (arabic > 1000)
    return NULL;
  for (i = 0; i < 13; i++) {
    while (arabic >= digs[i]) {
      int slen = strlen(strsu[i]);
      arabic -= digs[i];
      ret = realloc(ret, len + slen + 1);
      strncpy(ret + len, use_upper ? strsu[i] : strsl[i], slen);
      len += slen;
    }
  }
  ret = realloc(ret, len + 1);
  ret[len] = 0;
  return ret;
}

char *ruin_util_get_parent_directory(char *filename) {
  char *rstr = NULL;
  char *abs_path = malloc(sizeof(char) * PATH_MAX);
  char *ret = NULL;
  realpath(filename, abs_path);
  rstr = strrchr(abs_path, '/');
  ret = calloc(rstr - abs_path + 1, sizeof(char));
  strncat(ret, abs_path, rstr - abs_path);
  free(abs_path);
  return ret;
}

long ruin_util_current_time_millis() {
  struct timeval tv = { 0, 0 };
  (void) gettimeofday(&tv, NULL);
  return (tv.tv_sec * 1000) + (tv.tv_usec / 1000);
}

SCM ruin_util_string2scm(char *str) {
  return scm_mem2string(str, strlen(str));
}
