/* window.c: Window-management routines for libRUIN
 * Copyright (C) 2006 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifdef LINUX
#include <pty.h>
#endif /* LINUX */
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>

#include "layout.h"
#include "load.h"
#include "scheme.h"
#include "util.h"
#include "window.h"

extern ruin_windows_t *_ruin_windows;

extern long ruin_load_layout_and_render(ruin_window_t *);

static pthread_mutex_t ruin_window_signal_handler_SIGWINCH_entry_lock;

void ruin_window_signal_handler_SIGWINCH(int n, siginfo_t *s, void *u) {
  static int busy = 0;
  int num_keys = 0, i = 0;
  int oldx = 0, oldy = 0, newx = 0, newy = 0;
  int iocx = 0, iocy = 0;
  char **keys = NULL;
  struct winsize ws;

  pthread_mutex_lock(&ruin_window_signal_handler_SIGWINCH_entry_lock);
  if (busy) {
    pthread_mutex_unlock(&ruin_window_signal_handler_SIGWINCH_entry_lock);
    return;
  }
  pthread_mutex_unlock(&ruin_window_signal_handler_SIGWINCH_entry_lock);
  busy = TRUE;

  keys = ruin_util_hash_get_keys(_ruin_windows->windows, &num_keys);

  (void) ioctl(0, TIOCGWINSZ, &ws);
  iocx = ws.ws_col;
  iocy = ws.ws_row;

  for (i = 0; i < num_keys; i++) {
    ruin_window_t *w = (ruin_window_t *) ruin_util_string_to_ptr 
      (ruin_util_hash_retrieve(_ruin_windows->windows, keys[i]));

    oldx = w->top->width.used;
    oldy = w->top->height.used;
    
    newy = iocy;
    newx = iocx;

    if (oldx != newx || oldy != newy) {
      w->top->max_width.computed = w->top->width.computed = newx;
      w->top->max_height.computed = w->top->height.computed = newy;

      resizeterm(newy, newx); 
      ruin_load_layout_and_render(w);
    }
  }
  busy = FALSE;
}

ruin_windows_t *ruin_windows_new() {
  char *xul = NULL;
  char *xhtml = NULL;

  char *env_val = getenv("RUIN_CSS_PATH");

  SCM port = SCM_EOL;

  ruin_windows_t *t = calloc(1, sizeof(ruin_windows_t));
  t->windows = ruin_util_hash_new();

  if (env_val == NULL)
    env_val = RUIN_CSS_PATH;
  xul = calloc(strlen(env_val) + 9, sizeof(char));
  xhtml = calloc(strlen(env_val) + 11, sizeof(char));
  strcat(xul, env_val);
  strcat(xul, "/xul.css");
  strcat(xhtml, env_val);
  strcat(xhtml, "/xhtml.css");

  port = scm_open_file(scm_makfrom0str(xul), scm_makfrom0str("r"));
  t->xul_agent_css = scm_call_1(scm_c_eval_string("scss:css->scss"), port);
  scm_gc_protect_object(t->xul_agent_css);

  port = scm_open_file(scm_makfrom0str(xhtml), scm_makfrom0str("r"));
  t->xhtml_agent_css = scm_call_1(scm_c_eval_string("scss:css->scss"), port);
  scm_gc_protect_object(t->xhtml_agent_css);

  free(xul);
  free(xhtml);

  pthread_mutex_init(&ruin_window_signal_handler_SIGWINCH_entry_lock, NULL);
  return t;
}

void ruin_windows_free(ruin_windows_t *t) {
  ruin_util_hash_free(t->windows);
  scm_gc_unprotect_object(t->xul_agent_css);
  scm_gc_unprotect_object(t->xhtml_agent_css);
  free(t);
  return;
}

ruin_window_t *ruin_window_new(WINDOW *w, FILE *f) {
  ruin_window_t *t = calloc(1, sizeof(ruin_window_t));
  t->window = w;
  t->log = f;

  t->internal_id = ruin_util_generate_id();
  ruin_util_hash_insert(_ruin_windows->windows, 
			ruin_util_long_to_string(t->internal_id), 
			ruin_util_ptr_to_string((void *) t));
  
  t->ids = ruin_util_hash_new();
  t->internal_ids = ruin_util_hash_new();

  t->scm_hash = scm_call_1(scm_c_eval_string("make-hash-table"),
			   scm_int2num(RUIN_WINDOW_SCM_HASH_SIZE));
  scm_gc_protect_object(t->scm_hash);

  t->tab_order = ruin_util_list_new();

  t->font_height = RUIN_WINDOW_DEFAULT_FONT_HEIGHT;
  t->font_width = RUIN_WINDOW_DEFAULT_FONT_WIDTH;
  t->dpi = RUIN_WINDOW_DEFAULT_DPI;

  return t;
}

void ruin_window_free(ruin_window_t *t) {
  ruin_util_hash_remove(_ruin_windows->windows, 
			ruin_util_long_to_string(t->internal_id));
  ruin_util_hash_free(t->ids);
  ruin_util_hash_free(t->internal_ids);
  scm_gc_unprotect_object(t->scm_hash);
  ruin_util_list_free(t->tab_order);
  free(t);
  return;
}

ruin_element_t *ruin_window_lookup_scm(SCM elt) {
  SCM doc = SCM_EOL;
  SCM type = scm_call_1(scm_c_eval_string("sdom:node-type"), elt);

  ruin_window_t *containing_win = NULL;

  char **keys = NULL;
  int num_keys, i;
  
  if (scm_eqv_p(type, scm_c_eval_string("sdom:node-type-document")) == 
      SCM_BOOL_T)
    doc = elt;
  else doc = scm_call_2(scm_c_eval_string("sdom:get-dom-property"), elt,
			scm_makfrom0str("sdom:owner-document"));

  keys = ruin_util_hash_get_keys(_ruin_windows->windows, &num_keys);
  for (i = 0; i < num_keys; i++) {
    ruin_window_t *win = (ruin_window_t *)
      (ruin_util_string_to_ptr
       (ruin_util_hash_retrieve(_ruin_windows->windows, keys[i])));
    if (scm_eq_p(win->top->doc, doc) == SCM_BOOL_T) {
      containing_win = win;
      break;
    }
  }

  if (containing_win != NULL)
    return (ruin_element_t *) ruin_util_string_to_ptr
      (scm_to_locale_string
       (scm_hashq_ref(containing_win->scm_hash, elt, SCM_EOL)));
  return NULL;  
}

