/* xhtml.c: XHTML-parsing support for libRUIN
 * Copyright (C) 2007 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <ctype.h>
#include <libguile.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "css.h"
#include "dialect.h"
#include "layout.h"
#include "scheme.h"
#include "util.h"
#include "window.h"
#include "xhtml.h"

void ruin_xhtml_generate_tree_parse_attrs(ruin_element_t *t) {
  SCM attr_str = scm_call_2(scm_c_eval_string("sdom:get-attribute"),
			    t->element, scm_makfrom0str("style"));
  if (attr_str != SCM_BOOL_F) {
    SCM parsed_style = ruin_dialect_parse_in_document_style
      (t->doc, scm_string_append(scm_list_3(scm_makfrom0str("* {"),
					    attr_str,
					    scm_makfrom0str("}"))));
    if (scm_eq_p(t->additional_attribute_style, SCM_EOL) == SCM_BOOL_T) {
      t->additional_attribute_style = parsed_style;
      scm_gc_protect_object(t->additional_attribute_style);
    } else scm_append_x(scm_list_2(t->additional_attribute_style, 
				   SCM_CADAR(parsed_style)));
  } 
}

ruin_element_t *ruin_xhtml_generate_tree(ruin_window_t *w, SCM node, 
					 ruin_element_t *parent,
					 ruin_element_t *sibling) {
  int not_sibling = FALSE, ignore_children = FALSE;
  ruin_element_t *children_result = NULL;
  ruin_element_t *my_result = NULL;
  SCM node_next = SCM_EOL; 
  SCM node_type = SCM_EOL;

  if (node == SCM_BOOL_F)
    return NULL;

  node_next = scm_call_2(scm_c_eval_string("sdom:get-dom-property"), node,
			 scm_makfrom0str("sdom:next-sibling"));
  node_type = scm_call_1(scm_c_eval_string("sdom:node-type"), node);
  
  if (scm_eqv_p(scm_c_eval_string("sdom:node-type-document"), node_type) == 
      SCM_BOOL_T)
    return ruin_xhtml_generate_tree
      (w, scm_call_2(scm_c_eval_string("sdom:get-dom-property"), 
		     node, scm_makfrom0str("sdom:document-element")), 
       parent, NULL);

  /* If this is a processing-instruction node, we need to try to retrieve a
     potential CSS stylesheet and apply it to our document. */
  
  if (scm_eqv_p(scm_c_eval_string("sdom:node-type-processing-instruction"),
		node_type) == SCM_BOOL_T) { 
  }

  /* Otherwise, we should check for any of the user-specified parts of the CSS
     cascade; this means looking for stuff attached to this node that's in the
     "css" namespace.   If there is no namespace, we should treat this as an
     XHTML attribute and add it accordingly. */

  else if (scm_eqv_p(scm_c_eval_string("sdom:node-type-text"), node_type) == 
	   SCM_BOOL_T)
    my_result = ruin_dialect_generate_text_node(node, parent, sibling);
  else if (scm_eqv_p(scm_c_eval_string("sdom:node-type-element"), 
		     node_type) == SCM_BOOL_T) {
    char *node_name = ruin_dialect_get_node_name(node);
    SCM aas_list = parent->additional_attribute_style;
    if (scm_eq_p(aas_list, SCM_EOL) != SCM_BOOL_T) {
      aas_list = SCM_CDAR(aas_list);
    }

    my_result = ruin_element_new();
    
    my_result->dialect = parent->dialect;
    my_result->cascade = parent->cascade;
    my_result->doc = parent->doc;
    my_result->parent = parent;
    my_result->parent_window = my_result->parent->parent_window;
    
    /* Attempt to grab any inheritable additional style attributes...
       Is this a reasonable approach?  We'll need to recompute this stuff if
       the inheritance tree changes. */

    while(scm_eq_p(aas_list, SCM_EOL) != SCM_BOOL_T) {
      if (ruin_scheme_scss_is_inherited(SCM_CAAR(aas_list))) {
	ruin_layout_add_style
	  (&my_result->additional_attribute_style,
	   SCM_STRING_CHARS(SCM_CAAR(aas_list)), 
	   SCM_STRING_CHARS(SCM_CADAR(aas_list)));
      }
      aas_list = SCM_CDR(aas_list);
    }

    if (my_result->parent->parent == NULL) {
      my_result->parent_window->top = my_result;
    }
    
    my_result->element = node;
    scm_hashq_set_x
      (my_result->parent_window->scm_hash, node, 
       scm_makfrom0str(ruin_util_ptr_to_string((void*) my_result)));
    
    my_result->ids = (parent != NULL) ? parent->ids : NULL;
    my_result->prev_sibling = sibling;

    if (strcmp((char *) node_name, "a") == 0) {
      my_result->focusable = TRUE;
    } else if (strcmp((char *) node_name, "input") == 0) {
      if (strcmp((char *) node_name, "button") == 0) 
	my_result->focusable = TRUE;
      else if (strcmp((char *) node_name, "checkbox") == 0) {
	my_result->focusable = TRUE;
	my_result->extra_content = RUIN_LAYOUT_EXTRA_CONTENT_CHECKBOX;
      }
      else if (strcmp((char *) node_name, "radio") == 0) {
	my_result->focusable = TRUE;
	my_result->extra_content = RUIN_LAYOUT_EXTRA_CONTENT_RADIO;
      }
    }
    else if (strcmp((char *) node_name, "style") == 0) {
      SCM child = scm_call_2
	(scm_c_eval_string("sdom:get-dom-property"), node, 
	 scm_makfrom0str("sdom:first-child"));
      ignore_children = TRUE;
      if (scm_eq_p(child, SCM_EOL) != SCM_BOOL_T) {
	SCM style_text = scm_call_2
	  (scm_c_eval_string("sdom:get-dom-property"), child,
	   scm_makfrom0str("sdom:node-value"));
	SCM sheet = ruin_dialect_parse_in_document_style
	  (my_result->doc, style_text);
	scm_call_2(scm_c_eval_string("scss:set-author-stylesheet!"),
		   my_result->cascade, sheet);
      }
    }

    /* Insert in a preliminary position in the tab ordering and add the 
       default handler for focus-switches... */
    
    if (my_result->focusable) {
      ruin_dialect_update_tab_position(my_result, 0);
    }

    if (strcmp(ruin_css_lookup(my_result, "display", NULL), "inline") == 0 &&
	((sibling != NULL && strcmp(ruin_css_lookup(sibling, "display", NULL), 
				   "inline") == 0) ||
	 (parent != NULL && parent->prev_was_inline)))
      my_result->prev_was_inline = TRUE;

    my_result->attributes = 
      scm_call_2(scm_c_eval_string("sdom:get-dom-property"),
		 node, scm_makfrom0str("sdom:attributes"));
    scm_gc_protect_object(my_result->attributes);
    ruin_xhtml_generate_tree_parse_attrs(my_result);
    
    /* Looking at one of them should take care of all non-attribute 
       children. */
    
    if (!ignore_children) { 
      SCM children = scm_call_2(scm_c_eval_string("sdom:get-dom-property"),
				node, scm_makfrom0str("sdom:child-nodes"));
      if (children != SCM_EOL) {
        children_result = ruin_xhtml_generate_tree
	  (w, SCM_CAR(children), my_result, NULL); 
	my_result->first_child = children_result;
      }
    }
  }    

  ruin_dialect_add_table_parents(my_result);
  ruin_dialect_add_table_children(my_result);
  ruin_dialect_add_table_columns(my_result);

  if ((my_result != NULL) && (!not_sibling)) {
    ruin_element_t *return_ptr = my_result;
    ruin_element_t *parent_ptr = my_result->parent;
    ruin_util_hash_insert
      (w->internal_ids, ruin_util_long_to_string(my_result->internal_id),
       my_result);

    my_result->next_sibling = 
	ruin_xhtml_generate_tree(w, node_next, parent_ptr, my_result);

    return return_ptr;
  } else return ruin_xhtml_generate_tree(w, node_next, parent, sibling);
}
