/* node-fns.c: Node-management functions for the libRUIN Scheme API
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <libguile.h>
#include <string.h>

#include "../css.h"
#include "../parse.h"
#include "../scheme.h"
#include "../window.h"

SCM_DEFINE (ruin_scm_api_node_pseudo_element_handler, 
	    "ruin:scss-pseudo-element-handler", 2, 0, 0, (SCM node, SCM pe), 
	    "Get the SCSS pseudo element handler.")
{
  ruin_window_t *w = ruin_window_get_current_window ();
  char *pelt_name = scm_to_locale_string (pe);

  enum _ruin_css_pseudo_element e = 0;
  if (strcmp (pelt_name, "after") == 0) e = RUIN_CSS_PSEUDO_ELEMENT_AFTER;
  else if (strcmp (pelt_name, "before") == 0) 
    e = RUIN_CSS_PSEUDO_ELEMENT_BEFORE;
  else if (strcmp (pelt_name, "first-letter") == 0)
    e = RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER;
  else if (strcmp (pelt_name, "first-line") == 0)
    e = RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE;

  if (e & w->render_state->active_pseudo_elements)
    return SCM_BOOL_T;
  return SCM_BOOL_F;
}

static int is_link (ruin_window_t *w, SCM node) 
{
  if (w->render_state->dialect == RUIN_XML_DIALECT_XHTML) 
    {
      if (strcmp (ruin_scheme_sdom_node_name (w, node), "a") == 0 &&
	  ruin_scheme_sdom_get_attribute (w, node, "href") != NULL)
	return TRUE;
    }
  return FALSE;
}

static int has_visited_target (ruin_window_t *w, SCM node) 
{
  if (w->render_state->dialect == RUIN_XML_DIALECT_XHTML) 
    if (strcmp (ruin_scheme_sdom_node_name (w, node), "a") == 0)
      {
	char *href = ruin_scheme_sdom_get_attribute (w, node, "href");
	if (href != NULL)
	  {
	    if (strlen (href) == 0)
	      return TRUE;
	  }
      }
  return FALSE;
}

SCM_DEFINE (ruin_scm_api_node_pseudo_class_handler, 
	    "ruin:scss-pseudo-class-handler", 2, 0, 0, (SCM node, SCM str), 
	    "Get the SCSS pseudo class handler.")
{
  char *class = scm_to_locale_string (str);
    
  ruin_window_t *w = ruin_window_get_current_window ();

  int link = is_link (w, node);
  int visited = link ? has_visited_target (w, node) : FALSE;

  if (strcmp (class, "visited") == 0 && visited)
    return SCM_BOOL_T;
  else if (strcmp (class, "link") == 0 && link && !visited)
    return SCM_BOOL_T;
  else if (strcmp (class, "focus") == 0) { }
  else if (strcmp (class, "first-child") == 0)
    {
      SCM parent = ruin_scheme_sdom_parent_node (w, node);
      SCM first_child = ruin_scheme_sdom_first_child (w, parent);
      if (scm_eq_p (node, first_child) == SCM_BOOL_T) 
	return SCM_BOOL_T;
    }
  return SCM_BOOL_F;
}

void _ruin_scm_node_init_api (void)
{
  #include "node-fns.x"
}
