/* debug.c: Debug functions for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <libguile.h>
#include <string.h>

#include "css.h"
#include "debug.h"
#include "layout.h"
#include "parse.h"
#include "scheme.h"
#include "util.h"

static void print_box_tree (ruin_box_t *box, int indent)
{
  GList *children = box->children;
  char *indents = calloc(indent + 1, sizeof(char));
  int i = 0;
  for (; i < indent; i++)
    strncat(indents, " ", 1);

  switch (box->type)
    {
    case RUIN_LAYOUT_BOX_TYPE_INLINE: 
      ruin_util_log (box->window, "%s[inline %d:%d]", indents, box->width.used,
		     box->height.used); 
      break;
    case RUIN_LAYOUT_BOX_TYPE_BLOCK: 
      ruin_util_log (box->window, "%s[block %d:%d]", indents, box->width.used,
		     box->height.used); 
      break;
    case RUIN_LAYOUT_BOX_TYPE_LINE: 
      ruin_util_log (box->window, "%s[line %d:%d]", indents, box->width.used, 
		     box->height.used); 
      break;
    case RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE: 
      ruin_util_log (box->window, "%s[anonymous inline]", indents); break;
    case RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK: 
      ruin_util_log (box->window, "%s[anonymous block]", indents); break;
    case RUIN_LAYOUT_BOX_TYPE_MARKER:
      ruin_util_log (box->window, "%s[marker %d:%d]", indents, box->width.used,
		     box->height.used); 
      break;
    case RUIN_LAYOUT_BOX_TYPE_NONE:
      ruin_util_log (box->window, "%s[none]", indents); break;
    default:
      ruin_util_log (box->window, "%s[%d]", indents, box->type);
    }

  free(indents);
  while (children != NULL) 
    {
      print_box_tree (children->data, indent + 2);
      children = children->next;
    }
}

void ruin_debug_print_box_tree (ruin_box_t *box)
{
  print_box_tree (box, 0);
}
