/* layout.c: Layout routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <ctype.h>
#include <glib.h>
#include <libguile.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "box.h"
#include "css.h"
#include "layout.h"
#include "parse.h"
#include "render.h"
#include "scheme.h"
#include "tables.h"
#include "util.h"

static GList *layout_inline 
(ruin_box_t *, ruin_inline_box_t *, int, int, ruin_layout_line_state_t *);
static void layout_box (ruin_box_t *, int, int);
static void layout_block (ruin_box_t *, int, int);

static void parse_block_sizes (ruin_box_t *);
static void parse_inline_sizes (ruin_box_t *);

static void normalize_lengths (ruin_box_t *cb, ruin_box_t *b) 
{  
  if (cb == NULL)
    cb = b;
 
  /* First need to determine maximums and minimums. */

  if (b->max_width.computed == RUIN_CSS_VALUE_NONE)
    b->max_width.used = SHRT_MAX;
  else ruin_box_normalize_width (&b->max_width, b, TRUE);
  if (b->max_height.computed == RUIN_CSS_VALUE_NONE)
    b->max_height.used = SHRT_MAX;
  else ruin_box_normalize_height (&b->max_height, b, TRUE);
  ruin_box_normalize_width (&b->min_width, cb, TRUE);
  ruin_box_normalize_height (&b->min_height, cb, TRUE);

  /* Next, we normalize the computed values against the maximums we just
     obtained. */

  ruin_box_normalize_width (&b->width, cb, TRUE);
  ruin_box_normalize_width (&b->margin_left, cb, TRUE);
  ruin_box_normalize_width (&b->margin_right, cb, TRUE);
  ruin_box_normalize_width (&b->padding_left, cb, TRUE);
  ruin_box_normalize_width (&b->padding_right, cb, TRUE);
  ruin_box_normalize_width (&b->border_left_width, cb, FALSE);
  ruin_box_normalize_width (&b->border_right_width, cb, FALSE);

  ruin_box_normalize_height (&b->height, cb, TRUE);
  ruin_box_normalize_height (&b->margin_top, cb, TRUE);
  ruin_box_normalize_height (&b->margin_bottom, cb, TRUE);
  ruin_box_normalize_height (&b->padding_top, cb, TRUE);
  ruin_box_normalize_height (&b->padding_bottom, cb, TRUE);
  ruin_box_normalize_height (&b->border_top_width, cb, FALSE);
  ruin_box_normalize_height (&b->border_bottom_width, cb, FALSE);

  ruin_box_normalize_width (&b->text_indent, cb, TRUE);
  ruin_box_normalize_width (&b->letter_spacing, cb, TRUE);
  ruin_box_normalize_width (&b->word_spacing, cb, TRUE);
}

static int set_block_width (ruin_box_t *b, int cb_width) 
{
  int autos = 0, i = 0, limit = 7, width_auto = FALSE;
  int tentative_width = b->width.used;
  ruin_length_t *widths[7];
  
  /* The order here is important. */

  widths[0] = &b->margin_left;
  widths[1] = &b->margin_right;
  widths[2] = &b->padding_left;
  widths[3] = &b->padding_right;
  widths[4] = &b->border_left_width;
  widths[5] = &b->border_right_width;
  widths[6] = &b->width;

  for (i = 0; i < limit; i++) {
    if (widths[i]->computed == RUIN_CSS_VALUE_AUTO) {
      if (widths[i] == &b->width)
	width_auto = TRUE;
      autos++; }
  }

  for (i = 0; i < 3; i++) {
    int autos_copy = autos, which = -1, total_width = 0, j = 0;
    switch(i) {
    case 0: break;
    case 1:
      limit = 6;
      if (width_auto)
	autos_copy--;
      if (tentative_width > b->max_width.used)
	tentative_width = b->max_width.used;
      else continue;
    case 2:
      if (width_auto)
	autos_copy--;
      if (tentative_width < b->min_width.used)
	tentative_width = b->min_width.used;
      else continue;
    }
        
    for (j = 0; j < limit; j++) {
      if (widths[j]->computed == RUIN_CSS_VALUE_AUTO) {
	if (autos_copy == 1) {
	  which = j;
	  break;
	} else {
	  widths[j]->used = 0;
	  autos_copy--;
	}
      }
    }

    if (which == -1) 
      {
	char *direction = ruin_box_css_lookup (b, "direction");
	if (strcmp ("ltr", direction) == 0)
	  which = 1;
	else which = 0;
    }
    
    if (limit == 6)
      total_width = tentative_width;
    else total_width = 0;
    for (j = 0; j < limit; j++)
      if (j != which)
	total_width += widths[j]->used;
    widths[which]->used = cb_width - total_width;
    if ((widths[which]->used < 0) && 
	(widths[which] != &b->margin_left) &&
	(widths[which] != &b->margin_right))
      widths[which] = 0;
    if (limit == 7)
      tentative_width = b->width.used;
  }

  return tentative_width;
}

static int get_total_height (ruin_box_t *box) {
  return box->margin_top.used +
    box->border_top_width.used +
    box->padding_top.used +
    box->height.used +
    box->padding_bottom.used +
    box->border_bottom_width.used +
    box->margin_bottom.used;
}

static int is_visible (ruin_window_t *w, int top, int left) 
{
  return left >= 0 && left < w->width && top >= 0 && top < w->height;
}

static int layout_text
(ruin_box_t *l, ruin_inline_box_t *ib, int top, int left, 
 ruin_layout_line_state_t *state)
{
  ruin_box_t *b = (ruin_box_t *) ib;
  char *white_space = ruin_css_lookup_text_style 
    (b->window, (ruin_node_text_t *) b->generator, "white-space");

  ruin_inline_box_t *overflow = NULL;

  parse_inline_sizes ((ruin_box_t *) b);
  normalize_lengths (b->containing_block, b);

  b->top = top;
  b->left = left;
  b->height.used = 1;

  if (strcmp (white_space, "normal") == 0) {
    char *content_ptr = ib->content_ptr, *last_word_start = NULL;
    int len = strlen (content_ptr), width = 0, idx = 0, seen_space = FALSE;
    int offset = left - l->left;

    while (idx < len && offset + width < l->width.used)
      {
	char c = content_ptr[idx];

	if (isspace (c)) 
	  {
	    short more_non_space = FALSE;
	    seen_space = TRUE;

	    state->last_break_opportunity = ib;
	    state->last_break_opportunity_ptr = &content_ptr[idx];

	    while (idx < len)
	      {
		if (!isspace (content_ptr[idx++]))
		  {
		    more_non_space = TRUE;
		    idx--;
		    break;
		  }
	      }

	    if ((state->is_first 
		 && last_word_start != NULL 
		 && (more_non_space || !state->is_last)) 
		|| (!state->is_first 
		    && ((last_word_start == NULL && !isspace (state->last_char))
			|| (last_word_start != NULL && 
			    (more_non_space || !state->is_last)))))
	      {
		width += 1;
		if (more_non_space)
		  width += b->word_spacing.used;
		state->last_char = c;
	      }
	  }
	else 
	  {
	    if (seen_space)
	      {
		seen_space = FALSE;
		last_word_start = &content_ptr[idx];
	      }
	    else if (last_word_start == NULL)
	      last_word_start = content_ptr;

	    idx++;
	    width += 1;
	    state->last_char = c;
	    if (idx < len && offset + width < l->width.used)
	      width += b->letter_spacing.used;

	    if (ruin_css_pseudo_element_is_active 
		(b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER)
		&& !ispunct (c))
	      ruin_css_deactivate_pseudo_element 
		(b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);
	  }
      }

    if (idx < len)
      {
	if (state->last_break_opportunity == NULL)
	  assert (1 == 0);
	if (state->last_break_opportunity == ib)
	  {
	    if (isspace (content_ptr[idx]) && !seen_space)
	      state->last_break_opportunity_ptr = &content_ptr[idx];

	    b->width.used = width 
	      - ((&content_ptr[idx] - state->last_break_opportunity_ptr)
		 * (1 + b->letter_spacing.used));
	  }

	return FALSE;
      }
    else b->width.used = width;

  } else if (strcmp (white_space, "pre") == 0) {
    int len = strlen (ib->content_ptr);
    int width = 0;
    char *next_line = strchr (ib->content_ptr, '\a');
    int last_was_space = FALSE;
    int idx = 0;

    if (next_line != NULL)
      {
	char *overflow_ptr = next_line + sizeof (char);
	len = next_line - ib->content_ptr;
	overflow = ruin_inline_box_new 
	  (RUIN_LAYOUT_BOX_TYPE_INLINE, l, l->containing_block, b->window, 
	   b->generator, overflow_ptr);
      }
    
    while (idx < len)
      {
	width += 1;
	state->last_char = ib->content_ptr[idx++];
	if (isspace (state->last_char))
	  {
	    if (!last_was_space)
	      width += b->word_spacing.used;
	    last_was_space = TRUE;
	  }
	else 
	  {
	    if (ruin_css_pseudo_element_is_active 
		(b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER)
		&& !ispunct (state->last_char));
	      ruin_css_deactivate_pseudo_element 
		(b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);

	    if (last_was_space) last_was_space = FALSE;
	    else if (idx < len) width += b->letter_spacing.used;
	  }
      }

    b->width.used = width;    
  } else if (strcmp (white_space, "nowrap") == 0) {
    assert (1 == 0);
  } else if (strcmp (white_space, "pre-wrap") == 0) {
    assert (1 == 0);
  } else if (strcmp (white_space, "pre-line") == 0) {
    assert (1 == 0);
  }

  return TRUE;
}

static GList *split_inline_overflow
(ruin_box_t *b, char *content_ptr, GList *children)
{
  ruin_inline_box_t *target = (ruin_inline_box_t *) b;
  ruin_inline_box_t *c = (ruin_inline_box_t *) ruin_box_clone (b, FALSE);
  ruin_box_t *bc = (ruin_box_t *) c;
  GList *target_ptr = g_list_find (b->parent->children, target);
  GList *child_ptr = children;
  GList *ret = target_ptr->next;

  c->content_ptr = content_ptr;
  
  while (child_ptr != NULL)
    {
      ruin_box_t *child_box = (ruin_box_t *) child_ptr->data;

      child_box->parent = bc;
      child_ptr = child_ptr->next;
    }

  bc->children = children;

  if (target_ptr->next != NULL)
    {
      target_ptr->next->prev = NULL;
      target_ptr->next = NULL;
    }
  
  ret = g_list_prepend (ret, c);

  if (b->parent->type == RUIN_LAYOUT_BOX_TYPE_LINE)
    return ret;
  return split_inline_overflow (b->parent, NULL, ret);
}

static GList *layout_inline 
(ruin_box_t *l, ruin_inline_box_t *ib, int top, int left, 
 ruin_layout_line_state_t *state)
{
  if (ib->content_ptr != NULL)
    {
      if (!layout_text (l, ib, top, left, state))
	{
	  return split_inline_overflow 
	    ((ruin_box_t *) state->last_break_opportunity, 
	     state->last_break_opportunity_ptr, 
	     NULL);
	}
      else return NULL;
    }  
  else 
    {
      ruin_box_t *b = (ruin_box_t *) ib;
      GList *overflow = NULL;
      GList *child_ptr = b->children;
      ruin_layout_line_state_t substate = *state;

      b->top = top;
      b->left = left;
      b->height.used = 1;
      b->width.used = 0;
      
      while (child_ptr != NULL)
	{
	  ruin_box_t *child_box = (ruin_box_t *) child_ptr->data;
	  ruin_inline_box_t *child_inline_box = (ruin_inline_box_t *) child_box;

	  substate.is_last &= child_ptr->next == NULL;
	  overflow = layout_inline (l, child_inline_box, top, left, &substate);
	  
	  b->width.used += child_box->width.used;
	  
	  if (overflow != NULL)
	    return overflow;

	  else left += child_box->width.used;

	  substate.is_first = FALSE;
	  child_ptr = child_ptr->next;
	}

      state->is_first = FALSE;
      state->last_char = substate.last_char;
      state->last_break_opportunity = substate.last_break_opportunity;
      state->last_break_opportunity_ptr = substate.last_break_opportunity_ptr;

      return NULL;
    }
}

static void layout_line (ruin_box_t *b, int top, int left)
{
  GList *child_ptr = b->children;

  ruin_box_t *current_line = b;

  int line_width = b->containing_block->width.used;
  int offer_left = left;
  ruin_layout_line_state_t state;

  b->top = top;
  b->left = left;
  b->height.used = 1;
  b->width.used = line_width;

  state.is_first = TRUE;
  state.is_last = g_list_length (child_ptr) == 1;
  state.last_char = 0x0;
  state.last_break_opportunity = 0x0;
  state.last_break_opportunity_ptr = 0x0;

  ruin_css_activate_pseudo_element 
    (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);
  ruin_css_activate_pseudo_element 
    (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);

  while (child_ptr != NULL)
    {
      ruin_box_t *child_box = (ruin_box_t *) child_ptr->data;
      ruin_inline_box_t *child_inline_box = (ruin_inline_box_t *) child_box;
      GList *overflow = NULL;

      state.is_last = child_ptr->next == NULL;
      overflow = layout_inline 
	(current_line, child_inline_box, current_line->top, offer_left, &state);

      child_box->parent = current_line;
      
      if (overflow != NULL)
	{
	  GList *overflow_ptr = overflow;
	  GList *current_line_context = current_line->parent->children;
	  ruin_box_t *new_line = ruin_box_clone (current_line, FALSE);

	  new_line->top += 1;
	  
	  current_line->parent->children = g_list_insert_before
	    (current_line_context, 
	     (g_list_find (current_line_context, current_line))->next,
	     new_line);
	  
	  new_line->children = overflow;
	  while (overflow_ptr != NULL)
	    {
	      ruin_box_t *c = (ruin_box_t *) overflow_ptr->data;
	      c->parent = new_line;
	      overflow_ptr = overflow_ptr->next;
	    }

	  state.is_first = TRUE;
	  state.last_char = 0x0;
	  state.last_break_opportunity = 0x0;
	  state.last_break_opportunity_ptr = 0x0;

	  offer_left = left;

	  if (ruin_css_pseudo_element_is_active 
	      (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE))
	    ruin_css_deactivate_pseudo_element 
	      (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);

	  current_line = new_line;
	  child_ptr = overflow;
	}
      else 
	{
	  state.is_first = FALSE;

	  offer_left += child_box->width.used;
	  child_ptr = child_ptr->next;
	}
    }

  if (ruin_css_pseudo_element_is_active 
      (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE))
    ruin_css_deactivate_pseudo_element 
      (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);
}

static int decimal_width (int n)
{
  return ceil (log (n + 1) / log (10));
}

static void layout_list_marker (ruin_marker_box_t *m, int t, int l)
{
  ruin_box_t *b = (ruin_box_t *) m;

  b->height.used = 1;
  b->top = t;
  b->left = l;

  switch (m->style)
    {
    case RUIN_LAYOUT_LIST_STYLE_TYPE_DISC:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_CIRCLE:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_SQUARE:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_LATIN:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_LATIN:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ALPHA:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ALPHA: b->width.used = 1; break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL: 
      b->width.used = decimal_width (m->ordinal + 1) + 1; break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL_LEADING_ZERO:
      b->width.used = decimal_width (m->length) + 1; break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ROMAN:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ROMAN:
      b->width.used = ruin_util_arabic_to_roman_length (m->length) + 1; break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_GREEK:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_ARMENIAN:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_GEORGIAN: assert (1 == 0);
    case RUIN_LAYOUT_LIST_STYLE_TYPE_NONE: b->width.used = 0; break;
    }
}

static void layout_list_item_outside
(ruin_marker_box_t *marker, ruin_box_t *block, int t, int l)
{
  layout_list_marker (marker, t, l);
  layout_box (block, t, l + ((ruin_box_t *) marker)->width.used);
}

static void layout_block_children (ruin_box_t *b, int offer_top, int offer_left)
{
  GList *child_ptr = b->children;

  while (child_ptr != NULL)
    {
      ruin_box_t *child_box = (ruin_box_t *) child_ptr->data;

      if (child_box->type == RUIN_LAYOUT_BOX_TYPE_MARKER) 
	{
	  ruin_marker_box_t *list_item_marker = (ruin_marker_box_t *) child_box;
	  ruin_box_t *list_item_sibling = NULL;
	  
	  child_ptr = child_ptr->next;
	  assert (child_ptr != NULL);

	  list_item_sibling = (ruin_box_t *) child_ptr->data;
	  assert (list_item_sibling->generator == child_box->generator);
	  layout_list_item_outside 
	    (list_item_marker, list_item_sibling, offer_top, offer_left);

	  offer_top += b->padding_bottom.used;
	  offer_top += MAX (get_total_height (child_box), 
			    get_total_height (list_item_sibling));
	  offer_top += b->padding_top.used;

	  child_ptr = child_ptr->next;
	  continue;
	}
      else if (child_box->type == RUIN_LAYOUT_BOX_TYPE_LINE)
	{
	  layout_line (child_box, offer_top, offer_left);

	  offer_top += b->padding_bottom.used;
	  offer_top += get_total_height (child_box);      
	  offer_top += b->padding_top.used;

	  child_ptr = child_ptr->next;
	  while (child_ptr != NULL)
	    {
	      child_box = (ruin_box_t *) child_ptr->data;
	      if (child_box->type != RUIN_LAYOUT_BOX_TYPE_LINE)
		break;

	      offer_top += b->padding_bottom.used;
	      offer_top += get_total_height (child_box);      
	      offer_top += b->padding_top.used;

	      child_ptr = child_ptr->next;
	    }
	  continue;
	}
      else 
	{
	  layout_box (child_box, offer_top, offer_left);
      
	  if (child_ptr->next != NULL)
	    {
	      offer_top += b->padding_bottom.used;
	      offer_top += get_total_height (child_box);      
	      offer_top += b->padding_top.used;
	    }
	  child_ptr = child_ptr->next;
	}
    }

  if (b->height.computed == RUIN_CSS_VALUE_AUTO && b->children != NULL)
    {
      GList *last = g_list_last (b->children);
      ruin_box_t *b_ptr = (ruin_box_t *) last->data;
      b->height.used = b_ptr->top + get_total_height (b_ptr) - b->top;
    }
}

static void layout_block (ruin_box_t *b, int top, int left)
{
  int offer_top = 0, offer_left = 0;

  parse_block_sizes (b);
  normalize_lengths (b->containing_block, b);
  set_block_width (b, b->containing_block->width.used);

  if (!is_visible (b->window, top, left))
    {
      b->visible = FALSE;
      return;
    }
  
  offer_top = top + 
    b->margin_top.used +
    b->border_top_width.used +
    b->padding_top.used;

  offer_left = left +
    b->margin_left.used +
    b->border_left_width.used +
    b->padding_left.used;

  layout_block_children (b, offer_top, offer_left);
}

static void layout_table_cell 
(ruin_layout_table *table, ruin_layout_table_row *row, 
 ruin_layout_table_column *column, ruin_layout_table_cell *cell, 
 int top, int left)
{
  ruin_box_t *box = cell->node;
  int offer_top = 0, offer_left = 0;

  box->top = top;
  box->left = left;
  
  parse_block_sizes (box);
  normalize_lengths (box->containing_block, box);
  box->width.used = column->width;  

  offer_top = top 
    + box->margin_top.used
    + box->border_top_width.used 
    + box->padding_top.used;

  offer_left = left 
    + box->margin_left.used 
    + box->border_left_width.used 
    + box->padding_left.used;

  layout_block_children (box, offer_top, offer_left);
}

static void layout_table_row 
(ruin_layout_table *table, ruin_layout_table_row *row, int top, int left)
{
  GList *column_ptr = table->columns;
  GList *cell_ptr = row->cells;

  int max_height = 0;
  int offer_left = left;

  while (cell_ptr != NULL)
    {
      ruin_layout_table_cell *cell = (ruin_layout_table_cell *) cell_ptr->data;
      ruin_layout_table_column *column = (ruin_layout_table_column *) 
	column_ptr->data;

      layout_table_cell (table, row, column, cell, top, offer_left);

      if (cell->node->height.used > max_height)
	max_height = cell->node->height.used;

      offer_left += column->width;
      column_ptr = column_ptr->next;
      cell_ptr = cell_ptr->next;
    }

  cell_ptr = row->cells;
  while (cell_ptr != NULL)
    {
      ruin_layout_table_cell *cell = 
	(ruin_layout_table_cell *) cell_ptr->data;
      cell->node->width.used = max_height;
      cell_ptr = cell_ptr->next;
    }
}

static void layout_table (ruin_box_t *b, int t, int l)
{
  char *layout_algorithm = ruin_box_css_lookup (b, "table-layout");
  char *width = ruin_box_css_lookup (b, "width");

  ruin_layout_table *table = NULL;
  GList *row_ptr = NULL;

  int offer_top = t;

  if (strcmp (width, "auto") != 0 || strcmp (layout_algorithm, "fixed") == 0)
    table = ruin_layout_table_fixed (b);
  else table = ruin_layout_table_auto (b);

  row_ptr = table->rows;
  
  while (row_ptr != NULL)
    {
      layout_table_row (table, row_ptr->data, offer_top, l);
      row_ptr = row_ptr->next;
    }
}

static void layout_box (ruin_box_t *b, int t, int l)
{
  char *display = NULL;

  b->top = t;
  b->left = l;

  switch (b->type)
    {
    case RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK:
    case RUIN_LAYOUT_BOX_TYPE_BLOCK:
      display = ruin_box_css_lookup (b, "display");
      if (strcmp (display, "table") == 0)
	layout_table (b, t, l);
      else layout_block (b, t, l); 
      break;
    case RUIN_LAYOUT_BOX_TYPE_LINE:
      layout_line (b, t, l); break;
    case RUIN_LAYOUT_BOX_TYPE_MARKER:
      layout_list_marker ((ruin_marker_box_t *) b, t, l); break;
    default: assert (1 == 0);
    }
}

void ruin_layout_layout_elements (GList *list) 
{
  GList *list_ptr = list;

  while (list_ptr != NULL)
    {
      ruin_box_t *b = (ruin_box_t *) list_ptr->data;

      b->containing_block->width.used = b->window->width;
      b->containing_block->height.used = b->window->height;

      layout_box (b, 0, 0);
      list_ptr = list_ptr->next;
    }
}

GList *ruin_layout_generate_and_layout_elements 
(ruin_window_t *w, ruin_node_t *t) 
{
  GList *list = ruin_box_generate (w, t);
  ruin_layout_layout_elements (list);
  return list;
}

static void parse_size (ruin_box_t *b, ruin_length_t *l, char *p) 
{
  ruin_length_t r = ruin_css_parse_size (ruin_box_css_lookup (b, p));
  
  l->computed = r.computed;
  l->units = r.units;
}

static void parse_block_sizes (ruin_box_t *b) 
{
  parse_size(b, &(b->width), "width");
  parse_size(b, &(b->min_width), "min-width");
  parse_size(b, &(b->max_width), "max-width");

  parse_size(b, &(b->height), "height");
  parse_size(b, &(b->min_height), "min-height");
  parse_size(b, &(b->max_height), "max-height");

  parse_size(b, &(b->padding_top), "padding-top");
  parse_size(b, &(b->padding_left), "padding-left");
  parse_size(b, &(b->padding_bottom), "padding-bottom");
  parse_size(b, &(b->padding_right), "padding-right");
  
  parse_size(b, &(b->text_indent), "text-indent");

  parse_size(b, &(b->margin_top), "margin-top");
  parse_size(b, &(b->margin_left), "margin-left");
  parse_size(b, &(b->margin_bottom), "margin-bottom");
  parse_size(b, &(b->margin_right), "margin-right");

  if (strcmp (ruin_box_css_lookup (b, "line-height"), "normal") != 0)
    parse_size (b, &(b->line_height), "line-height");
  if (strcmp (ruin_box_css_lookup(b, "border-top-style"), "none") != 0)
    parse_size (b, &(b->border_top_width), "border-top-width");
  if (strcmp (ruin_box_css_lookup(b, "border-left-style"), "none") != 0)
    parse_size (b, &(b->border_left_width), "border-left-width");
  if (strcmp (ruin_box_css_lookup(b, "border-bottom-style"), "none") != 0)
    parse_size (b, &(b->border_bottom_width), "border-bottom-width");
  if (strcmp(ruin_box_css_lookup(b, "border-right-style"), "none") != 0)
    parse_size (b, &(b->border_right_width), "border-right-width");
}

static void parse_inline_sizes (ruin_box_t *b)
{
  parse_size (b, &(b->margin_left), "margin-left");
  parse_size (b, &(b->margin_right), "margin-right");

  if (strcmp (ruin_box_css_lookup (b, "line-height"), "normal") != 0)
    parse_size (b, &(b->line_height), "line-height");
  if (strcmp (ruin_box_css_lookup (b, "letter-spacing"), "normal") != 0)
    parse_size (b, &(b->letter_spacing), "letter-spacing");
  if (strcmp (ruin_box_css_lookup (b, "word-spacing"), "normal") != 0)
    parse_size (b, &(b->word_spacing), "word-spacing");
}

void ruin_layout_add_style (SCM *list, char *prop, char *value) 
{
  SCM addition = scm_list_2(scm_from_locale_symbol(prop), 
			    scm_from_locale_string(value));
  if (scm_eq_p(*list, SCM_EOL) != SCM_BOOL_T)
    scm_append_x(scm_list_2(SCM_CDR(*list), scm_list_1(addition)));
  else 
    {
      *list = scm_list_2(scm_from_locale_symbol("css"),
			 scm_list_2(scm_from_locale_symbol("*"), addition));
      scm_gc_protect_object(*list);
    }
}
