/* load.c: Load and draw routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <curses.h>
#include <libguile.h>
#include <limits.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>

#include "api/api.h"
#include "debug.h"
#include "handlers/handlers.h"
#include "layout.h"
#include "load.h"
#include "parse.h"
#include "render.h"
#include "scheme.h"
#include "util.h"
#include "window.h"
#include "xml.h"

#define GUILE_LOAD_PATH "%load-path"

static int ruin_initialized = FALSE;
static struct sigaction ruin_sigaction_old_action;

ruin_windows_t *_ruin_windows = NULL;

extern void _ruin_scm_init_api ();
extern void _ruin_scm_init_handlers ();
extern void _ruin_box_init ();
extern void _ruin_css_init ();

long ruin_load_layout_and_render (ruin_window_t *w, ruin_node_t *t) 
{
  long start_time = ruin_util_current_time_millis ();
  GList *root_boxes_ptr = NULL;

  w->root_boxes = ruin_layout_generate_and_layout_elements (w, t);
  root_boxes_ptr = w->root_boxes;

  wclear (w->window);
  
  while (root_boxes_ptr != NULL)
    {
      ruin_box_t *root_box = (ruin_box_t *) root_boxes_ptr->data;
      
      ruin_debug_print_box_tree (root_box);
      ruin_render_render_tree (root_box);

      root_boxes_ptr = root_boxes_ptr->next;
    }

  wrefresh (w->window);

  return ruin_util_current_time_millis () - start_time;
}

int ruin_draw (ruin_window_t *w, SCM doc) 
{
  ruin_node_t *tree = NULL;
  int top, left, bottom, right;
  
  SCM cascade;
  SCM selection_context;

  enum ruin_xml_dialect lang = ruin_parse_determine_dialect (w, doc);

  cascade = ruin_scheme_scss_make_cascade (w);
  scm_gc_protect_object (cascade);

  selection_context = ruin_scheme_scss_make_selection_context 
    (w, ruin_scheme_scss_document_interface_sdom, 
     ruin_scheme_scss_make_rendering_interface 
     (w, scm_c_eval_string ("ruin:scss-pseudo-class-handler"), 
      scm_c_eval_string ("ruin:scss-pseudo-element-handler")), 
     cascade);
  scm_gc_protect_object (selection_context);

  w->cascade = cascade;
  w->selection_context = selection_context;

  ruin_window_clear (w);

  /* Generate default stylesheet.  If the document has an attached stylesheet
     or adds style on a per-element basis, we can apply that on top of the 
     default. */

  getbegyx (w->window, top, left);
  getmaxyx (w->window, bottom, right);

  ruin_util_log (w, "window dimensions are %d, %d", right, bottom);

  switch (lang) {
  case RUIN_XML_DIALECT_XUL:
    ruin_scheme_scss_set_cascade_agent
      (w, cascade, scm_copy_tree (_ruin_windows->xul_agent_css));
    break;
  case RUIN_XML_DIALECT_XHTML:
    ruin_scheme_scss_set_cascade_agent
      (w, cascade, scm_copy_tree (_ruin_windows->xhtml_agent_css));
  default:
    break;
  }

  tree = ruin_parse_document (w, doc);

  ruin_scheme_sdom_dispatch_event (w, tree->doc, "load"); 
  ruin_util_log (w, "total time %ldms", ruin_load_layout_and_render (w, tree));

  return TRUE;
}

int ruin_draw_string(ruin_window_t *w, char *doc) {
  if (doc != NULL) {
    SCM dom_doc = ruin_scheme_sdom_xml_to_sdom
      (w, scm_open_input_string(scm_from_locale_string(doc)), SCM_EOL);
    return ruin_draw(w, dom_doc);
  }
  return FALSE;
}

int ruin_draw_file(ruin_window_t *w, char *filename) { 
  if (filename != NULL) {
    char *abs_path = ruin_util_get_parent_directory(filename);
    SCM dom_doc = ruin_scheme_sdom_xml_to_sdom
      (w, scm_open_file(scm_from_locale_string(filename), 
			scm_from_locale_string("r")), 
       SCM_EOL);
    scm_call_2(scm_c_eval_string("sdom:set-document-uri!"), dom_doc,
	       scm_string_append(scm_list_2(scm_from_locale_string("file://"),
					    scm_from_locale_string(abs_path))));
    free(abs_path);
    return ruin_draw(w, dom_doc);
  }
  return FALSE;
}

void _set_ruin_temp_load_path(SCM oldval, char *varname) {
  char *tmpval = getenv(varname);
  scm_set_car_x(scm_c_eval_string(GUILE_LOAD_PATH),
		scm_from_locale_string 
		(tmpval != NULL ? tmpval : RUIN_SCHEME_PATH));
  scm_set_cdr_x(scm_c_eval_string(GUILE_LOAD_PATH), oldval);
  return;
}

int ruin_init() {
  SCM old_path = scm_list_copy(scm_c_eval_string(GUILE_LOAD_PATH));
  extern pthread_mutex_t _ruin_util_id_lock;

  struct sigaction new_action;
  new_action.sa_sigaction = ruin_window_signal_handler_SIGWINCH;
  new_action.sa_flags = SA_SIGINFO;
  sigemptyset(&new_action.sa_mask);

  if (ruin_initialized)
    return FALSE;

  pthread_mutex_init(&_ruin_util_id_lock, NULL);
  (void) sigaction(SIGWINCH, NULL, &ruin_sigaction_old_action);
  (void) sigaction(SIGWINCH, &new_action, NULL);

  /* 
   * Upon load, `(rnrs exceptions)' registers the r6rs:exception 
   * exception-printer. 
   */
  
  scm_c_use_module("rnrs exceptions");

  _set_ruin_temp_load_path(old_path, "RUIN_SCHEME_SXML_PATH");
  scm_c_use_module("sxml ssax");

  _set_ruin_temp_load_path(old_path, "RUIN_SCHEME_SDOM_PATH");
  scm_c_use_module("sdom core");
  scm_c_use_module("sdom events");

  _set_ruin_temp_load_path(old_path, "RUIN_SCHEME_SCSS_PATH");
  scm_c_use_module("scss scss");
  scm_c_use_module("scss interface sdom");
  scm_set_car_x(scm_c_eval_string(GUILE_LOAD_PATH), SCM_CAR(old_path));
  scm_set_cdr_x(scm_c_eval_string(GUILE_LOAD_PATH), SCM_CDR(old_path));

  ruin_parse_init ();

  /* Also initialize our own Scheme API. */

  ruin_scheme_init ();
  _ruin_scm_init_api ();
  _ruin_scm_init_handlers ();
  _ruin_css_init ();
  _ruin_box_init ();

  /* Set up a default color pair... */
  init_pair(1, COLOR_WHITE, COLOR_BLACK);

  _ruin_windows = ruin_windows_new();
  return TRUE;
}

void ruin_shutdown() {
  extern pthread_mutex_t _ruin_util_id_lock;
  if (!ruin_initialized)
    return;
  pthread_mutex_destroy(&_ruin_util_id_lock);
  ruin_windows_free(_ruin_windows);
  (void) sigaction(SIGWINCH, &ruin_sigaction_old_action, NULL);
  return;
}
