/* scheme.c: Initializations for libRUIN's Scheme interface
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <libguile.h>
#include <string.h>

#include "scheme.h"
#include "window.h"

static SCM ruin_scheme_pe = SCM_EOL;

static SCM ruin_scheme_scss_sip_p = SCM_EOL;
static SCM ruin_scheme_scss_cth = SCM_EOL;
static SCM ruin_scheme_scss_sca_x = SCM_EOL;
static SCM ruin_scheme_scss_ca = SCM_EOL;
static SCM ruin_scheme_scss_scua_x = SCM_EOL;
static SCM ruin_scheme_scss_sv = SCM_EOL;
static SCM ruin_scheme_scss_gdv = SCM_EOL;
static SCM ruin_scheme_scss_mc = SCM_EOL;
static SCM ruin_scheme_scss_cts = SCM_EOL;
static SCM ruin_scheme_scss_mri = SCM_EOL;
static SCM ruin_scheme_scss_msc = SCM_EOL;
static SCM ruin_scheme_scss_svip = SCM_EOL;
static SCM ruin_scheme_scss_svp = SCM_EOL;
static SCM ruin_scheme_scss_svs = SCM_EOL;
static SCM ruin_scheme_scss_svv = SCM_EOL;
static SCM ruin_scheme_scss_ss = SCM_EOL;

static SCM ruin_scheme_sdom_cd = SCM_EOL;
static SCM ruin_scheme_sdom_cdt = SCM_EOL;
static SCM ruin_scheme_sdom_ce = SCM_EOL;
static SCM ruin_scheme_sdom_ctn = SCM_EOL;
static SCM ruin_scheme_sdom_sa = SCM_EOL;
static SCM ruin_scheme_sdom_snn = SCM_EOL;
static SCM ruin_scheme_sdom_ssa = SCM_EOL;
static SCM ruin_scheme_sdom_sga = SCM_EOL;
static SCM ruin_scheme_sdom_sn = SCM_EOL;
static SCM ruin_scheme_sdom_sde_prop = SCM_EOL;
static SCM ruin_scheme_sdom_sd_p = SCM_EOL;
static SCM ruin_scheme_sdom_sdt = SCM_EOL;
static SCM ruin_scheme_sdom_sdu = SCM_EOL;
static SCM ruin_scheme_sdom_se_p = SCM_EOL;
static SCM ruin_scheme_sdom_sac = SCM_EOL;
static SCM ruin_scheme_sdom_sfc = SCM_EOL;
static SCM ruin_scheme_sdom_sns = SCM_EOL;
static SCM ruin_scheme_sdom_snv = SCM_EOL;
static SCM ruin_scheme_sdom_sod = SCM_EOL;
static SCM ruin_scheme_sdom_spn = SCM_EOL;
static SCM ruin_scheme_sdom_stc = SCM_EOL;
static SCM ruin_scheme_sdom_stn_p = SCM_EOL;
static SCM ruin_scheme_sdom_sv = SCM_EOL;

static SCM ruin_scheme_sdom_ael = SCM_EOL;
static SCM ruin_scheme_sdom_sde = SCM_EOL;
static SCM ruin_scheme_sdom_xts = SCM_EOL;

static SCM ruin_scheme_pre_unwind_handler (void *data, SCM key, SCM args) {
  ((void **) data)[3] = scm_make_stack (SCM_BOOL_T, SCM_EOL);
  return SCM_UNDEFINED;
}

static SCM ruin_scheme_catch_body(void *expr) {
  SCM e = *((SCM *) expr);
  return scm_apply (SCM_CAR (e), SCM_CDR (e), SCM_EOL);
}

static SCM ruin_scheme_catch_handler(void *data, SCM key, SCM args) {
  ruin_window_t *w = (ruin_window_t *) ((void **) data)[0];
  SCM stack = (SCM) ((void **) data)[3];
  SCM frame = SCM_BOOL_F;
  SCM port = scm_open_output_string ();
  char *port_str = NULL;

  if (scm_stack_p (stack) == SCM_BOOL_T)    
    frame = scm_stack_ref (stack, scm_from_uint (0));

  *((int *) ((void **) data)[1]) = FALSE;

  scm_display(scm_from_locale_string ("error calling "), port);
  scm_display(scm_from_locale_string ((char *) ((void **) data)[2]), port);
  scm_display(scm_from_locale_string (": "), port);

  scm_call_4 (ruin_scheme_pe, port, frame, key, args);

  if (frame != SCM_BOOL_F)
    scm_display_backtrace (stack, port, SCM_BOOL_F, SCM_BOOL_F);
  
  port_str = scm_to_locale_string (scm_get_output_string (port));
  ruin_util_log (w, port_str);
  free (port_str);

  return SCM_UNDEFINED;
}

int ruin_scheme_wrap(ruin_window_t *w, char *fn, SCM expr, SCM *result) {
  int success = TRUE;
  SCM res = SCM_EOL;

  void *handler_data[4];
  handler_data[0] = w;
  handler_data[1] = &success;
  handler_data[2] = fn;
  handler_data[3] = NULL;

  res = scm_c_catch(SCM_BOOL_T,
		    ruin_scheme_catch_body, &expr, 
		    ruin_scheme_catch_handler, handler_data,
		    ruin_scheme_pre_unwind_handler, handler_data);
  if (success && result != NULL)
    *result = res;
  return success;
}

SCM call_scm_function (ruin_window_t *w, SCM fn, char *fn_name, SCM arg)
{
  SCM r = SCM_EOL;
  ruin_scheme_wrap (w, fn_name, scm_cons (fn, arg), &r);
  return r;  
}

int call_boolean_function (ruin_window_t *w, SCM fn, char *fn_name, SCM arg)
{
  SCM r = SCM_EOL;
  ruin_scheme_wrap (w, fn_name, scm_cons (fn, arg), &r);
  return r == SCM_BOOL_T ? 1 : 0;  
}

char *call_char_function (ruin_window_t *w, SCM fn, char *fn_name, SCM arg)
{
  SCM r = SCM_EOL;
  ruin_scheme_wrap (w, fn_name, scm_cons (fn, arg), &r);
  return scm_string_p (r) == SCM_BOOL_T ? scm_to_locale_string (r) : NULL;
}

SCM ruin_scheme_scss_color_to_hex(ruin_window_t *w, char *c) {
  SCM r = scm_from_locale_string ("#000000");
  SCM sc = scm_from_locale_string (c);
  (void) ruin_scheme_wrap
    (w, "scss:color->hex", scm_list_2 (ruin_scheme_scss_cth, sc), &r);
  return r;
}

void ruin_scheme_scss_set_cascade_author (ruin_window_t *w, SCM c, SCM s) {
  ruin_scheme_wrap(w, "scss:set-cascade-author!", 
		   scm_list_3 (ruin_scheme_scss_sca_x, c, s), NULL); 
}

SCM ruin_scheme_scss_cascade_author (ruin_window_t *w, SCM c)
{
  return call_scm_function 
    (w, ruin_scheme_scss_ca, "sdom:cascade-author", scm_list_1 (c));
}

void ruin_scheme_scss_set_cascade_agent (ruin_window_t *w, SCM c, SCM s) {
  ruin_scheme_wrap(w, "scss:set-cascade-agent!", 
		   scm_list_3 (ruin_scheme_scss_scua_x, c, s), NULL); 
}

int ruin_scheme_scss_is_inherited(ruin_window_t *w, char *prop) {
  SCM r = SCM_BOOL_F;
  SCM sp = scm_from_locale_symbol (prop);
  (void) ruin_scheme_wrap
    (w, "scss:inherited?", scm_list_2 (ruin_scheme_scss_sip_p, sp), &r);
  return r == SCM_BOOL_T ? 1 : 0; 
}

char *ruin_scheme_scss_get_default_value(ruin_window_t *w, char *p) {
  return call_char_function 
    (w, ruin_scheme_scss_gdv, "sdom:get-default-value", 
     scm_list_1 (scm_from_locale_symbol (p)));
}

SCM ruin_scheme_scss_make_rendering_interface 
(ruin_window_t *w, SCM pch, SCM peh)
{
  return call_scm_function 
    (w, ruin_scheme_scss_mri, "scss:make-rendering-interface", 
     scm_list_2 (pch, peh));
}

SCM ruin_scheme_scss_make_cascade(ruin_window_t *w) {
  return call_scm_function
    (w, ruin_scheme_scss_mc, "scss:make-cascade",
     scm_list_3 (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F));
}

SCM ruin_scheme_scss_css_to_scss(ruin_window_t *w, char *c, char *d) {
  return ruin_scheme_scss_css_to_scss_port
    (w, scm_open_input_string (scm_from_locale_string(c)), d);
}

SCM ruin_scheme_scss_css_to_scss_port(ruin_window_t *w, SCM p, char *d) {
  SCM ret = SCM_EOL;  
  long start = ruin_util_current_time_millis(), end = 0;

  if (d != NULL)
    (void) ruin_scheme_wrap
      (w, "scss:css->scss", scm_list_3 
       (ruin_scheme_scss_cts, p, scm_from_locale_string (d)), &ret);
  else (void) ruin_scheme_wrap 
	 (w, "scss:css->scss", scm_list_2 (ruin_scheme_scss_cts, p), &ret);

  end = ruin_util_current_time_millis();
  ruin_util_log(w, "parsing stylesheet took %d ms", end - start);

  scm_gc_protect_object(ret);
  return ret;
}

SCM ruin_scheme_scss_make_selection_context 
(ruin_window_t *w, SCM document_interface, SCM rendering_interface, SCM cascade)
{
  return call_scm_function 
    (w, ruin_scheme_scss_msc, "scss:make-selection-context", 
     scm_list_3 (document_interface, rendering_interface, cascade));
}

int ruin_scheme_scss_selected_value_important_p
(ruin_window_t *w, SCM selected_value)
{
  return call_boolean_function 
    (w, ruin_scheme_scss_svip, "scss:selected-value-important?", 
     scm_list_1 (selected_value));
}

char *ruin_scheme_scss_selected_value_property
(ruin_window_t *w, SCM selected_value)
{
  SCM property_symbol = call_scm_function 
    (w, ruin_scheme_scss_svp, "scss:selected-value-property", 
     scm_list_1 (selected_value));
  return scm_to_locale_string (scm_symbol_to_string (property_symbol));
}

char *ruin_scheme_scss_selected_value_source
(ruin_window_t *w, SCM selected_value)
{
  SCM source_symbol = call_scm_function
    (w, ruin_scheme_scss_svs, "scss:selected-value-source", 
     scm_list_1 (selected_value));
  return scm_to_locale_string (scm_symbol_to_string (source_symbol));
}

char *ruin_scheme_scss_selected_value_value
(ruin_window_t *w, SCM selected_value)
{
  return call_char_function
    (w, ruin_scheme_scss_svv, "scss:selected-value-value", 
     scm_list_1 (selected_value));
}

SCM ruin_scheme_scss_select_values 
(ruin_window_t *w, SCM selection_context, SCM node)
{
  return call_scm_function
    (w, ruin_scheme_scss_sv, "scss:select-values", 
     scm_list_2 (selection_context, node));
}

char *ruin_scheme_scss_specify (ruin_window_t *w, char *prop, char *value)
{
  return call_char_function 
    (w, ruin_scheme_scss_ss, "scss:specify",
     scm_list_2 (scm_from_locale_symbol (prop), 
		 scm_from_locale_string (value)));
}

SCM ruin_scheme_sdom_create_document (ruin_window_t *w, char *r, SCM d)
{
  return call_scm_function
    (w, ruin_scheme_sdom_cd, "sdom:create-document", 
     scm_list_2 (scm_from_locale_string (r), d));
}

SCM ruin_scheme_sdom_create_document_type (ruin_window_t *w, char *r)
{
  return call_scm_function
    (w, ruin_scheme_sdom_cdt, "sdom:create-document-type",
     scm_list_1 (scm_from_locale_string (r)));
}

SCM ruin_scheme_sdom_create_element (ruin_window_t *w, SCM d, char *e)
{
  return call_scm_function
    (w, ruin_scheme_sdom_ce, "sdom:create-element", 
     scm_list_2 (d, scm_from_locale_string (e)));
}

SCM ruin_scheme_sdom_create_text_node (ruin_window_t *w, SCM d, char *c)
{
  return call_scm_function
    (w, ruin_scheme_sdom_ctn, "sdom:create-text-node",
     scm_list_2 (d, scm_from_locale_string (c)));
}

SCM ruin_scheme_sdom_attributes (ruin_window_t *w, SCM e)
{
  return call_scm_function 
    (w, ruin_scheme_sdom_sa, "sdom:attributes", scm_list_1 (e));
}

SCM ruin_scheme_sdom_doctype (ruin_window_t *w, SCM doc) 
{
  return call_scm_function 
    (w, ruin_scheme_sdom_sdt, "sdom:doctype", scm_list_1 (doc));
}

SCM ruin_scheme_sdom_document_element (ruin_window_t *w, SCM doc)
{
  return call_scm_function 
    (w, ruin_scheme_sdom_sde_prop, "sdom:document-element", scm_list_1 (doc));
}

SCM ruin_scheme_sdom_owner_document (ruin_window_t *w, SCM e)
{
  return call_scm_function 
    (w, ruin_scheme_sdom_sod, "sdom:owner-document", scm_list_1 (e));
}

SCM ruin_scheme_sdom_parent_node (ruin_window_t *w, SCM e)
{
  return call_scm_function 
    (w, ruin_scheme_sdom_spn, "sdom:parent-node", scm_list_1 (e));
}

void ruin_scheme_sdom_append_child (ruin_window_t *w, SCM n, SCM c)
{
  call_scm_function
    (w, ruin_scheme_sdom_sac, "sdom:append-child!", scm_list_2 (n, c));
}

SCM ruin_scheme_sdom_first_child (ruin_window_t *w, SCM e)
{
  return call_scm_function 
    (w, ruin_scheme_sdom_sfc, "sdom:first-child", scm_list_1 (e));
}

SCM ruin_scheme_sdom_next_sibling (ruin_window_t *w, SCM e)
{
  return call_scm_function 
    (w, ruin_scheme_sdom_sns, "sdom:next-sibling", scm_list_1 (e));
}

int ruin_scheme_sdom_document_p (ruin_window_t *w, SCM e)
{
  return call_boolean_function 
    (w, ruin_scheme_sdom_sd_p, "sdom:document?", scm_list_1 (e));
}

int ruin_scheme_sdom_element_p (ruin_window_t *w, SCM e)
{
  return call_boolean_function 
    (w, ruin_scheme_sdom_se_p, "sdom:element?", scm_list_1 (e));
}

int ruin_scheme_sdom_text_node_p (ruin_window_t *w, SCM e)
{
  return call_boolean_function 
    (w, ruin_scheme_sdom_stn_p, "sdom:text-node?", scm_list_1 (e));
}

char *ruin_scheme_sdom_document_uri (ruin_window_t *w, SCM d)
{
  return call_char_function 
    (w, ruin_scheme_sdom_sdu, "sdom:document-uri", scm_list_1 (d));
}

char *ruin_scheme_sdom_name (ruin_window_t *w, SCM n)
{
  return call_char_function 
    (w, ruin_scheme_sdom_sn, "sdom:name", scm_list_1 (n));
}

char *ruin_scheme_sdom_node_name (ruin_window_t *w, SCM n)
{
  return call_char_function 
    (w, ruin_scheme_sdom_snn, "sdom:node-name", scm_list_1 (n));
}

char *ruin_scheme_sdom_node_value (ruin_window_t *w, SCM n)
{
  return call_char_function 
    (w, ruin_scheme_sdom_snv, "sdom:node-value", scm_list_1 (n));
}

char *ruin_scheme_sdom_text_content (ruin_window_t *w, SCM e)
{
  return call_char_function 
    (w, ruin_scheme_sdom_stc, "sdom:text-content", scm_list_1 (e));
}

char *ruin_scheme_sdom_value (ruin_window_t *w, SCM n)
{
  return call_char_function 
    (w, ruin_scheme_sdom_sv, "sdom:value", scm_list_1 (n));
}

void ruin_scheme_sdom_set_attribute 
(ruin_window_t *w, SCM n, char *attr, char *val)
{
  call_scm_function 
    (w, ruin_scheme_sdom_ssa, "sdom:set-attribute!", 
     scm_list_3 (n, scm_from_locale_string (attr), 
		 scm_from_locale_string (val)));
}

char *ruin_scheme_sdom_get_attribute(ruin_window_t *w, SCM n, char *attr) {
  SCM sprop = scm_from_locale_string (attr);
  SCM r = SCM_EOL;

  (void) ruin_scheme_wrap
    (w, "sdom:get-attribute", scm_list_3 (ruin_scheme_sdom_sga, n, sprop), &r);

  if (scm_string_p(r) != SCM_BOOL_T)
    return NULL;
  return scm_to_locale_string(r);
}

void ruin_scheme_sdom_dispatch_event(ruin_window_t *w, SCM n, char *event) {
  SCM sevent = scm_from_locale_string(event);
  ruin_scheme_wrap(w, "sdom:dispatch-event", 
		   scm_list_3(ruin_scheme_sdom_sde, n, sevent), NULL);
}

void ruin_scheme_sdom_add_event_listener(ruin_window_t *w, SCM n, char *event,
					 char *group, char *handler) {
  SCM sevent = scm_from_locale_string (event);
  SCM sgroup = scm_from_locale_string (group);
  SCM handler_proc = scm_c_eval_string(handler);

  ruin_scheme_wrap(w, "sdom:add-event-listener", 
		   scm_list_n(ruin_scheme_sdom_ael, n, sevent, 
			      sgroup, handler_proc, SCM_BOOL_F, SCM_UNDEFINED),
		   NULL);
}

SCM ruin_scheme_sdom_xml_to_sdom(ruin_window_t *w, SCM p, SCM ns) {
  SCM r = SCM_EOL;
  (void) ruin_scheme_wrap
    (w, "sdom:xml->sdom", scm_list_3 (ruin_scheme_sdom_xts, p, ns), &r);
  scm_gc_protect_object(r);
  return r;
}

#define PROTECTED_BINDING(var, expr) \
  var = scm_c_eval_string (expr); \
  scm_gc_protect_object (var);

void ruin_scheme_init() {
  PROTECTED_BINDING (ruin_scheme_pe, "print-exception");

  PROTECTED_BINDING (ruin_scheme_scss_document_interface_sdom, 
		     "scss:document-interface-sdom");
  
  PROTECTED_BINDING (ruin_scheme_scss_sip_p, "scss:inherited?");
  PROTECTED_BINDING (ruin_scheme_scss_cth, "scss:color->hex");
  PROTECTED_BINDING (ruin_scheme_scss_sca_x, "scss:set-cascade-author!");
  PROTECTED_BINDING (ruin_scheme_scss_ca, "scss:cascade-author");
  PROTECTED_BINDING (ruin_scheme_scss_scua_x, "scss:set-cascade-agent!");
  PROTECTED_BINDING (ruin_scheme_scss_gdv, "scss:get-default-value");
  PROTECTED_BINDING (ruin_scheme_scss_mc, "scss:make-cascade");
  PROTECTED_BINDING (ruin_scheme_scss_cts, "scss:css->scss");
  PROTECTED_BINDING (ruin_scheme_scss_mri, "scss:make-rendering-interface");
  PROTECTED_BINDING (ruin_scheme_scss_msc, "scss:make-selection-context");
  PROTECTED_BINDING (ruin_scheme_scss_svip, "scss:selected-value-important?");
  PROTECTED_BINDING (ruin_scheme_scss_svp, "scss:selected-value-property");
  PROTECTED_BINDING (ruin_scheme_scss_svs, "scss:selected-value-source");
  PROTECTED_BINDING (ruin_scheme_scss_svv, "scss:selected-value-value");
  PROTECTED_BINDING (ruin_scheme_scss_sv, "scss:select-values");
  PROTECTED_BINDING (ruin_scheme_scss_ss, "scss:specify");

  PROTECTED_BINDING (ruin_scheme_sdom_cd, "sdom:create-document");
  PROTECTED_BINDING (ruin_scheme_sdom_cdt, "sdom:create-document-type");
  PROTECTED_BINDING (ruin_scheme_sdom_ce, "sdom:create-element");
  PROTECTED_BINDING (ruin_scheme_sdom_ctn, "sdom:create-text-node");
  PROTECTED_BINDING (ruin_scheme_sdom_sa, "sdom:attributes");
  PROTECTED_BINDING (ruin_scheme_sdom_snn, "sdom:node-name");
  PROTECTED_BINDING (ruin_scheme_sdom_sn, "sdom:name");
  PROTECTED_BINDING (ruin_scheme_sdom_sde_prop, "sdom:document-element");
  PROTECTED_BINDING (ruin_scheme_sdom_sdt, "sdom:doctype");
  PROTECTED_BINDING (ruin_scheme_sdom_sdu, "sdom:document-uri");
  PROTECTED_BINDING (ruin_scheme_sdom_sd_p, "sdom:document?");
  PROTECTED_BINDING (ruin_scheme_sdom_se_p, "sdom:element?");
  PROTECTED_BINDING (ruin_scheme_sdom_sac, "sdom:append-child!");
  PROTECTED_BINDING (ruin_scheme_sdom_sfc, "sdom:first-child");
  PROTECTED_BINDING (ruin_scheme_sdom_sns, "sdom:next-sibling");
  PROTECTED_BINDING (ruin_scheme_sdom_snv, "sdom:node-value");
  PROTECTED_BINDING (ruin_scheme_sdom_stc, "sdom:text-content");
  PROTECTED_BINDING (ruin_scheme_sdom_stn_p, "sdom:text-node?");
  PROTECTED_BINDING (ruin_scheme_sdom_sv, "sdom:value");
  PROTECTED_BINDING (ruin_scheme_sdom_ssa, "sdom:set-attribute!");
  PROTECTED_BINDING (ruin_scheme_sdom_sga, "sdom:get-attribute");
  PROTECTED_BINDING (ruin_scheme_sdom_ael, "sdom:add-event-listener!");
  PROTECTED_BINDING (ruin_scheme_sdom_sde, "sdom:dispatch-event");
  PROTECTED_BINDING (ruin_scheme_sdom_xts, "sdom:xml->sdom");
  PROTECTED_BINDING (ruin_scheme_sdom_sod, "sdom:owner-document");
  PROTECTED_BINDING (ruin_scheme_sdom_spn, "sdom:parent-node");
}
