;; scss.scm: main module exports and implementations for SCSS
;; Copyright (C) 2011 Julian Graham

;; SCSS is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

#!r6rs

(library (scss scss)
  (export scss:color->hex

	  scss:make-selection-context
	  scss:make-document-interface
	  scss:make-rendering-interface
	  
	  scss:set-uri-import-handler!
	  
	  scss:stylesheet?
	  
	  scss:make-cascade
	  scss:cascade?
	  scss:cascade-agent
	  scss:cascade-author
	  scss:cascade-user
	  scss:set-cascade-agent!
	  scss:set-cascade-author!
	  scss:set-cascade-user!
	  
	  scss:inherited?
	  scss:get-default-value
	  
	  scss:select-values
	  
	  scss:css->scss
	  scss:scss->css
	  
	  scss:specify
	  
	  scss:selected-value?
	  scss:selected-value-selector
	  scss:selected-value-property
	  scss:selected-value-source
	  scss:selected-value-value
	  scss:selected-value-important?)
  
  (import (rnrs)

	  (scss common)
	  (scss lexer)
	  (scss parser)
	  (scss properties)
	  (only (srfi :1) filter-map)
	  (only (srfi :13) string-prefix? 
		           string-index 
			   string-index-right 
			   string-tokenize)
	  (srfi :14))

  (define-record-type (scss:cascade scss:make-cascade scss:cascade?)
    (fields (mutable agent scss:cascade-agent scss:set-cascade-agent!)
	    (mutable author scss:cascade-author scss:set-cascade-author!)
	    (mutable user scss:cascade-user scss:set-cascade-user!)))
  
  (define-record-type (scss:document-interface 
		       scss:make-document-interface 
		       scss:document-interface?)
    (fields name-function 
	    
	    parent-function
	    prev-sibling-function
	    
	    attribute-function 
	    class-function 
	    id-function))
  
  (define-record-type (scss:rendering-interface
		       scss:make-rendering-interface
		       scss:rendering-interface?)
    (fields pseudo-class-predicate pseudo-element-predicate))
  
  (define-record-type (scss:selection-context 
		       scss:make-selection-context 
		       scss:selection-context?)
    (fields document-interface rendering-interface cascade))
  
  (define-record-type (scss:selected-value
		       scss:make-selected-value
		       scss:selected-value?)
    (fields selector property value important? source order))
  
  (define (apply-interface-fn fn . x) (and fn (apply fn x)))
  (define (scss:parent iface x) 
    (apply-interface-fn (scss:document-interface-parent-function iface) x))
  (define (scss:prev-sibling iface x) 
    (apply-interface-fn 
     (scss:document-interface-prev-sibling-function iface) x))
  (define (scss:name iface x) 
    (apply-interface-fn (scss:document-interface-name-function iface) x))
  (define (scss:attr iface x attr)
    (apply-interface-fn 
     (scss:document-interface-attribute-function iface) x attr))
  (define (scss:class iface x)
    (apply-interface-fn (scss:document-interface-class-function iface) x))
  (define (scss:id iface x) 
    (apply-interface-fn (scss:document-interface-id-function iface) x))
  (define (scss:pseudo-class? iface x y)
    (apply-interface-fn 
     (scss:rendering-interface-pseudo-class-predicate iface) x y))
  (define (scss:pseudo-element? iface x y)
    (apply-interface-fn
     (scss:rendering-interface-pseudo-element-predicate iface) x y))
  
  (define (internal-uri-import-handler uri) (open-string-input-port ""))
  (define (scss:set-uri-import-handler! p)
    (verify-arg-types "scss:set-uri-import-handler!"
		      (list procedure?)
		      (list p)
		      1)
    (set! internal-uri-import-handler p))
  
  (define (end-index str start-char from)
    (define (index-filter x) (string-index str x from))
    (let* ((chrs (remove start-char (list #\. #\: #\x007c #\[)))
	   (indices (filter-map index-filter chrs)))
      (if (null? indices)
	  (string-length str)
	  (car (list-sort < indices)))))
  
  (define (class? x recurse) 
    (define (f y z)
      (define (g a) (substring y (+ z 1) a))
      (or (let ((ei (end-index y #\. (+ z 1)))) (and ei (g ei)))
	  (substring y (+ z 1))))
    
    (if (list? x)
	(and (eq? (car x) 'class)
	     (case (length x)
	       ((2) (cadr x))
	       ((3) (and (if recurse (selector? (cadr x) #t) #t)
			 (symbol? (caddr x)) 
			 (caddr x)))
	       (else #f)))
	(let ((y (symbol->string x)))
	  (let ((z (string-index y #\.)))
	    (if (and z (not (eqv? z (string-index-right y #\. (+ z 1)))))
		(f y z)
		#f)))))
  
  (define (colon-match? x sym lst recurse)
    (define (xor x y) (if x (and (not y) x) y))
    
    (if (list? x)
	(and (eq? (car x) sym)
	     (case (length x)
	       ((2) (let ((m (member (cadr x) lst))) (and m (car m))))
	       ((3) (and (if recurse (selector? (cadr x) recurse) #t) 
			 (let ((m (member (caddr x) lst))) (and m (car m)))))
	       (else #f)))
	(let ((y (symbol->string x)))
	  (let* ((z1 (string-index y #\:))
		 (z2 (if z1 (string-index y #\: (+ z1 1)) #f))
		 (s1 (and z1 (if z2 
				 (substring y (+ z1 1) z2)
				 (substring y (+ z1 1)))))
		 (s2 (if z2 (substring y (+ z2 1)) #f)))
	    (xor (let ((m (member s1 lst))) (and m (car m))) 
		 (let ((m (member s2 lst))) (and m (car m))))))))
  
  (define (pseudo-class? x recurse) 
    (colon-match? x 'pclass pseudo-classes recurse))
  (define (pseudo-element? x recurse) 
    (colon-match? x 'pelement pseudo-elements recurse))
  
  (define (id? x recurse) 
    (if (list? x)
	(and (eq? (car x) 'id)
	     (case (length x)
	       ((2) (cadr x))
	       ((3) (and (if recurse (selector? (cadr x) #t) #t)
			 (symbol? (caddr x)) 
			 (caddr x)))
	       (else #f)))
	(let* ((y (symbol->string x))
	       (ididx (string-index y #\x007c))
	       (endidx (and ididx (string-index y #\x007c (+ ididx 1)))))
	  (and ididx
	       endidx
	       (eqv? (string-ref y (+ ididx 1)) #\#)
	       (substring y (+ ididx 2) endidx)))))
  
  (define (attrib? x recurse)
    (if (list? x)
	(and (= (length x) 3) 
	     (eq? (car x) 'attrib) 
	     (if recurse (selector? (cadr x) #t) #t) 
	     (caddr x))
	(let* ((str (symbol->string x))
	       (start (string-index str #\[))
	       (end (string-index str #\] (if start start 0))))
	  (and start end (parse-attrib-str (substring str start (+ end 1)))))))
  
  (define (and-wrapper x y) (and x y))
  (define (selector-recurse? x) (selector? x #t))
  (define (contextual? x recurse)
    (and (list? x) 
	 (eq? (car x) '//)
	 (if recurse 
	     (fold-left and-wrapper #t (map selector-recurse? (cdr x)))
	     #t)))
  (define (simple? x recurse)
    (and (symbol? x)
	 (let* ((str (symbol->string x))
		(len (string-length str))
		(end (min (or (string-index str #\.) len)
			  (or (string-index str #\:) len)
			  (or (string-index str #\x007c) len)
			  (or (string-index str #\[) len))))
	   (if (eqv? end 0) "*" (substring str 0 end)))))
  
  (define (grouping? x recurse)
    (and (list? x) 
	 (not (memq (car x) 
		    (list '// '+ '> 'class 'pclass 'pelement 'attrib 'id 
			  (string->symbol "@import"))))
	 (if recurse (fold-left and-wrapper #t (map selector-recurse? x)) #t)
	 x))
  
  (define (combinator? x recurse)
    (and (list? x) (let ((y (car x)))
		     (and (memq y '(+ >))
			  (= (length x) 3)
			  (or (not recurse)
			      (and (selector? (cadr x) #t) 
				   (selector? (caddr x) #t)))
			  y))))
  
  (define (css-path? x recurse)
    (or (combinator? x recurse) (contextual? x recurse)))
  
  (define (selector? x recurse)
    (or (and (list? x) (or (contextual? x recurse) (grouping? x recurse)))
	(class? x recurse)
	(pseudo-class? x recurse)
	(pseudo-element? x recurse)
	(id? x recurse)
	(attrib? x recurse)
	(combinator? x recurse)
	(simple? x recurse)))
  
  (define (simple-selector-match? selection-context sel node)
    (define di (scss:selection-context-document-interface selection-context))
    (define ri (scss:selection-context-rendering-interface selection-context))
    
    (define (f preds tests c)
      (or (and (null? preds) (cons #t c))
	  (let* ((x ((car preds) sel #f))
		 (y (if x ((car tests) x) #t)))	  
	    (and y
		 (f (cdr preds) 
		    (cdr tests)
		    (or c (not (and x (eq? (car preds) pseudo-element?)))))))))
    
    (define (imp? x y) (and (list? x) (eq? (car x) (string->symbol "@import"))))
    
    (define (test-imp x) #f)
    (define (test-simple x) (or (equal? x "*") (equal? x (scss:name di node))))
    (define (test-class x) (equal? (scss:attr di node "class") x))
    (define (test-pseudo-class x) (scss:pseudo-class? ri node x))
    (define (test-pseudo-element x) (scss:pseudo-element? ri node x))
    (define (test-id x) (equal? (scss:id di node) x))
    (define (test-attr x)
      (if (list? x)
	  (let ((y (scss:attr di node (symbol->string (cadr x)))))
	    (and y (case (car x)
		     (('~=) (member (caddr x) (string-tokenize y)))
		     (((string->symbol "|=")) 
		      (member (caddr x) (string-tokenize y char-set:letter)))
		     (else (equal? y (caddr x))))))
	  (scss:attr di node (symbol->string x))))
    
    (f (list imp? simple? class? pseudo-class? pseudo-element? id? attrib?)
       (list test-imp
	     test-simple
	     test-class
	     test-pseudo-class
	     test-pseudo-element
	     test-id
	     test-attr)
       #f))
  
  (define (combinator-selector-match? selection-context sel node combinator)
    (define document-interface 
      (scss:selection-context-document-interface selection-context))
    (if (eq? combinator '+)
	(let ((sib (scss:prev-sibling document-interface node)))
	  (and sib
	       (simple-selector-match? selection-context (cadr sel) sib)
	       (simple-selector-match? selection-context (caddr sel) node)))
	(and (simple-selector-match? selection-context (cadr sel) node)
	     (simple-selector-match? selection-context (caddr sel) node))))
  
  (define (path-selector-match? selection-context sel node)
    (define di (scss:selection-context-document-interface selection-context))
    (define (ftail s x)
      (define (f y) (equal? (scss:name di y) s))
      (let loop ((n x)) (and n (or (and (f n) n) (loop (scss:parent di n))))))
    (let* ((csel (car sel))
	   (combinator (combinator? csel #f))
	   (r (if combinator
		  (combinator-selector-match? 
		   selection-context csel combinator node)
		  (simple-selector-match? selection-context csel node))))
      (if (null? (cdr sel)) 
	  r
	  (let* ((s (simple? (cadr sel) #f))
		 (p (scss:parent di node))
		 (t (or (and (equal? s "*") p) (ftail s p)))
		 (u (and t 
			 (path-selector-match? selection-context (cdr sel) t))))
	    (and r u)))))
  
  (define (selector-match? selection-context sel n)
    (let ((x (combinator? sel #f)))
      (if x 
	  (combinator-selector-match? selection-context sel n x)
	  (if (contextual? sel #f)
	      (let* ((rsel (reverse (cdr sel))))
		(path-selector-match? selection-context rsel n))
	      (simple-selector-match? selection-context sel n)))))
  
  (define (calc-precedence match)
    (case (scss:selected-value-source match)
      ((agent) 1)
      ((user) (if (scss:selected-value-important? match) 5 2))
      (else (if (scss:selected-value-important? match) 4 3))))
  
  (define (calc-specificity sel)
    (define (count pred lst) (length (filter pred lst)))
    (define (id-pred x) (id? x #f))
    (define (alt-pred-1 x) 
      (or (pseudo-class? x #f) (attrib? x #f) (class? x #f)))
    (define (alt-pred-2 x) (or (pseudo-element? x #f) (simple? x #f)))
    (let* ((lsel (if (contextual? sel #f) (cdr sel) (list sel))))
      (+ (* (count id-pred lsel) 100)
	 (* (count alt-pred-1 lsel) 10)
	 (count alt-pred-2 lsel))))
  
  (define (verify-arg-types name predicates args startnum)
    (if (not (= (length predicates) (length args)))
	(error #t "predicate argument mismatch"))
    (letrec ((f (lambda (l1 l2 i)
		  (or (null? l1) 
		      (if (apply (car l1) (list (car l2)))
			  (f (cdr l1) (cdr l2) (+ i 1))
			  (error #t (string-append 
				     name ": wrong argument type in position "
				     (number->string (+ startnum 1)))))))))
      (f predicates args 0)))
  
  (define (parse port)
    (let* ((lexer (make-css21-lexer port))
	   (parser (make-css21-parser lexer)))
      (parser)))
  
  (define (scss:css->scss port . baseurl)
    (define (dirstr str) 
      (let ((s (string-index-right str #\/)))
	(if s (substring str 0 (+ s 1)) "")))
    (define (import-fn s i)
      (if (and (not (null? s)) (not (null? (cdr s))))
	  (if (eq? (caar s) (string->symbol "@import"))
	      (import-fn (cdr s) (cons (cadar s) i)) 
	      i)
	  i))
    
    (verify-arg-types "scss:css->scss" (list port?) (list port) 1)
    
    (if (not (null? baseurl)) 
	(verify-arg-types "scss:css->scss" (list string?) baseurl 1))
    
    (let* ((bu (if (null? baseurl) "/" (car baseurl)))
	   (parsed-sheet (parse port))
	   (imports (import-fn (cdr parsed-sheet) (list)))
	   (clean-sheet (list-tail (cdr parsed-sheet) (length imports)))
	   (final (fold-right 
		   (lambda (s1 s2)
		     (let* ((s1 (if (string-prefix? "file://" s1)
				    (substring s1 7)
				    s1))
			    (bu (if (string-prefix? "file://" bu)
				    (substring bu 7)
				    bu))
			    (nbu (dirstr (if (string-prefix? "/" s1)
					     s1
					     (string-append bu "/" s1))))
			    (p #f)
			    (n #f))
		       (if n (append (cdr n) s2) s2)))
		   clean-sheet
		   imports)))
      (cons 'css final)))
  
  (define (scss:scss->css stylesheet port)
    (let* 
	((ser-selector 
	  (lambda (selector)
	    (display 
	     (fold-left (lambda (x y) 
			  (string-append 
			   (fold-left (lambda (q r)
					(string-append 
					 q (if (or (ident? r)
						   (equal? r ">")
						   (equal? r "+")
						   (equal? r "*"))
					       " " "") r))
				      ""
				      (reverse x))
			   (if (null? y) "" (string-append ", " y))))
			'()
			(reverse selector))
	     port)))
	 (ser-property (lambda (pair) 
			 (display "\t" port)
			 (display (car pair) port)
			 (display ": " port)
			 (display (cadr pair) port)
			 (display ";" port)
			 (newline port)))
	 (ser-block (lambda (block)
		      (ser-selector (car block))
		      (display " {" port)
		      (newline port)
		      (for-each (lambda (x) (ser-property x)) (cadr block))
		      (display "}" port)
		      (newline port)
		      (newline port))))			  
      (verify-arg-types "scss:scss->css"
			(list scss:stylesheet? port?) 
			(list stylesheet port)
			1)
      (for-each (lambda (block) (ser-block block)) stylesheet)))
  
  (define (scss:stylesheet? stylesheet)
    (and (list? stylesheet)
	 (for-all (lambda (x) 
		    (and (list? x)
			 (>= (length x) 2)
			 (list? (car x))
			 (list? (cdr x))
			 (for-all (lambda (y)
				    (and (list? y)
					 (or (= (length y) 2)
					     (and (= (length y) 3)
						  (eq? (list-ref y 2) '!)))
					 (string? (car y))
					 (string? (cadr y))))
				  (cdr x))))
		  stylesheet)))
  
  (define (scss:color->hex color)
    (verify-arg-types "scss:color->hex" (list string?) (list color) 1)
    (let* ((dc (string-downcase color))
	   (c (find (lambda (x) (equal? (car x) dc)) color-table)))
      (if c (cadr c) (list))))
  
  (define (scss:specify sprop val) (specify (symbol->string sprop) val))
  
  (define (select-values selection-context selector-context)
    (define (less x y)
      (let ((px (calc-precedence x))
	    (py (calc-precedence y)))
	(if (not (eqv? px py))
	    (> px py)
	    (let ((sx (calc-specificity (scss:selected-value-selector x)))
		  (sy (calc-specificity (scss:selected-value-selector y))))
	      (if (not (eqv? sx sy))
		  (> sx sy)
		  (> (scss:selected-value-order x) 
		     (scss:selected-value-order y)))))))

    (define (product f lst1 lst2)
      (let loop ((lst1 lst1) (result '()))
	(if (null? lst1) result
	    (loop (cdr lst1) 
		  (append (let ((x (car lst1))) (map (lambda (y) (f x y)) lst2))
			  result)))))
    
    (define (find-matches source-symbol sheet)
      
      (define (filter-matches selectors counter)
	(let loop ((selectors selectors) (counter counter) (matches '()))
	  (if (null? selectors) 
	      (reverse matches)
	      (let ((selector (car selectors)))
		(if (selector-match? 
		     selection-context selector selector-context)
		    (loop (cdr selectors) 
			  (+ counter 1)
			  (cons (list selector source-symbol counter) matches))
		    (loop (cdr selectors) counter matches))))))
      
      (let loop ((sheet (if (not sheet) '() (cdr sheet)))
		 (counter 0)
		 (matches '()))
	(if (null? sheet)
	    (reverse matches)
	    (let* ((rule (car sheet))
		   (selectors (car rule))
		   (selectors (or (grouping? selectors #f) (list selectors)))
		   (selector-matches (filter-matches selectors counter)))
	      (loop (cdr sheet)
		    (+ counter (length selector-matches))
		    (append (product
			     (lambda (rule declaration)
			       (if (eq? (car declaration) '!)

				   (scss:make-selected-value
				    (car rule)
				    (caadr declaration)
				    (cadadr declaration)
				    #t
				    (cadr rule)
				    (caddr rule))

				   (scss:make-selected-value
				    (car rule)
				    (car declaration)
				    (cadr declaration)
				    #f
				    (cadr rule)
				    (caddr rule))))
			     
			     selector-matches (cdr rule))
			    matches))))))
    
    (let* ((cascade (scss:selection-context-cascade selection-context))
	   (agent-matches (find-matches 'agent (scss:cascade-agent cascade)))
	   (user-matches (find-matches 'user (scss:cascade-user cascade)))
	   (author-matches 
	    (find-matches 'author (scss:cascade-author cascade))))
      (list-sort less 
		 (append '() author-matches user-matches agent-matches))))
  
  (define (scss:select-values context selector)
    (select-values context selector))
  
  (define (scss:get-default-value prop)
    (verify-arg-types "scss:get-default-value" (list symbol?) (list prop) 1)
    (get-default-prop-value prop))
  
  (define (scss:inherited? prop) 
    (let ((p (get-prop-entry prop))) (and p (css-property:inherited? p))))
 
)
