#include <glib.h>
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>

#include "../box.h"
#include "../load.h"
#include "../parse.h"
#include "../scheme.h"
#include "../window.h"
#include "util.h"

void test_generation_simple_none (box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  g_assert_cmpint (0, ==, g_list_length (boxes));
}

void test_generation_simple_block (box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  ruin_box_t *box = NULL;

  g_assert_cmpint (1, ==, g_list_length (boxes));

  box = (ruin_box_t *) boxes->data;
  g_assert (box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);
  g_assert (box->generator == f->node);
}

void test_generation_simple_inline (box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  ruin_box_t *box = NULL;

  g_assert_cmpint (1, ==, g_list_length (boxes));

  box = (ruin_box_t *) boxes->data;
  g_assert (box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (box->children));
  box = (ruin_box_t *) box->children->data;
  g_assert (box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
}

void test_generation_complex_mixed_block_inline 
(box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  ruin_box_t *inline_box = NULL;
  ruin_box_t *block_box = NULL;

  g_assert_cmpint (1, ==, g_list_length (boxes));
  
  block_box = (ruin_box_t *) boxes->data;
  g_assert (block_box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);
  boxes = block_box->children;
  g_assert_cmpint (2, ==, g_list_length (boxes));
  block_box = (ruin_box_t *) boxes->data;
  g_assert (block_box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);
  inline_box = (ruin_box_t *) boxes->next->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK);  
  
  g_assert_cmpint (1, ==, g_list_length (inline_box->children));
  inline_box = (ruin_box_t *) inline_box->children->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (inline_box->children));
  inline_box = (ruin_box_t *) inline_box->children->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
}

void test_generation_complex_mixed_inline_block
(box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  ruin_box_t *inline_box = NULL;
  ruin_box_t *block_box = NULL;

  g_assert_cmpint (1, ==, g_list_length (boxes));
  
  block_box = (ruin_box_t *) boxes->data;
  g_assert (block_box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);
  boxes = block_box->children;
  g_assert_cmpint (2, ==, g_list_length (boxes));
  inline_box = (ruin_box_t *) boxes->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK);  
  block_box = (ruin_box_t *) boxes->next->data;
  g_assert (block_box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);
  
  g_assert_cmpint (1, ==, g_list_length (inline_box->children));
  inline_box = (ruin_box_t *) inline_box->children->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (inline_box->children));
  inline_box = (ruin_box_t *) inline_box->children->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
}

void test_generation_complex_nested_inline
(box_generation_fixture *f, gconstpointer d)
{
}

void test_generation_complex_nested_inline_block
(box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  ruin_box_t *inline_outer_box_1 = NULL;
  ruin_box_t *inline_outer_box_2 = NULL;
  ruin_box_t *block_box = NULL;
  ruin_box_t *line_box = NULL;
  ruin_box_t *inline_box = NULL;

  g_assert_cmpint (3, ==, g_list_length (boxes));
  
  inline_outer_box_1 = (ruin_box_t *) boxes->data;
  g_assert (inline_outer_box_1->type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK);  
  g_assert_cmpint (1, ==, g_list_length (inline_outer_box_1->children));
  line_box = (ruin_box_t *) inline_outer_box_1->children->data;
  g_assert (line_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (line_box->children));
  inline_box = (ruin_box_t *) line_box->children->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);

  block_box = (ruin_box_t *) boxes->next->data;
  g_assert (block_box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);

  inline_outer_box_2 = (ruin_box_t *) boxes->next->next->data;
  g_assert (inline_outer_box_2->type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK);  
  g_assert_cmpint (1, ==, g_list_length (inline_outer_box_2->children));
  line_box = (ruin_box_t *) inline_outer_box_2->children->data;
  g_assert (line_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (line_box->children));
  inline_box = (ruin_box_t *) line_box->children->data;
  g_assert (inline_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
}

void test_generation_simple_list_item_inside 
(box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  ruin_box_t *box = NULL;

  g_assert_cmpint (1, ==, g_list_length (boxes));

  box = (ruin_box_t *) boxes->data;
  g_assert (box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);
  g_assert (box->generator == f->node);
  g_assert_cmpint (1, ==, g_list_length (box->children));

  ruin_box_t *line_box = (ruin_box_t *) box->children->data;
  g_assert (line_box->type = RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (line_box->children));

  ruin_box_t *marker_box = (ruin_box_t *) box->children->data;
  g_assert (line_box->type = RUIN_LAYOUT_BOX_TYPE_MARKER);
  g_assert_cmpint (1, ==, g_list_length (marker_box->children));
}

void test_generation_simple_list_item_outside 
(box_generation_fixture *f, gconstpointer d)
{
  GList *boxes = ruin_box_generate (f->window, f->node);
  ruin_box_t *marker_box = NULL;
  ruin_box_t *principal_box = NULL;

  g_assert_cmpint (2, ==, g_list_length (boxes));

  marker_box = (ruin_box_t *) boxes->data;
  g_assert (marker_box->type == RUIN_LAYOUT_BOX_TYPE_MARKER);
  g_assert (marker_box->generator == f->node);
  g_assert_cmpint (0, ==, g_list_length (marker_box->children));

  principal_box = (ruin_box_t *) boxes->next->data;
  g_assert (principal_box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK);
  g_assert (principal_box->generator == f->node);
  g_assert_cmpint (0, ==, g_list_length (principal_box->children));
}

int main (int argc, char *argv[])
{
  int ret = 0;
  FILE *dev_null = fopen ("/dev/null", "w+");

  GHashTable *style_block = g_hash_table_new (g_str_hash, g_str_equal);
  GHashTable *style_inline = g_hash_table_new (g_str_hash, g_str_equal);
  GHashTable *style_none = g_hash_table_new (g_str_hash, g_str_equal);

  GList *simple_block = NULL;
  GList *simple_inline = NULL;
  GList *simple_none = NULL;
  GList *complex_block_inline_block = NULL;
  GList *complex_block_block_inline = NULL;
  GList *complex_nested_inline_block = NULL;

  g_test_init (&argc, &argv, NULL);

  newterm (NULL, dev_null, stdin);
  scm_init_guile ();
  ruin_init ();

  g_hash_table_insert (style_block, "display", "block");
  g_hash_table_insert (style_inline, "display", "inline");
  g_hash_table_insert (style_none, "display", "none");

  simple_block = g_list_append 
    (NULL, box_generation_element_fixture_component_new (style_block, NULL));
  simple_inline = g_list_append 
    (NULL, box_generation_element_fixture_component_new (style_inline, NULL));
  simple_none = g_list_append 
    (NULL, box_generation_element_fixture_component_new (style_none, NULL));

  { box_generation_fixture_component_t *root_block =
      (box_generation_fixture_component_t *)
      box_generation_element_fixture_component_new (style_block, NULL);
    complex_block_inline_block = g_list_append 
      (NULL, root_block);
    complex_block_inline_block = g_list_append
      (complex_block_inline_block, box_generation_element_fixture_component_new
       (style_inline, root_block));
    complex_block_inline_block = g_list_append
      (complex_block_inline_block, box_generation_element_fixture_component_new
       (style_block, root_block));
  }

  { box_generation_fixture_component_t *root_block = 
      (box_generation_fixture_component_t *)
      box_generation_element_fixture_component_new (style_block, NULL);
    complex_block_block_inline = g_list_append 
      (NULL, root_block);
    complex_block_block_inline = g_list_append
      (complex_block_block_inline, box_generation_element_fixture_component_new
       (style_block, root_block));
    complex_block_block_inline = g_list_append
      (complex_block_block_inline, box_generation_element_fixture_component_new
       (style_inline, root_block));
  }

  { box_generation_fixture_component_t *root_inline =
      (box_generation_fixture_component_t *)
      box_generation_element_fixture_component_new (style_inline, NULL);
    complex_nested_inline_block = g_list_append
      (NULL, root_inline);
    complex_nested_inline_block = g_list_append
      (complex_nested_inline_block, box_generation_element_fixture_component_new
       (style_inline, root_inline));
    complex_nested_inline_block = g_list_append
      (complex_nested_inline_block, box_generation_element_fixture_component_new
       (style_block, root_inline));
    complex_nested_inline_block = g_list_append
      (complex_nested_inline_block, box_generation_element_fixture_component_new
       (style_inline, root_inline));
  }

  g_test_add 
    ("/box/generation/simple/block", box_generation_fixture, simple_block, 
     setup_box_generation_fixture, test_generation_simple_block, 
     teardown_box_generation_fixture);
  g_test_add 
    ("/box/generation/simple/inline", box_generation_fixture, simple_inline,
     setup_box_generation_fixture, test_generation_simple_inline, 
     teardown_box_generation_fixture);
  g_test_add
    ("/box/generation/simple/none", box_generation_fixture, simple_none,
     setup_box_generation_fixture, test_generation_simple_none,
     teardown_box_generation_fixture);

  { 
    GHashTable *inside_style = g_hash_table_new (g_str_hash, g_str_equal);
    GHashTable *outside_style = g_hash_table_new (g_str_hash, g_str_equal);
    GList *inside_components = NULL;
    GList *outside_components = NULL;

    g_hash_table_insert (inside_style, "display", "list-item");
    g_hash_table_insert (inside_style, "list-style-position", "inside");

    g_hash_table_insert (outside_style, "display", "list-item");
    g_hash_table_insert (outside_style, "list-style-position", "outside");

    inside_components = g_list_append 
      (NULL, box_generation_element_fixture_component_new (inside_style, NULL));
    outside_components = g_list_append
      (NULL, box_generation_element_fixture_component_new 
       (outside_style, NULL));

    g_test_add
      ("/box/generation/simple/list-item/inside", box_generation_fixture,
       inside_components, setup_box_generation_fixture,
       test_generation_simple_list_item_inside,
       teardown_box_generation_fixture);
    g_test_add
      ("/box/generation/simple/list-item/outside", box_generation_fixture,
       outside_components, setup_box_generation_fixture,
       test_generation_simple_list_item_outside,
       teardown_box_generation_fixture);
  }  

  g_test_add
    ("/box/generation/complex/mixed/inline,block", box_generation_fixture,
     complex_block_inline_block, setup_box_generation_fixture,
     test_generation_complex_mixed_inline_block,
     teardown_box_generation_fixture);

  g_test_add
    ("/box/generation/complex/mixed/block,inline", box_generation_fixture,
     complex_block_block_inline, setup_box_generation_fixture,
     test_generation_complex_mixed_block_inline,
     teardown_box_generation_fixture);     

  g_test_add
    ("/box/generation/complex/nested/inline,block,inline", 
     box_generation_fixture, complex_nested_inline_block, 
     setup_box_generation_fixture, test_generation_complex_nested_inline_block,
     teardown_box_generation_fixture);

  ret = g_test_run ();
  
  endwin ();
  fclose (dev_null);

  return ret;
}
