#include <glib.h>
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>

#include "../box.h"
#include "../render.h"
#include "../scheme.h"
#include "../window.h"
#include "util.h"

void test_render_simple_text (box_fixture *f, gconstpointer d)
{
  chtype *chbuffer = calloc (5, sizeof (chtype));

  ruin_render_render_tree (f->root_box);

  wmove (f->root_box->window->window, 0, 0);
  g_assert_cmpint (4, ==, winchnstr (f->root_box->window->window, chbuffer, 4));
  g_assert_cmpint ('T', ==, chbuffer[0] & A_CHARTEXT);
  g_assert_cmpint ('e', ==, chbuffer[1] & A_CHARTEXT);
  g_assert_cmpint ('s', ==, chbuffer[2] & A_CHARTEXT);
  g_assert_cmpint ('t', ==, chbuffer[3] & A_CHARTEXT);
}

void test_render_style_color_simple (box_fixture *f, gconstpointer d)
{
  chtype ch = 0;
  short fg = 0, bg = 0;
  
  ruin_render_render_tree (f->root_box);
  
  wmove (f->root_box->window->window, 0, 0);
  ch = winch (f->root_box->window->window);

  pair_content (PAIR_NUMBER (ch & A_COLOR), &fg, &bg);

  g_assert_cmpint ('C', ==, ch & A_CHARTEXT);
  g_assert_cmpint (COLOR_RED, ==, fg);
}

void test_render_style_background_color_simple 
(box_fixture *f, gconstpointer d)
{
  chtype ch = 0;
  short fg = 0, bg = 0;
  
  ruin_render_render_tree (f->root_box);
  
  wmove (f->root_box->window->window, 0, 0);
  ch = winch (f->root_box->window->window);

  pair_content (PAIR_NUMBER (ch & A_COLOR), &fg, &bg);

  g_assert_cmpint ('B', ==, ch & A_CHARTEXT);
  g_assert_cmpint (COLOR_GREEN, ==, bg);
}

void test_render_style_background_color_transparent
(box_fixture *f, gconstpointer d)
{
  chtype ch = 0;
  short fg = 0, bg = 0;

  ruin_render_render_tree (f->root_box);
  
  wmove (f->root_box->window->window, 0, 0);
  ch = winch (f->root_box->window->window);

  pair_content (PAIR_NUMBER (ch & A_COLOR), &fg, &bg);

  g_assert_cmpint ('B', ==, ch & A_CHARTEXT);
  g_assert_cmpint (COLOR_GREEN, ==, bg);  
}

void test_render_style_font_variant_small_caps
(box_fixture *f, gconstpointer d)
{
  chtype *chbuffer = calloc (11, sizeof (chtype));

  ruin_render_render_tree (f->root_box);

  wmove (f->root_box->window->window, 0, 0);
  g_assert_cmpint (3, ==, winchnstr (f->root_box->window->window, chbuffer, 3));

  g_assert_cmpint ('F', ==, chbuffer[0] & A_CHARTEXT);
  g_assert_cmpint ('O', ==, chbuffer[1] & A_CHARTEXT);
  g_assert_cmpint ('O', ==, chbuffer[2] & A_CHARTEXT);
}

void test_render_style_letter_spacing_simple 
(box_fixture *f, gconstpointer d)
{
  chtype *chbuffer = calloc (11, sizeof (chtype));

  ruin_render_render_tree (f->root_box);

  wmove (f->root_box->window->window, 0, 0);
  g_assert_cmpint 
    (10, ==, winchnstr (f->root_box->window->window, chbuffer, 10));

  g_assert_cmpint ('T', ==, chbuffer[0] & A_CHARTEXT);
  g_assert_cmpint ('e', ==, chbuffer[3] & A_CHARTEXT);
  g_assert_cmpint ('s', ==, chbuffer[6] & A_CHARTEXT);
  g_assert_cmpint ('t', ==, chbuffer[9] & A_CHARTEXT);  
}

void test_render_style_word_spacing_simple (box_fixture *f, gconstpointer d)
{
  chtype *chbuffer = calloc (12, sizeof (chtype));

  ruin_render_render_tree (f->root_box);

  wmove (f->root_box->window->window, 0, 0);
  g_assert_cmpint 
    (11, ==, winchnstr (f->root_box->window->window, chbuffer, 11));

  g_assert_cmpint ('T', ==, chbuffer[0] & A_CHARTEXT);
  g_assert_cmpint ('e', ==, chbuffer[1] & A_CHARTEXT);
  g_assert_cmpint ('s', ==, chbuffer[2] & A_CHARTEXT);
  g_assert_cmpint ('t', ==, chbuffer[3] & A_CHARTEXT);  

  g_assert_cmpint ('T', ==, chbuffer[7] & A_CHARTEXT);
  g_assert_cmpint ('e', ==, chbuffer[8] & A_CHARTEXT);
  g_assert_cmpint ('s', ==, chbuffer[9] & A_CHARTEXT);
  g_assert_cmpint ('t', ==, chbuffer[10] & A_CHARTEXT);  
}

int main (int argc, char *argv[])
{
  int ret = 0;
  FILE *dev_null = fopen ("/dev/null", "w+");
  SCREEN *screen = NULL;
  
  render_fixture_component_t *line = render_fixture_component_new
    (g_hash_table_new (g_str_hash, g_str_equal), NULL, NULL);

  line->layout->type = RUIN_LAYOUT_BOX_TYPE_LINE;
  line->height = 1;
  line->width = 20;
    
  g_test_init (&argc, &argv, NULL);

  screen = newterm (NULL, dev_null, stdin);
  set_term (screen);
  start_color ();

  scm_init_guile ();
  ruin_init ();

  { render_fixture_component_t *c = render_fixture_component_new
      (g_hash_table_new (g_str_hash, g_str_equal), line, NULL);
    GList *render_components = g_list_append (g_list_append (NULL, line), c);

    c->layout->type = RUIN_LAYOUT_BOX_TYPE_INLINE;
    c->layout->content_ptr = "Test";
    c->layout->parent = line->layout;
    c->height = 1;
    c->width = 4;

    g_test_add ("/render/simple/text", box_fixture, render_components, 
		setup_render_fixture, test_render_simple_text, 
		teardown_box_fixture);
  }

  { GHashTable *background_color_style = 
      g_hash_table_new (g_str_hash, g_str_equal);
    render_fixture_component_t *c = render_fixture_component_new
      (background_color_style, line, NULL);
    GList *render_components = g_list_append (g_list_append (NULL, line), c);

    g_hash_table_insert (background_color_style, "background-color", "green");

    c->layout->type = RUIN_LAYOUT_BOX_TYPE_INLINE;
    c->layout->content_ptr = "B";
    c->layout->parent = line->layout;
    c->height = 1;
    c->width = 1;

    g_test_add ("/render/style/background-color/simple", 
		box_fixture, render_components, setup_render_fixture, 
		test_render_style_background_color_simple, 
		teardown_box_fixture);
  }

  { GHashTable *opaque_background_color_style = 
      g_hash_table_new (g_str_hash, g_str_equal);
    GHashTable *transparent_background_color_style =
      g_hash_table_new (g_str_hash, g_str_equal);    
    render_fixture_component_t *b = render_fixture_component_new
      (opaque_background_color_style, NULL, NULL);
    render_fixture_component_t *l = render_fixture_component_new
      (g_hash_table_new (g_str_hash, g_str_equal), b, b);
    render_fixture_component_t *c = render_fixture_component_new
      (transparent_background_color_style, l, b);
    GList *render_components = 
      g_list_append (g_list_append (g_list_append (NULL, b), l), c);

    g_hash_table_insert 
      (opaque_background_color_style, "background-color", "green");
    g_hash_table_insert
      (transparent_background_color_style, "background-color", "transparent");

    b->layout->type = RUIN_LAYOUT_BOX_TYPE_BLOCK;
    b->height = 1;
    b->width = 1;

    l->layout->type = RUIN_LAYOUT_BOX_TYPE_LINE;
    l->layout->parent = b->layout;
    l->layout->containing_block = b->layout;
    l->height = 1;
    l->width = 1;

    c->layout->type = RUIN_LAYOUT_BOX_TYPE_INLINE;
    c->layout->content_ptr = "B";
    c->layout->parent = l->layout;
    c->layout->containing_block = b->layout;
    c->height = 1;
    c->width = 1;

    g_test_add ("/render/style/background-color/transparent", 
		box_fixture, render_components, setup_render_fixture, 
		test_render_style_background_color_transparent, 
		teardown_box_fixture);
  }

  { GHashTable *font_variant_style = g_hash_table_new (g_str_hash, g_str_equal);
    render_fixture_component_t *c = render_fixture_component_new
      (font_variant_style, line, NULL);
    GList *render_components = g_list_append (g_list_append (NULL, line), c);

    g_hash_table_insert (font_variant_style, "font-variant", "small-caps");

    c->layout->type = RUIN_LAYOUT_BOX_TYPE_INLINE;
    c->layout->content_ptr = "fOo";
    c->layout->parent = line->layout;
    c->height = 1;
    c->width = 10;

    g_test_add ("/render/style/font-variant/small-caps", 
		box_fixture, render_components, setup_render_fixture, 
		test_render_style_font_variant_small_caps, 
		teardown_box_fixture);
  }

  { GHashTable *color_style = g_hash_table_new (g_str_hash, g_str_equal);
    render_fixture_component_t *c = render_fixture_component_new
      (color_style, line, NULL);
    GList *render_components = g_list_append (g_list_append (NULL, line), c);

    g_hash_table_insert (color_style, "color", "red");

    c->layout->type = RUIN_LAYOUT_BOX_TYPE_INLINE;
    c->layout->content_ptr = "C";
    c->layout->parent = line->layout;
    c->height = 1;
    c->width = 1;

    g_test_add ("/render/style/color/simple", 
		box_fixture, render_components, setup_render_fixture, 
		test_render_style_color_simple, teardown_box_fixture);
  }

  { render_fixture_component_t *c = render_fixture_component_new
      (g_hash_table_new (g_str_hash, g_str_equal), line, NULL);
    GList *render_components = g_list_append 
      (g_list_append (NULL, line), c);

    c->layout->type = RUIN_LAYOUT_BOX_TYPE_INLINE;
    c->layout->content_ptr = "Test";
    c->layout->parent = line->layout;
    c->letter_spacing = 2;
    c->height = 1;
    c->width = 10;

    g_test_add ("/render/style/letter-spacing/simple", 
		box_fixture, render_components, setup_render_fixture, 
		test_render_style_letter_spacing_simple, 
		teardown_box_fixture);
  }

  { render_fixture_component_t *c = render_fixture_component_new
      (g_hash_table_new (g_str_hash, g_str_equal), line, NULL);
    GList *render_components = g_list_append (g_list_append (NULL, line), c);

    c->layout->type = RUIN_LAYOUT_BOX_TYPE_INLINE;
    c->layout->content_ptr = "Test Test";
    c->layout->parent = line->layout;
    c->word_spacing = 2;
    c->height = 1;
    c->width = 11;

    g_test_add ("/render/style/word-spacing/simple", 
		box_fixture, render_components, setup_render_fixture, 
		test_render_style_word_spacing_simple, 
		teardown_box_fixture);
  }

  ret = g_test_run ();

  endwin ();
  fclose (dev_null);

  return ret;
}
