/* util.c: Utility functions for libRUIN test suite
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <assert.h>
#include <glib.h>
#include <libguile.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "../box.h"
#include "../css.h"
#include "../parse.h"
#include "../scheme.h"
#include "../window.h"
#include "util.h"

int counter = 1;

static ruin_box_t *make_layout_box_inner
(component_fixture_generation_context *context, 
 box_layout_fixture_component_t *c, ruin_box_t *parent, 
 ruin_box_t *containing_block)
{
  ruin_box_t *b = NULL;
  ruin_node_element_t *e = mock_element 
    (context->window, NULL, c->style);

  if (parent != NULL)
    ((ruin_node_t *) e)->parent = parent->generator;
  
  if (c->type == RUIN_LAYOUT_BOX_TYPE_INLINE)
    {
      if (c->content_ptr != NULL)
	{
	  ruin_node_text_t *t = mock_text_node 
	    (context->window, NULL, c->content_ptr);
	  ((ruin_node_t *) t)->parent = (ruin_node_t *) e;
	  b = (ruin_box_t *) ruin_inline_box_new
	    (c->type, parent, containing_block, context->window, 
	     (ruin_node_t *) t, c->content_ptr);
	}
      else b = (ruin_box_t *) ruin_inline_box_new 
	     (c->type, parent, containing_block, context->window, 
	      (ruin_node_t *) e, NULL);
    }
  else if (c->type == RUIN_LAYOUT_BOX_TYPE_LINE)
    b = (ruin_box_t *) ruin_box_new 
      (c->type, parent, containing_block, context->window, NULL);
  else if (c->type == RUIN_LAYOUT_BOX_TYPE_MARKER)
    {
      box_layout_marker_fixture_component_t *m = 
	(box_layout_marker_fixture_component_t *) c;
      b = (ruin_box_t *) ruin_marker_box_new
	(c->type, parent, containing_block, context->window, (ruin_node_t *) e,
	 m->style, m->ordinal, m->length);
    }
  else b = ruin_box_new 
	 (c->type, parent, containing_block, context->window, 
	  (ruin_node_t *) e);
  
  if (parent != NULL)
    parent->children = g_list_append (parent->children, b);	  

  return b;
}

static ruin_box_t *make_layout_box 
(component_fixture_generation_context *context, gconstpointer d)
{
  ruin_box_t *b = NULL;
  box_layout_fixture_component_t *c = (box_layout_fixture_component_t *) d;
  
  ruin_box_t *parent = NULL;
  ruin_box_t *containing_block = NULL;

  if (c->parent != NULL)
    parent = g_hash_table_lookup (context->component_box_map, c->parent);
  if (c->containing_block != NULL)
    containing_block = g_hash_table_lookup
      (context->component_box_map, c->containing_block);

  b = make_layout_box_inner (context, c, parent, containing_block);
  g_hash_table_insert (context->component_box_map, c, b);
  return b;
}

static void annotate_render_box 
(ruin_box_t *b, render_fixture_component_t *render_component)
{
  b->top = render_component->top;
  b->left = render_component->left;
  
  b->width.used = render_component->width;
  b->height.used = render_component->height;
  
  b->padding_top.used = render_component->padding_top;
  b->padding_left.used = render_component->padding_left;
  b->padding_bottom.used = render_component->padding_bottom;
  b->padding_right.used = render_component->padding_right;
  
  b->margin_top.used = render_component->margin_top;
  b->margin_left.used = render_component->margin_left;
  b->margin_bottom.used = render_component->margin_bottom;
  b->margin_right.used = render_component->margin_right;
  
  b->letter_spacing.used = render_component->letter_spacing;
  b->word_spacing.used = render_component->word_spacing;
}

static ruin_box_t *make_render_box 
(component_fixture_generation_context *context, gconstpointer d)
{
  ruin_box_t *b = NULL;
  render_fixture_component_t *c = (render_fixture_component_t *) d;

  ruin_box_t *parent = NULL;
  ruin_box_t *containing_block = NULL;

  if (c->layout->parent != NULL)
    parent = g_hash_table_lookup 
      (context->component_box_map, c->layout->parent);
  if (c->layout->containing_block != NULL)
    containing_block = g_hash_table_lookup
      (context->component_box_map, c->layout->containing_block);

  b = make_layout_box_inner (context, c->layout, parent, containing_block);
  g_hash_table_insert (context->component_box_map, c->layout, b);
  annotate_render_box (b, c);

  return b;
}

static ruin_box_t *make_root_layout_box 
(component_fixture_generation_context *context, gconstpointer d)
{
  box_layout_fixture_component_t *c = (box_layout_fixture_component_t *) d;
  ruin_box_t *b = ruin_box_new 
    (c->type, NULL, NULL, context->window, 
     (ruin_node_t *) (mock_root_element (context->window, c->style)));

  g_hash_table_insert (context->component_box_map, c, b);  

  return b;
}

static ruin_box_t *make_root_render_box 
(component_fixture_generation_context *context, gconstpointer d)
{
  render_fixture_component_t *c = (render_fixture_component_t *) d;
  ruin_box_t *b = make_root_layout_box (context, c->layout);

  annotate_render_box (b, c);
  g_hash_table_insert (context->component_box_map, c->layout, b);  

  return b;  
}

component_fixture_generation_strategy box_layout_fixture_generation_strategy = 
  {
    make_root_layout_box,
    make_layout_box
  };

component_fixture_generation_strategy render_fixture_generation_strategy = 
  {
    make_root_render_box,
    make_render_box
  };

void teardown_box_fixture (box_fixture *f, gconstpointer d)
{
  teardown_window_fixture (f->window, d);

  ruin_window_free (f->window);
  ruin_box_free (f->viewport_box);
  ruin_box_free (f->root_box);
}

static void setup_layout_fixture_inner 
(component_fixture_generation_context *context, box_fixture *f, 
 gconstpointer d)
{
  GList *components = (GList *) d;
  GList *component_ptr = components;

  while (component_ptr != NULL)
    {
      ruin_box_t *b = NULL;

      if (f->viewport_box == NULL)
	{
	  f->viewport_box = ruin_box_new 
	    (RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK, NULL, NULL, context->window, 
	     NULL);
	  f->viewport_box->height.used = context->window->height;
	  f->viewport_box->width.used = context->window->width;
	}

      if (f->root_box == NULL)
	{
	  b = context->strategy->root_box_constructor 
	    (context, component_ptr->data);
	  b->containing_block = f->viewport_box;
	  f->root_box = b;
	}
      else b = context->strategy->box_constructor 
	     (context, component_ptr->data);

      component_ptr = component_ptr->next;
    }

  f->window = context->window;
}

void setup_box_layout_fixture (box_fixture *f, gconstpointer d)
{
  ruin_window_t *w = ruin_window_new 
    (newwin (10, 20, 0, 0), fopen ("/dev/null", "w+"));
  setup_window_fixture (w, d);
 
  component_fixture_generation_context context;
  context.strategy = &box_layout_fixture_generation_strategy;
  context.component_box_map = g_hash_table_new (g_direct_hash, g_direct_equal);
  context.window = w;

  setup_layout_fixture_inner (&context, f, d);

  g_hash_table_destroy (context.component_box_map);
}

void setup_box_layout_fixture_single (box_fixture *f, gconstpointer d)
{
  GList *l = g_list_append (NULL, (gpointer) d);

  setup_box_layout_fixture (f, l);

  g_list_free (l);
}

void setup_render_fixture (box_fixture *f, gconstpointer d)
{
  ruin_window_t *w = ruin_window_new 
    (newwin (10, 20, 0, 0), fopen ("/dev/null", "w+"));
  setup_window_fixture (w, d);

  component_fixture_generation_context context;
  context.strategy = &render_fixture_generation_strategy;
  context.component_box_map = g_hash_table_new (g_direct_hash, g_direct_equal);
  context.window = w;

  setup_layout_fixture_inner (&context, f, d);

  g_hash_table_destroy (context.component_box_map);
}

void setup_render_fixture_single (box_fixture *f, gconstpointer d)
{
  GList *l = g_list_append (NULL, (gpointer) d);

  setup_render_fixture (f, l);

  g_list_free (l);  
}

void teardown_render_fixture (box_fixture *f, gconstpointer d)
{
  teardown_window_fixture (f->root_box->window, d);
}

void setup_window_fixture (ruin_window_t *w, gconstpointer d)
{  
  w->cascade = ruin_scheme_scss_make_cascade (w);
  w->selection_context = ruin_scheme_scss_make_selection_context
    (w, ruin_scheme_scss_document_interface_sdom,
     ruin_scheme_scss_make_rendering_interface (w, SCM_BOOL_F, SCM_BOOL_F), 
     w->cascade);
  
  getmaxyx (w->window, w->height, w->width);

  scm_gc_protect_object (w->cascade);
  scm_gc_protect_object (w->selection_context);
}

void teardown_window_fixture (ruin_window_t *w, gconstpointer d)
{
  delwin (w->window);
  fclose (w->log);
  scm_gc_unprotect_object (w->cascade);
  scm_gc_unprotect_object (w->selection_context);
}

static void add_style (gpointer key, gpointer value, gpointer user_data)
{
  ruin_window_t *w = (ruin_window_t *) ((void **) user_data)[0];
  char *elt_name = (char *) ((void **) user_data)[1];
  SCM author = ruin_scheme_scss_cascade_author (w, w->cascade);

  if (author == SCM_BOOL_F)
    author = SCM_EOL;

  ruin_css_add_named_style (&author, elt_name, (char *) key, (char *) value);
  ruin_scheme_scss_set_cascade_author (w, w->cascade, author);
}

void free_node_recursive (ruin_node_t *e)
{
  GList *children = e->children;
  while (children != NULL)
    {
      free_node_recursive ((ruin_node_t *) children->data);
      children = children->next;
    }

  ruin_node_free (e);  
}

ruin_node_element_t *mock_element (ruin_window_t *w, SCM doc, GHashTable *ht)
{
  int digit_chars = (int) floor ((log (counter) / log (10)) + 1);
  char *elt_name = calloc (5 + digit_chars, sizeof (char));
  void **user_data = calloc (2, sizeof (void *));
  SCM elt = SCM_EOL;

  snprintf (elt_name, 5 + digit_chars, "test%d", counter++);

  elt = ruin_scheme_sdom_create_element (w, doc, elt_name);
  scm_gc_protect_object (elt);
  
  user_data[0] = w;
  user_data[1] = elt_name;

  g_hash_table_foreach (ht, add_style, user_data);

  free (user_data);
  free (elt_name);

  return ruin_node_element_new (doc, elt, NULL);
}

ruin_node_text_t *mock_text_node (ruin_window_t *w, SCM doc, char *content)
{
  SCM elt = ruin_scheme_sdom_create_text_node (w, doc, content);
  ruin_node_text_t *t = ruin_node_text_new (doc, elt, NULL);
  t->content = content;
  return t;
}

ruin_node_element_t *mock_root_element (ruin_window_t *w, GHashTable *ht)
{
  SCM doc = ruin_scheme_sdom_create_document 
    (w, "test", SCM_BOOL_F);
  scm_gc_protect_object (doc);
  return mock_element (w, doc, ht);
}

box_generation_element_fixture_component_t *
box_generation_element_fixture_component_new
(GHashTable *style, box_generation_fixture_component_t *parent)
{
  box_generation_element_fixture_component_t *bgefc = 
    calloc (1, sizeof (box_generation_element_fixture_component_t));
  box_generation_fixture_component_t *bgfc = 
    (box_generation_fixture_component_t *) bgefc;

  bgefc->style = style;
  bgfc->parent = parent;

  return bgefc;
}

box_generation_text_fixture_component_t *
box_generation_text_fixture_component_new
(char *content, box_generation_fixture_component_t *parent)
{
  box_generation_text_fixture_component_t *bgtfc = 
    calloc (1, sizeof (box_generation_text_fixture_component_t));
  box_generation_fixture_component_t *bgfc = 
    (box_generation_fixture_component_t *) bgtfc;

  bgtfc->content = content;
  bgfc->parent = parent;

  return bgtfc;
}

box_layout_fixture_component_t *box_layout_fixture_component_new 
(GHashTable *style, box_layout_fixture_component_t *parent, 
 box_layout_fixture_component_t *containing_block)
{
  box_layout_fixture_component_t *c = 
    calloc (1, sizeof (box_layout_fixture_component_t));  

  c->style = style;
  c->parent = parent;
  c->containing_block = containing_block;

  return c;
}

render_fixture_component_t *render_fixture_component_new 
(GHashTable *style, render_fixture_component_t *parent, 
 render_fixture_component_t *containing_block)
{
  render_fixture_component_t *c = calloc 
    (1, sizeof (render_fixture_component_t));
  c->layout = calloc (1, sizeof (box_layout_fixture_component_t));  

  c->layout->style = style;

  if (parent != NULL)
    c->layout->parent = parent->layout;
  if (containing_block != NULL)
    c->layout->containing_block = containing_block->layout;

  return c;
}

box_layout_marker_fixture_component_t *box_layout_marker_fixture_component_new
(GHashTable *style, box_layout_fixture_component_t *parent,
 box_layout_fixture_component_t *containing_block)
{
  box_layout_marker_fixture_component_t *m = calloc 
    (1, sizeof (box_layout_marker_fixture_component_t));
  box_layout_fixture_component_t *c = (box_layout_fixture_component_t *) m;

  c->style = style;
  c->parent = parent;
  c->containing_block = containing_block;

  return m;
}

render_fixture_component_t *marker_render_fixture_component_new
(GHashTable *style, render_fixture_component_t *parent,
 render_fixture_component_t *containing_block)
{
  render_fixture_component_t *m = calloc
    (1, sizeof (render_fixture_component_t));

  m->layout = (box_layout_fixture_component_t *) 
    calloc (1, sizeof (box_layout_marker_fixture_component_t));

  m->layout->style = style;

  if (parent != NULL)
    m->layout->parent = parent->layout;
  if (containing_block != NULL)
    m->layout->containing_block = containing_block->layout;

  return m;
}

void render_fixture_component_free (render_fixture_component_t *c)
{
  free (c);
}

void setup_box_generation_fixture (box_generation_fixture *f, gconstpointer d)
{
  GList *components = (GList *) d;
  GList *component_ptr = components;
  GHashTable *component_elements = g_hash_table_new 
    (g_direct_hash, g_direct_equal);

  f->window = ruin_window_new (newwin (1, 1, 0, 0), fopen ("/dev/null", "w+"));
  f->window->cascade = ruin_scheme_scss_make_cascade (f->window);
  f->window->selection_context = ruin_scheme_scss_make_selection_context 
    (f->window, ruin_scheme_scss_document_interface_sdom, 
     ruin_scheme_scss_make_rendering_interface 
     (f->window, SCM_BOOL_F, SCM_BOOL_F), 
     f->window->cascade);

  scm_gc_protect_object (f->window->cascade);
  scm_gc_protect_object (f->window->selection_context);

  while (component_ptr != NULL)
    {
      box_generation_fixture_component_t *c = 
	(box_generation_fixture_component_t *) component_ptr->data;

      box_generation_element_fixture_component_t *ec = NULL;
      box_generation_text_fixture_component_t *tc = NULL;

      ruin_node_t *e = NULL;

      switch (c->node_type)
	{
	case RUIN_PARSE_NODE_TYPE_ELEMENT:
	  ec = (box_generation_element_fixture_component_t *) c;
	  if (f->node == NULL)
	    {
	      e = (ruin_node_t *) mock_root_element (f->window, ec->style);
	      f->node = e;
	    }
	  else e = (ruin_node_t *) mock_element 
		 (f->window, f->node->doc, ec->style);
	  break;
	case RUIN_PARSE_NODE_TYPE_TEXT:
	  tc = (box_generation_text_fixture_component_t *) c;
	  e = (ruin_node_t *) mock_text_node 
	    (f->window, f->node->doc, tc->content);
	  break;
	default: assert (1 == 0);
	}
      
      g_hash_table_insert (component_elements, c, e);
      
      if (c->parent != NULL)
	{
	  ruin_node_t *parent_element = g_hash_table_lookup
	    (component_elements, c->parent);

	  GList *last_child = g_list_last (parent_element->children);
	  if (parent_element->first_child == NULL)
	    parent_element->first_child = e;
	  if (last_child != NULL)
	    ((ruin_node_t *) last_child->data)->next_sibling = e;

	  parent_element->children = g_list_append 
	    (parent_element->children, e);
	  e->parent = parent_element;
	}

      component_ptr = component_ptr->next;
    }

  g_hash_table_destroy (component_elements);
}

void teardown_box_generation_fixture 
(box_generation_fixture *f, gconstpointer d)
{
  delwin (f->window->window);
  fclose (f->window->log);
  scm_gc_unprotect_object (f->window->cascade);
  scm_gc_unprotect_object (f->window->selection_context);
  ruin_window_free (f->window);

  scm_gc_unprotect_object (f->node->node);
  free_node_recursive (f->node);
}
