/* text.c: Text analysis routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <ctype.h>
#include <glib.h>
#include <stdlib.h>
#include <string.h>

#include "box.h"
#include "css.h"
#include "tables.h"
#include "text.h"
#include "window.h"

typedef struct _content_pointer {
  char *content;

  int index;
  int length;
} content_pointer;

static ruin_layout_text_fragment *analyze_normal_first_letter
(content_pointer *content, int l_spacing)
{
  ruin_layout_text_fragment *fragment = 
    calloc (1, sizeof (ruin_layout_text_fragment));
  char c = content->content[content->index];
      
  if (isspace (c)) 
    while (content->index < content->length)
      {
	if (!isspace (content->content[content->index]))
	  {
	    c = content->content[content->index];	    
	    break;
	  }
	content->index++;
	fragment->char_width++;
      }

  if (content->index == content->length)
    {
      free (fragment);
      return NULL;
    }
  
  fragment->content_ptr = &content->content[content->index];
  while (TRUE)
    {
      fragment->char_width++;
      fragment->real_width++;
      if (++content->index == content->length)
	break;
      
      c = content->content[content->index];
      if (!ispunct (c))
	break;
      else fragment->real_width += l_spacing;
    }

  return fragment;
}

static GList *analyze_normal_line
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  int seen_space = FALSE;
  int total_real_width = 0;
  GList *fragments = NULL;

  ruin_layout_text_fragment *fragment = NULL;
   
  while (content->index < content->length && total_real_width < width)
    {
      char c = content->content[content->index];
      
      if (isspace (c)) 
	{
	  seen_space = TRUE;
	  
	  if (fragment != NULL)
	    fragments = g_list_append (fragments, fragment);
	    
	  fragment = calloc (1, sizeof (ruin_layout_text_fragment));      
	  fragment->content_ptr = &content->content[content->index];
		
	  while (content->index < content->length)
	    {
	      if (!isspace (content->content[content->index]))
		{
		  if (is_first)
		    is_first = FALSE;
		  else 
		    {
		      fragment->content_ptr = 
			&content->content[content->index - 1];
		      fragment->real_width = 1 + w_spacing;
		      total_real_width += 1 + w_spacing;
		    }
		  break;
		}
	      content->index++;
	      fragment->char_width++;
	    }
	}
      else 
	{
	  if (fragment == NULL)
	    fragment = calloc (1, sizeof (ruin_layout_text_fragment));
	  
	  content->index++;
	  fragment->char_width++;
	  fragment->real_width++;
	  total_real_width++;

	  if (seen_space)
	    seen_space = FALSE;
	  else if (content->index < content->length && total_real_width < width)
	    {
	      fragment->real_width += l_spacing;
	      total_real_width += l_spacing;
	    }
	}
    }

  if (total_real_width >= width)
    {
      assert (fragment != NULL);
      if (g_list_length (fragments) == 0)
	fragments = g_list_append (fragments, fragment);
      else
	{
	  content->index -= fragment->char_width;
	  free (fragment);
	}
    }
  else fragments = g_list_append (fragments, fragment);

  return fragments;  
}

static GList *analyze_normal_first_line
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  return analyze_normal_line 
    (content, l_spacing, w_spacing, width, is_first, is_last);
}

static GList *analyze_normal 
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  GList *fragments = NULL;
  while (content->index < content->length)
    {
      fragments = g_list_concat 
	(fragments, analyze_normal_line 
	 (content, l_spacing, w_spacing, width, is_first, is_last));

      is_first = TRUE;
    }

  return fragments;
}

static ruin_layout_text_fragment *analyze_nowrap_first_letter
(content_pointer *content, int l_spacing)
{
  assert (1 == 0);
  return NULL;
}

static GList *analyze_nowrap_first_line
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  assert (1 == 0);
  return NULL;
}

static GList *analyze_nowrap
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  assert (1 == 0);
  return NULL;
}

static ruin_layout_text_fragment *analyze_pre_first_letter
(content_pointer *content, int l_spacing)
{
  ruin_layout_text_fragment *fragment = 
    calloc (1, sizeof (ruin_layout_text_fragment));
  char c = content->content[content->index];

  fragment->content_ptr = &content->content[content->index];
      
  while (isspace (c) && content->index < content->length)
    {
      content->index++;
      fragment->real_width++;
      fragment->char_width++;
    }

  while (TRUE)
    {
      fragment->char_width++;
      fragment->real_width++;
      if (++content->index == content->length)
	break;
      
      c = content->content[content->index];
      if (!ispunct (c))
	break;
      else fragment->real_width += l_spacing;
    }

  return fragment;
}

static GList *_analyze_pre_line
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  int last_was_space = FALSE;
  ruin_layout_text_fragment *fragment = 
    calloc (1, sizeof (ruin_layout_text_fragment));
  
  fragment->content_ptr = &content->content[content->index];

  while (content->index < content->length)
    {
      fragment->real_width++;

      if (isspace (content->content[content->index++]))
	{
	  if (content->content[content->index] == '\n')
	    break;
	  if (!last_was_space)
	    fragment->real_width += w_spacing;
	  last_was_space = TRUE;
	}
      else 
	{
	  if (last_was_space) last_was_space = FALSE;
	  else if (content->index < content->length) 
	    fragment->real_width += l_spacing;
	}
    }
  
  if (content->index < content->length)
    fragment->break_required = TRUE;

  return g_list_append (NULL, fragment);
}

static GList *analyze_pre_first_line
(content_pointer *content, int l_spacing, int w_spacing, int width,
 int is_first, int is_last)
{
  return _analyze_pre_line 
    (content, l_spacing, w_spacing, width, is_first, is_last);
}

static GList *analyze_pre
(content_pointer *content, int l_spacing, int w_spacing, int width,
 int is_first, int is_last)
{
  GList *fragments = NULL;
  while (content->index < content->length)
    {
      fragments = g_list_concat 
	(fragments, _analyze_pre_line 
	 (content, l_spacing, w_spacing, width, is_first, is_last));

      is_first = TRUE;
    }

  return fragments;
}

static ruin_layout_text_fragment *analyze_pre_wrap_first_letter
(content_pointer *content, int l_spacing)
{
  assert (1 == 0);
  return NULL;
}

static GList *analyze_pre_wrap_first_line
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  assert (1 == 0);
  return NULL;
}

static GList *analyze_pre_wrap
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  assert (1 == 0);
  return NULL;
}

static ruin_layout_text_fragment *analyze_pre_line_first_letter
(content_pointer *content, int l_spacing)
{
  assert (1 == 0);
  return NULL;
}

static GList *analyze_pre_line_first_line
(content_pointer *content, int l_spacing, int w_spacing, int width, 
 int is_first, int is_last)
{
  assert (1 == 0);
  return NULL;
}

static GList *analyze_pre_line
(content_pointer *content, int l_spacing, int w_spacing, int width,
 int is_first, int is_last)
{
  assert (1 == 0);
  return NULL;
}

ruin_layout_text_analysis *ruin_layout_text_analyze 
(ruin_inline_box_t *ib, ruin_layout_text_context *context)
{
  ruin_box_t *box = (ruin_box_t *) ib;
  int extent_offset = 0;
  ruin_layout_text_line_width_constraint *constraint = NULL;
  GList *constraint_ptr = context->width_constraints;

  ruin_layout_text_analysis *analysis = 
    calloc (1, sizeof (ruin_layout_text_analysis));
  
  ruin_window_render_state_t *old_render_state = box->window->render_state;
  ruin_window_render_state_t new_render_state;
  content_pointer content;

  content.content = ib->content_ptr;  
  content.index = 0;
  content.length = strlen (ib->content_ptr);  
  
  new_render_state.dialect = old_render_state->dialect;
  new_render_state.active_pseudo_elements = 0;
  box->window->render_state = &new_render_state;

  if (context->first_letter)
    {
      char *white_space = NULL;
      ruin_length_t l_spacing;
      ruin_layout_text_fragment *(*analyzer)(content_pointer *, int) = NULL;

      constraint = (ruin_layout_text_line_width_constraint *) 
	constraint_ptr->data;
      constraint_ptr = constraint_ptr->next;

      ruin_css_activate_pseudo_element 
	(box->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);
      ruin_css_activate_pseudo_element 
	(box->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);

      white_space = ruin_box_css_lookup (box, "white-space");
      l_spacing = ruin_css_parse_spacing 
	(ruin_box_css_lookup (box, "letter-spacing"));
      ruin_css_normalize_length 
	(&l_spacing, &box->containing_block->width, box->window->font_width, 
	 box->window->dpi, TRUE);  

      ruin_css_deactivate_pseudo_element 
	(box->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);
      
      if (strcmp (white_space, "normal") == 0)
	analyzer = analyze_normal_first_letter;
      else if (strcmp (white_space, "nowrap") == 0)
	analyzer = analyze_nowrap_first_letter;
      else if (strcmp (white_space, "pre") == 0)
	analyzer = analyze_pre_first_letter;
      else if (strcmp (white_space, "pre-line") == 0)
	analyzer = analyze_pre_line_first_letter;
      else if (strcmp (white_space, "pre-wrap") == 0)
	analyzer = analyze_pre_wrap_first_letter;
      else assert (1 == 0);
      
      analysis->fragments = g_list_append 
	(analysis->fragments, analyzer (&content, l_spacing.used));
      
      context->first_letter = FALSE;
    }
  
  if (context->first_line)
    {
      char *white_space = NULL;
      ruin_length_t l_spacing, w_spacing;
      GList *(*analyzer)(content_pointer *, int, int, int, int, int) = NULL;

      if (constraint == NULL)
	{
	  constraint = (ruin_layout_text_line_width_constraint *)
	    constraint_ptr->data;
	  constraint_ptr = constraint_ptr->next;
	}
	  
      ruin_css_activate_pseudo_element 
	(box->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);

      white_space = ruin_box_css_lookup (box, "white-space");
      l_spacing = ruin_css_parse_spacing
	(ruin_box_css_lookup (box, "letter-spacing"));
      w_spacing = ruin_css_parse_spacing
	(ruin_box_css_lookup (box, "word-spacing"));

      ruin_css_normalize_length 
	(&l_spacing, &box->containing_block->width, box->window->font_width, 
	 box->window->dpi, TRUE);  
      ruin_css_normalize_length 
	(&w_spacing, &box->containing_block->width, box->window->font_width, 
	 box->window->dpi, TRUE);  

      ruin_css_deactivate_pseudo_element 
	(box->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);

      if (strcmp (white_space, "normal") == 0)
	analyzer = analyze_normal_first_line;
      else if (strcmp (white_space, "nowrap") == 0)
	analyzer = analyze_nowrap_first_line;
      else if (strcmp (white_space, "pre") == 0)
	analyzer = analyze_pre_first_line;
      else if (strcmp (white_space, "pre-line") == 0)
	analyzer = analyze_pre_line_first_line;
      else if (strcmp (white_space, "pre-wrap") == 0)
	analyzer = analyze_pre_wrap_first_line;
      else assert (1 == 0);
      
      analysis->fragments = g_list_concat 
	(analysis->fragments, analyzer 
	 (&content, l_spacing.used, w_spacing.used, constraint->width, 
	  context->first_on_line, context->last_in_region));

      if (content.index < content.length)
	{
	  context->first_line = FALSE;
	  extent_offset++;
	}
    }

  if (content.index < content.length)
    {
      char *white_space = NULL;
      ruin_length_t l_spacing, w_spacing;
      GList *(*analyzer)(content_pointer *, int, int, int, int, int) = NULL;

      white_space = ruin_box_css_lookup (box, "white-space");
      l_spacing = ruin_css_parse_spacing 
	(ruin_box_css_lookup (box, "letter-spacing"));
      w_spacing = ruin_css_parse_spacing
	(ruin_box_css_lookup (box, "word-spacing"));
      ruin_css_normalize_length 
	(&l_spacing, &box->containing_block->width, box->window->font_width, 
	 box->window->dpi, TRUE);  
      ruin_css_normalize_length 
	(&w_spacing, &box->containing_block->width, box->window->font_width, 
	 box->window->dpi, TRUE);  

      if (strcmp (white_space, "normal") == 0)
	analyzer = analyze_normal;
      else if (strcmp (white_space, "nowrap") == 0)
	analyzer = analyze_nowrap;
      else if (strcmp (white_space, "pre") == 0)
	analyzer = analyze_pre;
      else if (strcmp (white_space, "pre-line") == 0)
	analyzer = analyze_pre_line;
      else if (strcmp (white_space, "pre-wrap") == 0)
	analyzer = analyze_pre_wrap;
      else assert (1 == 0);

      while (content.index < content.length)
	{
	  if (constraint == NULL || extent_offset >= constraint->extent)
	    {
	      constraint = (ruin_layout_text_line_width_constraint *)
		constraint_ptr->data;
	      constraint_ptr = constraint_ptr->next;
	      extent_offset = 0;
	    }

	  analysis->fragments = g_list_concat 
	    (analysis->fragments, analyzer 
	     (&content, l_spacing.used, w_spacing.used, constraint->width,
	      context->first_on_line, context->last_in_region));
	  extent_offset++;
	}
    }
  
  box->window->render_state = old_render_state;      

  return analysis;
}

void ruin_layout_text_analysis_free (ruin_layout_text_analysis *a, int deep)
{
  if (deep)
    g_list_free_full (a->fragments, free);
  free (a);
}
