/* window.c: Window-management routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#ifdef LINUX
#include <pty.h>
#endif /* LINUX */
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>

#include "layout.h"
#include "load.h"
#include "parse.h"
#include "scheme.h"
#include "util.h"
#include "window.h"

extern void ruin_box_free (ruin_box_t *);
extern ruin_windows_t *_ruin_windows;
static pthread_mutex_t ruin_window_signal_handler_SIGWINCH_entry_lock;

static void resize_window (gpointer key, gpointer value, gpointer user_data)
{
  ruin_window_t *window = (ruin_window_t *) value;
  int *newxy = (int *) user_data;
  int oldx = 0, oldy = 0, newx = 0, newy = 0;

  ruin_box_t *root_box = (ruin_box_t *) g_list_nth_data (window->root_boxes, 0);
  ruin_node_t *generator = root_box->generator;
  GList *root_box_ptr = window->root_boxes;

  oldx = window->width;
  oldy = window->height;
  
  newx = newxy[0];
  newy = newxy[1];
  
  if (oldx != newx || oldy != newy) 
    {
      window->width = newx;
      window->height = newy;
      
      resizeterm (newy, newx); 
      ruin_load_layout_and_render (window, generator);

      while (root_box_ptr != NULL)
	{
	  ruin_box_t *old_box = (ruin_box_t *) root_box_ptr->data;
	  ruin_box_free (old_box);
	  root_box_ptr = root_box_ptr->next;
	}
    }
}

void ruin_window_signal_handler_SIGWINCH (int n, siginfo_t *s, void *u) 
{
  static int busy = 0;
  struct winsize ws;
  int newxy[2];

  pthread_mutex_lock (&ruin_window_signal_handler_SIGWINCH_entry_lock);
  if (busy) 
    {
      pthread_mutex_unlock (&ruin_window_signal_handler_SIGWINCH_entry_lock);
      return;
    }
  pthread_mutex_unlock (&ruin_window_signal_handler_SIGWINCH_entry_lock);
  busy = TRUE;
  
  (void) ioctl (0, TIOCGWINSZ, &ws);
  
  newxy[0] = ws.ws_col;
  newxy[1] = ws.ws_row;

  g_hash_table_foreach (_ruin_windows->windows, resize_window, newxy);

  busy = FALSE;
}

ruin_windows_t *ruin_windows_new () 
{
  char *xul = NULL;
  char *xhtml = NULL;

  char *env_val = getenv ("RUIN_CSS_PATH");

  SCM port = SCM_EOL;

  ruin_windows_t *t = calloc (1, sizeof (ruin_windows_t));
  t->windows = g_hash_table_new (g_str_hash, g_str_equal);

  if (env_val == NULL)
    env_val = RUIN_CSS_PATH;

  xul = calloc (strlen (env_val) + 16, sizeof(char));
  xhtml = calloc (strlen (env_val) + 18, sizeof(char));
  strcat (xul, "file://");
  strcat (xul, env_val);
  strcat (xul, "/xul.css");
  strcat (xhtml, "file://");
  strcat (xhtml, env_val);
  strcat (xhtml, "/xhtml.css");

  port = scm_open_file
    (scm_from_locale_string (xul + 7), scm_from_locale_string ("r"));
  t->xul_agent_css = ruin_scheme_scss_css_to_scss_port (NULL, port, xul);

  port = scm_open_file
    (scm_from_locale_string (xhtml + 7), scm_from_locale_string ("r"));
  t->xhtml_agent_css = ruin_scheme_scss_css_to_scss_port (NULL, port, xhtml);

  free (xul);
  free (xhtml);

  pthread_mutex_init (&ruin_window_signal_handler_SIGWINCH_entry_lock, NULL);

  return t;
}

void ruin_windows_free (ruin_windows_t *t) 
{
  free (t->windows);
  scm_gc_unprotect_object (t->xul_agent_css);
  scm_gc_unprotect_object (t->xhtml_agent_css);
  free (t);
}

ruin_window_t *ruin_window_new (WINDOW *w, FILE *f) 
{
  ruin_window_t *t = calloc (1, sizeof (ruin_window_t));
  char *tmp = NULL, *tmp_ptr = NULL;
  int tmp_val = 0;

  t->window = w;
  t->log = f;
  t->render_state = calloc (1, sizeof (ruin_window_render_state_t));

  t->internal_id = ruin_util_generate_id ();
  g_hash_table_insert 
    (_ruin_windows->windows, ruin_util_long_to_string (t->internal_id), t);
  
  t->ids = ruin_util_hash_new ();
  t->internal_ids = ruin_util_hash_new ();

  if ((tmp = getenv (RUIN_WINDOW_FONT_HEIGHT_ENV)) &&
      ((tmp_val = strtol (tmp, &tmp_ptr, 10)) > 0) &&
      (tmp != tmp_ptr))
    t->font_height = tmp_val;
  else t->font_height = RUIN_WINDOW_DEFAULT_FONT_HEIGHT;

  if ((tmp = getenv (RUIN_WINDOW_FONT_WIDTH_ENV)) &&
      ((tmp_val = strtol (tmp, &tmp_ptr, 10)) > 0) &&
      (tmp != tmp_ptr))
    t->font_width = tmp_val;
  else t->font_width = RUIN_WINDOW_DEFAULT_FONT_WIDTH;

  if ((tmp = getenv (RUIN_WINDOW_DPI_ENV)) &&
      ((tmp_val = strtol (tmp, &tmp_ptr, 10)) > 0) &&
      (tmp != tmp_ptr))
    t->dpi = tmp_val;
  else t->dpi = RUIN_WINDOW_DEFAULT_DPI;

  getmaxyx (w, t->height, t->width);

  if (_ruin_windows->current_window == NULL)
    _ruin_windows->current_window = t;

  return t;
}

void ruin_window_free (ruin_window_t *t) 
{
  g_hash_table_remove
    (_ruin_windows->windows, ruin_util_long_to_string (t->internal_id));

  ruin_window_clear (t);
  ruin_util_hash_free (t->ids);
  ruin_util_hash_free (t->internal_ids);

  free (t->render_state);
  free (t);
  return;
}

void ruin_window_clear (ruin_window_t *w) 
{
  ruin_util_hash_clear (w->ids);
  ruin_util_hash_clear (w->internal_ids);
  return;
}

ruin_node_t *ruin_window_lookup_scm (SCM elt) 
{
  ruin_window_t *containing_win = ruin_window_lookup_window (elt);
  if (containing_win != NULL) 
    {
      SCM res = SCM_EOL; 
      if (scm_eq_p (res, SCM_EOL) == SCM_BOOL_T)
	ruin_util_log (containing_win, "found containing window but not node");
      return (ruin_node_t *) 
	ruin_util_string_to_ptr (scm_to_locale_string (res));	
    }
  return NULL;
}

ruin_window_t *ruin_window_lookup_window (SCM elt) 
{
  return NULL;
}

ruin_window_t *ruin_get_current_window ()
{
  return ruin_window_get_current_window ();
}

ruin_window_t *ruin_window_get_current_window ()
{
  return _ruin_windows->current_window;
}

void ruin_set_current_window (ruin_window_t *w)
{
  ruin_window_set_current_window (w);
  /* TODO: Render it! */
}

void ruin_window_set_current_window (ruin_window_t *w)
{
  _ruin_windows->current_window = w;
}
