/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

/*
 * This example show how to use libtermui console and builtin command
 * parsing feature over threaded telnet connections.
 */

#include <pthread.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include <termui/console.h>
#include <termui/command.h>

static COMMAND_EMPTY_ROOT(root);

static void *console_thread(void *soc_)
{
  struct console_ctx_s	*con;
  int	soc = (long)soc_;
  int	socb;

  if (!(con = console_alloc(&root, soc, soc, "xterm")))
    goto err;

  /* add telnet protocol support */
  console_telnet_setup(con);

  /* set console prompt */
  console_set_prompt(con, "[%32Aconsole%A] ");

  /* show welcome message */
  console_printf(con,
		 "\n"
		 "  Welcome to libtermui telnet example." "\n"
		 "  Copyright Alexandre Becoulet (c) 2006" "\n"
		 "\n"
		 "  You may type `%1Ahelp%A'\n"
		 "\n");

  /* process user commands */
  while (1)
    {
      int res = console_process(con);

      if (res == -EIO || res == -ECANCELED)
	break;
    }

  /* show bye message */
  console_printf(con, "Bye\n");

  close(soc);
  console_free(con);
 err:
  return NULL;
}

static void console_listen(int port)
{
  /* creat socket */
  int	soc = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  int	tmp = 1;

  struct sockaddr_in	addr = 
    {
      .sin_port = htons(port),
      .sin_family = AF_INET,
    };

  setsockopt(soc, SOL_SOCKET, SO_REUSEADDR, &tmp, sizeof(tmp));

  /* socket bind & listen */
  if (bind(soc, (struct sockaddr*)&addr, sizeof (struct sockaddr_in)) < 0)
    exit(1);

  if (listen(soc, 2) < 0)
    exit(2);

  printf("Listening for telnet connection on TCP port %u\n", port);

  while (1)
    {
      struct sockaddr_in	con_addr;
      int			addr_size = sizeof(con_addr);
      int			con;

      con = accept(soc, (struct sockaddr*)&con_addr, &addr_size);

      /* create new thread for each new connection */
      if (con > 0)
	{
	  struct hostent	*h;
	  pthread_t		t;
	  uint8_t		*addr = (uint8_t*)&con_addr.sin_addr;

	  if (pthread_create(&t, NULL, console_thread, (void*)con))
	    close(con);

	  pthread_detach(t);
	}
    }
}


static COMMAND_GROUP_DECL(builtin) =
{
  /* use builtin commands entris defined in libtermui */
  COMMAND_CMD_ENTRY (console_builtin_help, "help", COMMAND_ACL_ALL, "Display help about commands", NULL, 0, 1),
  COMMAND_CMD_ENTRY (console_builtin_list, "list", COMMAND_ACL_ALL, "List available commands", NULL, 0, 0),
  COMMAND_CMD_ENTRY (console_builtin_quit, "quit", COMMAND_ACL_ALL, "Leave console", NULL, 0, 0),

  COMMAND_LIST_END
};

int main()
{
  command_register_root(&root, con_group_builtin);
  console_listen(4567);
}

