/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "command_pv.h"
#include "console_pv.h"

/* 
 * Update command tree links and setup root. Old root is discarded
 */

static void
command_register_tree(struct command_entry_s **root,
		      struct command_entry_s *list)
{
  struct command_entry_s	*e;
  struct command_entry_s	**p = root;

  /* setup local linked list */

  for (e = list; e->flag; e++)
    {
      if (e->flag & COMMAND_FLAG_REGISTERED)
	continue;

      e->next = NULL;
      e->prev = p;
      if (p)
	*p = e;
      p = &e->next;

      e->flag |= COMMAND_FLAG_REGISTERED;

      if (e->flag & COMMAND_FLAG_ISDIR)
	command_register_tree(NULL, e->subdir);
    }

  if (p)
    *p = NULL;
}

/* 
 * Append a command tree at end of current root list.
 */

void
command_register_root(struct command_entry_s **root,
		     struct command_entry_s *list)
{
  struct command_entry_s	*i, **p = root;

  for (i = *root; i; i = i->next)
    p = &i->next;

  command_register_tree(p, list);
}

/* 
 * Unregister a single entry and return next
 */

static inline struct command_entry_s *
command_unregister_entry(struct command_entry_s *e)
{
  if (e->next)
    e->next->prev = e->prev;

  if (e->prev)
    *e->prev = e->next;

  return e->next;
}

/* 
 * Unregsiter a command tree
 */

void
command_unregister(struct command_entry_s *list)
{
  struct command_entry_s	*e;

  /* setup local linked list */

  for (e = list; e->flag; e++)
    {
      if (!(e->flag & COMMAND_FLAG_REGISTERED))
	continue;

      e->flag &= ~COMMAND_FLAG_REGISTERED;

      command_unregister_entry(e);

      if (e->flag & COMMAND_FLAG_ISDIR)
	command_unregister(e->subdir);
    }
}

static inline char *
command_cmdname_cmp(struct command_entry_s *e,
		   char *path)
{
  static const char console_iscmd[255] =
    {
      ['0'...'9'] = 1,
      ['a'...'z'] = 1,
      ['A'...'Z'] = 1,
      ['_'] = 1,
    };
  const char *name = e->cmd;

  while (console_iscmd[(unsigned)*path] && *name)
    if (*path++ != *name++)
      return NULL;

  return (!*name && !console_iscmd[(unsigned)*path]) ? path : NULL;
}

struct command_entry_s *
command_find_entry(int acl,
		   struct command_entry_s *root,
		   char **path_)
{
  char	*next, *path = *path_;

  while (root)
    {
      next = command_cmdname_cmp(root, path);

      if (next)
	{
	  if ((*next == '.')
	      && (root->flag & COMMAND_FLAG_ISDIR)
	      && (root->acl & acl))
	    {
	      root = root->subdir;
	      path = next + 1;
	      continue;
	    }

	  if (*next > ' ')
	    return NULL;

	  *path_ = next;
	  return root;
	}

      root = root->next;
    }

  return NULL;
}

/* 
 * Execute a command found in a tree (optional error output)
 */

int
command_execute(int acl, struct command_entry_s *root,
		char *line, struct console_ctx_s *con)
{
  int				argc;
  char				*argv[COMMAND_MAX_ARGS];
  struct command_entry_s	*e;
  int				res = 0;

  if (!(e = command_find_entry(acl, root, &line)))
    {
      console_printf(con, "error: Command not found.\n");
      return -ENOTSUP;
    }

  if (e->flag & COMMAND_FLAG_ISDIR)
    {
      console_printf(con, "error: `%1A%s%A' is a command group.\n", e->cmd);
      return -ENOTSUP;
    }

  if (!(e->acl & acl))
    {
      console_printf(con, "error: Access denied to `%1A%s%A' command.\n", e->cmd);
      return -EACCES;
    }

  argc = command_split_args(line, argv);

  if (e->args_desc)
    {
      void		*argctx = NULL;
      char		*paramv[COMMAND_MAX_ARGS];
      int		paramc;
      unsigned int	mask;

      /* allocate args parsing context */
      if (e->args_ctx_size)
	{
	  argctx = alloca(e->args_ctx_size);
	  memset(argctx, 0, e->args_ctx_size);
	}

      /* parse args using args descriptors */
      paramc = command_args_parse(acl, e, argctx, argc, argv, paramv, &mask, con);

      if (paramc >= 0)
	{
	  res = e->func(con, argctx, paramc, paramv, mask);

	  /* parsing context cleanup callback */
	}

      if (e->args_cleanup)
	e->args_cleanup(con, argctx, mask);
    }
  else
    {
      if (argc > e->args_max)
	{
	  console_printf(con, "error: Too many arguments `%1A%s ...%A'\n", argv[argc - 1]);
	  return -EOVERFLOW;
	}

      if (argc < e->args_min)
	{
	  console_printf(con, "error: %u extra argument%s expected, found %u\n",
			 e->args_min, e->args_min > 1 ? "s" : "", argc);
	  return -EOVERFLOW;
	}

      /* pass unprocessed splited args if no descriptor found */
      res = e->func(con, NULL, argc, argv, 0);
    }

  return res;
}

/*
 * Execute all commands in text file
 */

int
command_execute_file(int acl, struct command_entry_s *root,
		     const char *file, struct console_ctx_s *con,
		     int err_abort, int verbose)
{
  char		buffer[CONSOLE_LINE_MAXLEN + 1];
  char		*line;
  unsigned int	n = 1;
  FILE		*in;

  /* open script file */
  if (!(in = fopen(file, "r")))
    {
      console_printf(con, "error: Unable to open `%s' file\n", file);
      return -ENOENT;
    }

  /* read line from file */
  while ((line = fgets(buffer, CONSOLE_LINE_MAXLEN, in)))
    {
      line[CONSOLE_LINE_MAXLEN] = '\0';

      /* skip blank line and comments */
      line += strspn(line, " \t\n");

      if (*line == '\0' || *line == '#')
	continue;

      if (verbose)
	{
	  char		*endl;

	  if ((endl = strrchr(line, '\n')))
	    *endl = 0;

	  console_printf(con, "+ %s\n", line);
	}

      /* execute command */
      if (command_execute(acl, root, line, con) < 0)
	{
	  console_printf(con, "error: Batch command execution failed at `%s:%u'\n", file, n);

	  /* abort on error */
	  if (err_abort)
	    return -ECANCELED;
	}

      n++;
    }

  fclose(in);

  return 0;
}

