/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#include <string.h>
#include <ctype.h>

#include "console_pv.h"
#include "command_pv.h"

static inline int
console_match_len(const char *a, const char *b)
{
  unsigned int	i = 0;

  while (a[i] && (a[i] == b[i]))
    i++;

  return i;
}

struct command_entry_s *
console_complete_cmd(struct term_behavior_s *bhv,
		    struct console_ctx_s *con,
		    const char **start_,
		    const char *end)
{
  const char		*start = *start_;
  struct command_entry_s	*e = *con->root;
  struct command_entry_s	*candidate[CONSOLE_MAX_CANDIDATES];
  unsigned int		can_count = 0;
  unsigned int		can_len = 0;

  start += strspn(start, " \t"); /* skip blank */

  while (e && can_count < CONSOLE_MAX_CANDIDATES)
    {
      unsigned int	len;

      if (e->flag & COMMAND_FLAG_HIDDEN)
	goto next;

      if (!(e->acl & con->acl))
	goto next;

      /* get match lenght between line and command name */
      len = console_match_len(e->cmd, start);

      if ((e->cmd[len] == '\0') && (start + len + 1 <= end))
	{
	  /* if full name match for group, jump to sub group list */
	  if ((start[len] == '.') && (e->flag & COMMAND_FLAG_ISDIR))
	    {
	      e = e->subdir;
	      start += len + 1;
	      can_count = 0;
	      can_len = 0;
	      continue;
	    }

	  /* if full name match for command, end command completion */
	  if ((start[len] <= ' ') && (e->flag & COMMAND_FLAG_ISCMD))
	    {
	      *start_ = start + len;
	      return e;
	    }
	}

      /* not a candidate if match length do not reach cursor position */
      if (len < end - start)
	goto next;

      /* trim match length if cursor is before */
      if (len > end - start)
	len = end - start;

      /* check for new candidate with longer match */
      if (len > can_len)
	{
	  can_count = 0;
	  can_len = len;
	}

      /* add candidates with same match length */
      if (len == can_len)
	candidate[can_count++] = e;

    next:
      e = e->next;
    }

  switch (can_count)
    {
      /* no candidate */
    case (0):
      term_beep(con->tm);
      return NULL;

      /* one candidate */
    case (1): {
      struct command_entry_s	*c0 = candidate[0];
      const char		*sep = c0->flag & COMMAND_FLAG_ISDIR ? "." : " ";

      /* complete candidate name if needed */
      if (c0->cmd[can_len])
	{
	  getline_insert(bhv, c0->cmd + can_len, strlen(c0->cmd + can_len));
	  getline_insert(bhv, sep, 1);

	  return NULL;
	}

      /* insert separator if a end of token */
      if (start + can_len == end)
	{
	  getline_insert(bhv, sep, 1);
	}

      return NULL;
    }

      /* more than one candidates */
    default: {
      const char	*c0 = candidate[0]->cmd;
      unsigned int	len, i, max = 0;

      /* get common max match length of candidates */
      for (i = 1; i < can_count; i++)
	{
	  len = console_match_len(candidate[i]->cmd, c0);
	  max = len > max ? len : max;
	}

      /* complete common prefix */
      if (can_len < max)
	{
	  char		cmd[COMMAND_CMD_MAXLEN + 1];

	  strncpy(cmd, c0, COMMAND_CMD_MAXLEN);
	  cmd[max] = '\0';

	  getline_insert(bhv, cmd + can_len, strlen(cmd + can_len));

	  return NULL;
	}

      /* no completion possible, beep & display candidates list */

      term_beep(con->tm);

      console_printf(con, "\n");

      for (len = i = 0; i < can_count; i++)
	{
	  len += console_printf(con, "%s ", candidate[i]->cmd);

	  /* wrap to 80 columns */
	  if (len >= 80 - COMMAND_CMD_MAXLEN)
	    {
	      console_printf(con, "\n");
	      len -= 80 - COMMAND_CMD_MAXLEN;
	    }
	}

      console_printf(con, "\n");

      getline_reprompt(bhv);

      return NULL;
    }
    }
}

static void
console_args_list(struct console_ctx_s *con,
		  const struct command_entry_s *e)
{
  struct command_args_s	*desc = e->args_desc;
  unsigned int	i, len;

  for (len = i = 0; desc[i].str_short; i++)
    {
      if ((1 << i) & e->args_disabled)
	continue;		/* skip entry disabled options */

      if (!(desc[i].acl & con->acl))
	continue;

      len += console_printf(con, "%s %s ",
			    desc[i].str_short,
			    desc[i].str_long);

      if (len >= 80)
	{
	  console_printf(con, "\n");
	  len -= 80;
	}
    }

  console_printf(con, "\n");
}

GETLINE_FCN_COMPLETE(console_complete)
{
  struct console_ctx_s	*con = private;
  struct command_entry_s	*e;
  const char		*start, *cursor;

  start = getline_line_start(bhv);
  cursor = getline_line_cursor(bhv);

  if ((e = console_complete_cmd(bhv, con, &start, cursor)))
    {
      if (e->args_desc)
	{
	  console_printf(con, "\n");

	  console_args_list(con, e);

	  getline_reprompt(bhv);
	}
    }
}

