/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#ifndef TERM_UI_COMMAND_H_
#define TERM_UI_COMMAND_H_

struct				command_args_s;
struct				console_ctx_s;
struct				command_entry_s;

#define COMMAND_CMD_TYPE(f)		int f(struct console_ctx_s *con, void *argctx, \
					int argc, char *argv[], unsigned int mask)

#define COMMAND_ARGS_TYPE(f)		int f(struct console_ctx_s *con, \
					struct command_args_s *argdesc, void *argctx, \
					int argc, char *argv[])

#define COMMAND_ARGS_PROCESS_TYPE(f)	void f(struct console_ctx_s *con, \
					void *argctx, unsigned int mask)

#define COMMAND_CMD_MAXLEN	23

typedef COMMAND_CMD_TYPE(command_cmd_t);
typedef COMMAND_ARGS_TYPE(command_args_t);
typedef COMMAND_ARGS_PROCESS_TYPE(command_args_process_t);

#define COMMAND_ACL_ALL		0xffffffff
#define COMMAND_ACL_NONE	0x00000000

#define COMMAND_FLAG_ISCMD	0x00000001
#define COMMAND_FLAG_ISDIR	0x00000002
#define COMMAND_FLAG_HIDDEN	0x00000004
#define COMMAND_FLAG_REGISTERED	0x00000008

struct				command_args_s
{
  const char			*str_short; /* short option name */
  const char			*str_long; /* long option name */
  const char			*desc;
  int			exclude; /* exclude mask */
  int			depend; /* depend mask */
  int			acl; /* access control mask */
  command_args_t		*func;
};

struct				command_entry_s
{
  /* user initialized fields */
  const char			cmd[COMMAND_CMD_MAXLEN + 1];	/* command name */
  const char			*desc;	/* command short description */
  const char			*longdesc;
  int			flag;
  int			acl;	/* authorized groups mask */

  command_cmd_t			*func;

  /* mangement data */
  struct command_entry_s	*subdir;

  struct command_entry_s	*next;
  struct command_entry_s	**prev;

  /* command arguments descriptors */
  unsigned int			args_ctx_size; /* arg parse context size */
  struct command_args_s		*args_desc;
  command_args_process_t	*args_cleanup;
  unsigned int			args_min;	/* minimum args count */
  unsigned int			args_max;	/* maximum args count */
  unsigned int			args_mandatory;	/* mandatory options mask */
  unsigned int			args_disabled;	/* disables options mask */
};

#define COMMAND_ARGS_ENTRY(name, lname, sname, exclude_, depend_, acl_, desc_)	\
      {									\
	.str_short = sname,						\
	.str_long = lname,						\
	.exclude = exclude_,						\
	.depend = depend_,						\
	.acl = acl_,							\
	.func = con_args_##name,					\
	.desc = desc_							\
      }

#define COMMAND_CMD_ENTRY_(name, cname, acl_, desc_, longdesc_, minarg, maxarg)	\
	.cmd =	cname,							\
	.desc =	desc_,							\
	.longdesc = longdesc_,						\
	.flag =	COMMAND_FLAG_ISCMD,					\
	.acl =	acl_,							\
	.args_min = minarg,						\
	.args_max = maxarg,						\
	.func =	con_cmd_##name,

#define COMMAND_CMD_ENTRYA_(aname, amandatory, adisabled)		\
	.args_ctx_size = sizeof (struct con_args_ctx_##aname##_s),	\
	.args_desc = con_args_desc_##aname,				\
	.args_mandatory = amandatory,					\
	.args_disabled = adisabled,

#define COMMAND_CMD_ENTRYA(name, cname, acl_, desc_, longdesc_, minarg, maxarg, aname, amandatory, adisabled)	\
      {									\
	COMMAND_CMD_ENTRY_(name, cname, acl_, desc_, longdesc_, minarg, maxarg)	\
	COMMAND_CMD_ENTRYA_(aname, amandatory, adisabled)	\
	.args_cleanup = con_args_cleanup_##aname,			\
      }

#define COMMAND_CMD_ENTRY(name, cname, acl_, desc_, longdesc_, minarg, maxarg)	\
      {									\
	COMMAND_CMD_ENTRY_(name, cname, acl_, desc_, longdesc_, minarg, maxarg)	\
      }

#define COMMAND_GROUP_ENTRY(name, cname, acl_)					\
      {									\
	.cmd =	cname,							\
	.flag =	COMMAND_FLAG_ISDIR,					\
	.acl =	acl_,							\
	.subdir = con_group_##name,					\
      }

#define COMMAND_LIST_END	{ }

#define COMMAND_GROUP(name)		con_group_##name
#define COMMAND_CMD_FUNC(name)		COMMAND_CMD_TYPE(con_cmd_##name)
#define COMMAND_ARGS_DECL(name)		struct command_args_s con_args_desc_##name[]
#define COMMAND_GROUP_DECL(name)	struct command_entry_s con_group_##name[]
#define COMMAND_ARGS_CONTEXT(name)	struct con_args_ctx_##name##_s
#define COMMAND_ARGS_CLEANUP_FUNC(name)	COMMAND_ARGS_PROCESS_TYPE(con_args_cleanup_##name)
#define COMMAND_ARGS_PARSE_FUNC(name)	COMMAND_ARGS_TYPE(con_args_##name)
#define COMMAND_EMPTY_ROOT(name)	struct command_entry_s *name = 0


/* Regsiter a command tree */

void command_register_root(struct command_entry_s **root,
			   struct command_entry_s *list);

/* Unregsiter a command tree */

void command_unregister(struct command_entry_s *list);

/* Execute a command found in a tree (optional error output) */

int command_execute(int acl, struct command_entry_s *root,
		    char *line, struct console_ctx_s *con);

/* Execute all commands in text file */

int
command_execute_file(int acl, struct command_entry_s *root,
		     const char *file, struct console_ctx_s *con,
		     int err_abort, int verbose);

struct command_entry_s *
command_find_entry(int acl,
		   struct command_entry_s *root,
		   char **path_);

/*
 * console_builtin.c
 */

COMMAND_CMD_FUNC(console_builtin_help);
COMMAND_CMD_FUNC(console_builtin_list);
COMMAND_CMD_FUNC(console_builtin_batch);
COMMAND_CMD_FUNC(console_builtin_quit);

/* no action */

static inline COMMAND_ARGS_PARSE_FUNC(noaction)
{
  return 0;
}

#endif

