/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <gnome.h>
#include <eel/eel.h>
#include <translate.h>
#include "gt-util.h"
#include "gt-conf.h"
#include "gt-language-view.h"
#include "gt-app.h"
#include "gt-shell.h"
#include "gt-stock.h"
#include "gt-preferences.h"

#define MAX_HISTORY			1000
#define RESPONSE_PREFERENCES		1

typedef struct
{
  GtkWidget	*dialog;
  GtkWidget	*source_language_label;
  GtkWidget	*source_language_view;
  GtkWidget	*dest_language_label;
  GtkWidget	*dest_language_view;
  GtkWidget	*location_label;
  GtkWidget	*location_entry;
  GtkWidget	*progress_vbox;
  GtkWidget	*progressbar;
} Twp;

typedef struct
{
  TranslateSession		*session;
  char				*url;
  char				*from;
  char				*to;
  double			start_time;
  gboolean			cancelled;
} TranslateInfo;

static Twp twp = { NULL };
static gboolean twp_standalone = FALSE;
static TranslateInfo *current_translation = NULL;
static unsigned int stalled_timeout_id = 0;
static GtkWidget *error_dialog = NULL;

static void gt_twp_update_sensitivity (void);
static void gt_twp_location_history_notify_cb (GConfClient *client,
					       unsigned int cnxn_id,
					       GConfEntry *entry,
					       gpointer user_data);
static void gt_twp_set_location_history (GtkListStore *store,
					 GConfValue *history);
static void gt_twp_add_location_to_history (void);

static void gt_twp_translate (void);
static gpointer gt_twp_translate_thread (gpointer data);
static gboolean gt_twp_translate_progress_cb (double progress,
					      gpointer user_data);
static gboolean gt_twp_translate_stalled_cb (gpointer data);
static void gt_twp_stop_ui (void);

void
gt_twp_display (const char *url, gboolean standalone)
{
  GtkEntryCompletion *completion;
  GtkListStore *store;
  GConfValue *location_history;
  GList *icon_list;
  
  if (twp.dialog)
    {
      gtk_window_present(GTK_WINDOW(twp.dialog));
      return;
    }

  twp_standalone = standalone;

#define W(name) #name, &twp.name
  gt_create_interface("translate-web-page",
		      W(dialog),
		      W(source_language_label),
		      W(source_language_view),
		      W(dest_language_label),
		      W(dest_language_view),
		      W(location_label),
		      W(location_entry),
		      W(progress_vbox),
		      W(progressbar),
		      NULL);
#undef W

  gtk_dialog_add_button(GTK_DIALOG(twp.dialog), GTK_STOCK_HELP, GTK_RESPONSE_HELP);
  if (standalone)
    gtk_dialog_add_button(GTK_DIALOG(twp.dialog), GTK_STOCK_PREFERENCES, RESPONSE_PREFERENCES);
  gtk_dialog_add_button(GTK_DIALOG(twp.dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
  gtk_dialog_add_button(GTK_DIALOG(twp.dialog), GT_STOCK_TRANSLATE, GTK_RESPONSE_OK);
  
  eel_add_weak_pointer(&twp.dialog);

  icon_list = gt_window_icon_list_new("gnome-translate-web-page");
  gtk_window_set_icon_list(GTK_WINDOW(twp.dialog), icon_list);
  eel_g_object_list_free(icon_list);

  store = gtk_list_store_new(1, G_TYPE_STRING);

  completion = gtk_entry_completion_new();
  gtk_entry_completion_set_model(completion, GTK_TREE_MODEL(store));
  gtk_entry_completion_set_text_column(completion, 0);
  gtk_entry_set_completion(GTK_ENTRY(twp.location_entry), completion);
  g_object_unref(completion);

  location_history = eel_gconf_get_value(GT_CONF_TWP_LOCATION_HISTORY);
  if (location_history)
    {
      gt_twp_set_location_history(store, location_history);
      gconf_value_free(location_history);
    }
  
  g_object_unref(store);

  if (url)
    gtk_entry_set_text(GTK_ENTRY(twp.location_entry), url);
  else
    gtk_widget_grab_focus(twp.location_entry);
    
  gtk_entry_set_activates_default(GTK_ENTRY(twp.location_entry), TRUE);

  if (gt_app_window)
    gtk_window_set_transient_for(GTK_WINDOW(twp.dialog), gt_app_window);

  gtk_dialog_set_default_response(GTK_DIALOG(twp.dialog), GTK_RESPONSE_OK);

  gt_conf_notification_add(twp.dialog,
			   GT_CONF_TWP_LOCATION_HISTORY,
			   gt_twp_location_history_notify_cb,
			   NULL);
			   
  gt_language_view_setup_pair(GTK_TREE_VIEW(twp.source_language_view),
			      GT_CONF_TWP_SOURCE_LANGUAGE,
			      GTK_TREE_VIEW(twp.dest_language_view),
			      GT_CONF_TWP_DEST_LANGUAGE,
			      TRANSLATE_PAIR_WEB_PAGE,
			      gt_twp_update_sensitivity);

  gt_conf_link(twp.dialog, GT_CONF_TWP_WINDOW, NULL);

  gt_twp_update_sensitivity();
  gtk_widget_show(twp.dialog);
}

static void
gt_twp_update_sensitivity (void)
{
  const char *url = gtk_entry_get_text(GTK_ENTRY(twp.location_entry));
  gboolean controls_sensitive = current_translation == NULL;
  gboolean translate_sensitive = *url
    && ! current_translation
    && gt_language_view_has_selected(GTK_TREE_VIEW(twp.source_language_view))
    && gt_language_view_has_selected(GTK_TREE_VIEW(twp.dest_language_view));

  gtk_widget_set_sensitive(twp.source_language_label, controls_sensitive);
  gtk_widget_set_sensitive(twp.source_language_view, controls_sensitive);
  gtk_widget_set_sensitive(twp.dest_language_label, controls_sensitive);
  gtk_widget_set_sensitive(twp.dest_language_view, controls_sensitive);
  gtk_widget_set_sensitive(twp.location_label, controls_sensitive);
  gtk_widget_set_sensitive(twp.location_entry, controls_sensitive);

  gtk_dialog_set_response_sensitive(GTK_DIALOG(twp.dialog), GTK_RESPONSE_OK, translate_sensitive);
}

static void
gt_twp_location_history_notify_cb (GConfClient *client,
				   unsigned int cnxn_id,
				   GConfEntry *entry,
				   gpointer user_data)
{
  GConfValue *value = gconf_entry_get_value(entry);
  GtkListStore *store;

  GDK_THREADS_ENTER();

  store = GTK_LIST_STORE(gtk_entry_completion_get_model(gtk_entry_get_completion(GTK_ENTRY(twp.location_entry))));
  gtk_list_store_clear(store);

  if (value)
    gt_twp_set_location_history(store, value);

  GDK_THREADS_LEAVE();
}

static void
gt_twp_set_location_history (GtkListStore *store, GConfValue *history)
{
  GSList *list;
  GSList *l;
  int i = 0;

  g_return_if_fail(GTK_IS_LIST_STORE(store));
  g_return_if_fail(history != NULL);

  list = gconf_value_get_list(history);

  GT_LIST_FOREACH(l, list)
    {
      GConfValue *value = l->data;
      GtkTreeIter iter;

      if (i++ == MAX_HISTORY)
	break;

      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, 0, gconf_value_get_string(value), -1);
    }
}

static void
gt_twp_add_location_to_history (void)
{
  const char *url;

  url = gtk_entry_get_text(GTK_ENTRY(twp.location_entry));
  if (*url)
    {
      GSList *location_history;
      GSList *elem;

      location_history = eel_gconf_get_string_list(GT_CONF_TWP_LOCATION_HISTORY);
      elem = gt_g_utf8_slist_find(location_history, url);

      if (elem)
	{			/* bring existing entry to front */
	  location_history = g_slist_remove_link(location_history, elem);
	  location_history = g_slist_concat(elem, location_history);
	}
      else			/* prepend new entry */
	location_history = g_slist_prepend(location_history, g_strdup(url));
	
      if (g_slist_length(location_history) > MAX_HISTORY)
	{
	  GSList *elem;
	  
	  elem = g_slist_nth(location_history, MAX_HISTORY - 1);
	  g_return_if_fail(elem != NULL);
	  
	  eel_g_slist_free_deep(elem->next);
	  elem->next = NULL;
	}
      
      eel_gconf_set_string_list(GT_CONF_TWP_LOCATION_HISTORY, location_history);
      eel_g_slist_free_deep(location_history);
    }
}

static void
gt_twp_translate (void)
{
  TranslateInfo *info;

  gtk_widget_show(twp.progress_vbox);

  info = g_new(TranslateInfo, 1);
  info->session = g_object_ref(gt_shell_get_translate_session(gt_shell));
  info->url = g_strdup(gtk_entry_get_text(GTK_ENTRY(twp.location_entry)));
  info->from = gt_language_view_get_selected(GTK_TREE_VIEW(twp.source_language_view));
  info->to = gt_language_view_get_selected(GTK_TREE_VIEW(twp.dest_language_view));
  info->cancelled = FALSE;

  g_return_if_fail(info->from != NULL);
  g_return_if_fail(info->to != NULL);

  current_translation = info;

  gt_twp_update_sensitivity();
  gt_thread_create(GTK_WINDOW(twp.dialog), gt_twp_translate_thread, info);
}

static gpointer
gt_twp_translate_thread (gpointer data)
{
  TranslateInfo *info = data;
  char *translated_url;
  GError *err = NULL;
  
  info->start_time = gt_get_current_time();

  translated_url = translate_session_translate_web_page(info->session,
							info->url,
							info->from,
							info->to,
							gt_twp_translate_progress_cb,
							info,
							&err);

  GDK_THREADS_ENTER();

  if (! info->cancelled)
    {
      if (translated_url)
	{
	  if (! gnome_url_show(translated_url, &err))
	    {
	      error_dialog = gt_error_dialog(gt_app_window, _("Unable to display the translation"), "%s", err->message);
	      g_error_free(err);
	    }
	}
      else
	error_dialog = gt_error_dialog(gt_app_window, _("Unable to get the location of the translated web page"), "%s", err->message);

      gt_twp_stop_ui();
    }

  g_free(translated_url);
  if (err)
    g_error_free(err);

  gdk_flush();
  GDK_THREADS_LEAVE();

  g_object_unref(info->session);
  g_free(info->url);
  g_free(info->from);
  g_free(info->to);
  g_free(info);

  return NULL;
}

static gboolean
gt_twp_translate_progress_cb (double progress, gpointer user_data)
{
  TranslateInfo *info = user_data;
  gboolean ret;

  GDK_THREADS_ENTER();

  if (info->cancelled)
    ret = FALSE;
  else
    {
      gt_source_remove(&stalled_timeout_id);

      if (progress < 0)
	gtk_progress_bar_pulse(GTK_PROGRESS_BAR(twp.progressbar));
      else
	{
	  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(twp.progressbar), progress);
	  
	  if (progress > 0)
	    {
	      double elapsed;
	      int eta;
	      char *status;

	      elapsed = gt_get_current_time() - info->start_time;
	      eta = elapsed / progress - elapsed;
	      
	      status = g_strdup_printf(ngettext("About %i second left",
						"About %i seconds left",
						eta),
				       eta);
	      gtk_progress_bar_set_text(GTK_PROGRESS_BAR(twp.progressbar), status);
	      g_free(status);
	    }
	}

      stalled_timeout_id = g_timeout_add(3000, gt_twp_translate_stalled_cb, NULL);

      ret = TRUE;
    }

  gdk_flush();
  GDK_THREADS_LEAVE();

  return ret;
}

static gboolean
gt_twp_translate_stalled_cb (gpointer data)
{
  GDK_THREADS_ENTER();

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(twp.progressbar), _("Stalled"));

  stalled_timeout_id = 0;

  GDK_THREADS_LEAVE();

  return FALSE;			/* remove timeout */
}

static void
gt_twp_stop_ui (void)
{
  gt_source_remove(&stalled_timeout_id);

  current_translation = NULL;
  gtk_widget_destroy(GTK_WIDGET(twp.dialog));

  if (twp_standalone)
    {
      /*
       * Quit when the user closes the error dialog, or now if there
       * is no error dialog.
       */
      if (error_dialog)
	g_object_weak_ref(G_OBJECT(error_dialog), (GWeakNotify) gtk_main_quit, NULL);
      else
	gtk_main_quit();
    }
}

/* libglade callbacks */

void
gt_twp_response_h (GtkDialog *dialog, int response, gpointer user_data)
{
  switch (response)
    {
    case GTK_RESPONSE_HELP:
      gt_display_help(GTK_WINDOW(twp.dialog), "translate-web-page");
      break;

    case RESPONSE_PREFERENCES:
      gt_preferences_display(GTK_WINDOW(dialog));
      break;

    case GTK_RESPONSE_OK:
      gt_twp_add_location_to_history();
      gt_twp_translate();
      break;

    default: /* GTK_RESPONSE_CANCEL, GTK_RESPONSE_DELETE_EVENT, etc */
      if (current_translation)
	current_translation->cancelled = TRUE;

      gt_twp_stop_ui();
    }
}

void
gt_twp_location_changed_h (GtkEditable *editable, gpointer user_data)
{
  gt_twp_update_sensitivity();
}
