;;;; keyword-add.lisp --- example for libxml-clisp tutorial

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-08-25
;;; 
;;; $Hg: keyword-add.lisp,v ae6366215b15 2009-09-03T20:01:35+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP.EXAMPLES")



(defun new-storyinfo-keywords (filename &optional content)
  "Add certain new elements to the document from FILENAME.
A new `keyword' child with content CONTENT is added to every
`storyinfo' node in the `story' document parsed from FILENAME.
FILENAME must be a pathname designator, and CONTENT must be an XML
String designator.  The resulting document is written to the default
XML output stream.  Returns the number of octets written."
  (with-xml-file (story filename)
    (let ((document-element (document-element story)))
      (cond ((story-node-p document-element)
              (add-storyinfo-keywords document-element content)
              (write-document story :encoding charset:iso-8859-1))
            (t (restart-case (error 'story-error :document story)
                 (parse-new-file (new-filename)
                   :report "Parse another file."
                   :interactive read-new-value
                   (new-storyinfo-keywords new-filename content))))))))

(defun add-storyinfo-keywords (node content)
  (node-list-map #'(lambda (nd)
                     (append-child-with-text nd "keyword" :content content))
                 (node-children node)
                 #'(lambda (nd)
                     (string= (node-namestring nd) "storyinfo"))))

(defun test-new-storyinfo-keywords (string)
  (with-temp-file (test-file)
    (with-open-file (out test-file :direction :output)
      (write-string string out))
    (with-output-to-string (stream)
      (with-xml-output (stream)
        (new-storyinfo-keywords test-file "fresh & new <keyword>")))))

(defvar *new-storyinfo-keywords-test-data*
  "<?xml version=\"1.0\"?>
<story>
  <storyinfo>
    <author>Foo B. Quux</author>
    <datewritten>2009-05-17</datewritten>
    <keyword>storyinfo-1/keyword-1</keyword>
    <keyword>storyinfo-1/keyword-2</keyword>
  </storyinfo>
  <storyinfo>
    <keyword>storyinfo-2/keyword-1</keyword>
    <keyword>storyinfo-2/keyword-2</keyword>
  </storyinfo>
  <body>
    <headline>The Headline</headline>
    <keyword>body/keyword</keyword>
    <para>This is the body text.</para>
  </body>
</story>")

(defvar *new-storyinfo-keywords-test-value*
  "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>
<story>
  <storyinfo>
    <author>Foo B. Quux</author>
    <datewritten>2009-05-17</datewritten>
    <keyword>storyinfo-1/keyword-1</keyword>
    <keyword>storyinfo-1/keyword-2</keyword>
  <keyword>fresh &amp; new &lt;keyword&gt;</keyword></storyinfo>
  <storyinfo>
    <keyword>storyinfo-2/keyword-1</keyword>
    <keyword>storyinfo-2/keyword-2</keyword>
  <keyword>fresh &amp; new &lt;keyword&gt;</keyword></storyinfo>
  <body>
    <headline>The Headline</headline>
    <keyword>body/keyword</keyword>
    <para>This is the body text.</para>
  </body>
</story>
")

(defun test-keyword-add ()
  (test-libxml-clisp #'test-new-storyinfo-keywords
                     *new-storyinfo-keywords-test-data*
                     *new-storyinfo-keywords-test-value*))

(provide-example 'keyword-add)



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; keyword-add.lisp ends here