;;;; xpath.lisp --- example for libxml-clisp tutorial

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-09-13
;;; 
;;; $Hg: xpath.lisp,v 238ceb51574b 2009-09-13T19:14:52+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP.EXAMPLES")



(defun xpath-file-eval (filename expression)
  "Return the content of the elements specified by an XPath expression.
Evaluates the XPath expression EXPRESSION in the XPath Context
provided by the document in FILENAME.  The return value is an
arbitrarily ordered list of strings, each of which is the content of
an element specified by EXPRESSION.  EXPRESSION must be an XPath
expression designator."
  (with-xml-file (document filename)
    (with-xpath-context (xpath-context document)
      (if (null-item-p xpath-context)
        (error "Unable to create XPath Context.")
        (with-xpath-value
            (xpath-object expression xpath-context)
          (if (null-item-p xpath-object)
            (error "Unable to evaluate XPath expression.")
            (node-set-map #'(lambda (node)
                              (node-list-text document node))
                          (xpath-object-node-set xpath-object))))))))

(defun test-xpath-file-eval (string)
  (with-temp-file (test-file)
    (with-open-file (out test-file :direction :output)
      (write-string string out))
    (xpath-file-eval test-file "/*/storyinfo//keyword")))

(defvar *xpath-file-eval-test-data*
  "<?xml version=\"1.0\"?>
<story>
  <storyinfo>
    <author>Foo B. Quux</author>
    <datewritten>2009-05-17</datewritten>
    <keyword>storyinfo-1/keyword-1</keyword>
    <keyword>storyinfo-1/keyword-2</keyword>
  </storyinfo>
  <storyinfo>
    <keyword>storyinfo-2/keyword-1</keyword>
    <keyword>storyinfo-2/keyword-2</keyword>
  </storyinfo>
  <body>
    <headline>The Headline</headline>
    <keyword>body/keyword</keyword>
    <para>This is the body text.</para>
  </body>
</story>")

(defvar *xpath-file-eval-test-value*
  '("storyinfo-1/keyword-1" "storyinfo-1/keyword-2"
    "storyinfo-2/keyword-1" "storyinfo-2/keyword-2"))

(defun test-xpath ()
  (flet ((set-equal (set-1 set-2)
           (and (subsetp set-1 set-2 :test #'equal)
                (subsetp set-2 set-1 :test #'equal))))
    (test-libxml-clisp #'test-xpath-file-eval
                       *xpath-file-eval-test-data*
                       *xpath-file-eval-test-value*
                       :test #'set-equal)))

(provide-example 'xpath)



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; xpath.lisp ends here