;;;; types.lisp --- FFI definitions for libxml-clisp

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-08-02
;;; 
;;; $Hg: types.lisp,v 238ceb51574b 2009-09-13T19:14:52+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP")



;;; An address.

(deftype address ()
  "An address is either nil or a foreign address.
If non-nil, it is a foreign address specifying the location of a
foreign object in memory."
  '(or null foreign-address))

(defun null-address-p (address)
  "Return true if ADDRESS is a null address."
  (etypecase address
    (address (null address))))

;;; The basic wrapper type.

(defclass item ()
  ((address
    :initarg :address
    :initform nil
    :accessor item-address
    :type address
    :documentation
    "The foreign address of the foreign object wrapped in the Item."))
  (:documentation
   "A representation of the output of an XML processor.
A representation in memory of the output of an XML processor acting on
an XML document, or some constituent of that output.  In practice, an
Item is a wrapper for a foreign object."))

(defmethod print-object ((item item) stream)
  (print-unreadable-object (item stream :type t :identity nil)
    (format stream ":ADDRESS ~A" (item-address item))))

(defun null-item-p (item)
  "Return true if ITEM is considered null."
  (null-address-p (item-address item)))

;;; An Item-list.

(defclass item-list (item)
  ()
  (:documentation
   "A representation of an ordered sequence of Items.
Every Item in an Item-list has a NEXT property, whose value is an
Item.  Using the NEXT property of its elements, an Item-list is
identified with the Item that is its first element.  The Item-list is
terminated by an Item whose NEXT property has value a null Item."))

;;; An XML String.

(def-c-type $xml-char-ptr c-pointer) ;xmlChar * in `xmlstring.h'.

(defclass xml-string (item)
  ()
  (:documentation
   "A representation of a Unicode string.
The string is represented by the octet vector that is its UTF-8
encoding.  This octet vector is stored, one octet per cell, in memory
cells with consecutive addresses starting from the address of the XML
String.  A terminating memory cell with the zero octet is used to mark
the end of the memory segment occupied by the octet vector."))

;;; The different element types carried by an XML tree.

(def-c-enum $xml-element-type   ;xmlElementType in `tree.h'.
  (xml-element-node 1)
  (xml-attribute-node 2)
  (xml-text-node 3)
  (xml-cdata-section-node 4)
  (xml-entity-ref-node 5)
  (xml-entity-node 6)
  (xml-pi-node 7)
  (xml-comment-node 8)
  (xml-document-node 9)
  (xml-document-type-node 10)
  (xml-document-frag-node 11)
  (xml-notation-node 12)
  (xml-html-document-node 13)
  (xml-dtd-node 14)
  (xml-element-decl 15)
  (xml-attribute-decl 16)
  (xml-entity-decl 17)
  (xml-namespace-decl 18)
  (xml-xinclude-start 19)
  (xml-xinclude-end 20)
  (xml-docb-document-node 21))

;;; A Namespace.

(def-c-enum $xml-ns-type        ;xmlNsType in `tree.h'.
  (xml-element-node 1)
  (xml-attribute-node 2)
  (xml-text-node 3)
  (xml-cdata-section-node 4)
  (xml-entity-ref-node 5)
  (xml-entity-node 6)
  (xml-pi-node 7)
  (xml-comment-node 8)
  (xml-document-node 9)
  (xml-document-type-node 10)
  (xml-document-frag-node 11)
  (xml-notation-node 12)
  (xml-html-document-node 13)
  (xml-dtd-node 14)
  (xml-element-decl 15)
  (xml-attribute-decl 16)
  (xml-entity-decl 17)
  (xml-namespace-decl 18)
  (xml-xinclude-start 19)
  (xml-xinclude-end 20)
  (xml-docb-document-node 21))

(def-c-type $xml-ns-ptr c-pointer) ;xmlNsPtr in `tree.h'.

(defclass namespace (item)
  ()
  (:documentation
   "A representation of a namespace node."))

(defclass namespace-list (item-list)
  ()
  (:documentation
   "An Item-list each of whose elements is a Namespace."))

;;; A Dtd (Document type declaration), as defined by "<!DOCTYPE...".
;;; There is actually one for the internal subset, and one for the
;;; external subset.

(def-c-type $xml-dtd-ptr c-pointer) ;xmlDtdPtr in `tree.h'.

(defclass dtd (item)
  ()
  (:documentation
   "A representation of a document type declaration node."))

;;; An Attribute.

(def-c-enum $xml-attribute-type ;xmlAttributeType in `tree.h'.
  (xml-attribute-cdata 1)
  (xml-attribute-id)
  (xml-attribute-idref)
  (xml-attribute-idrefs)
  (xml-attribute-entity)
  (xml-attribute-entities)
  (xml-attribute-nmtoken)
  (xml-attribute-nmtokens)
  (xml-attribute-enumeration)
  (xml-attribute-notation))

(def-c-type $xml-attr-ptr c-pointer) ;xmlAttrPtr in `tree.h'.

(defclass attribute (item)
  ()
  (:documentation
   "A representation of an attribute node."))

(defclass attribute-list (item-list)
  ()
  (:documentation
   "An Item-list each of whose elements is an Attribute."))

;;; A Node.

(def-c-type $xml-node-ptr c-pointer) ;xmlNodePtr in `tree.h'.

(defclass node (item)
  ()
  (:documentation
   "A representation of a generic container item."))

(defclass node-list (item-list)
  ()
  (:documentation
   "An Item-list each of whose elements is a Node."))

;;; An Document.

(def-c-type $xml-doc-ptr c-pointer) ;xmlDocPtr in `tree.h'.

(defclass document (item)
  ()
  (:documentation
   "A representation of a document information item."))

;;; A Dictionary (of reusable strings).

;; No `defclass' for this since it is an internal type of libxml.
(def-c-type $xml-dict-ptr c-pointer) ;xmlDictPtr in `dict.h'.

;;; A Node-set.

(def-c-type $xml-node-set-ptr c-pointer) ;xmlNodeSetPtr in `xpath.h'.

(defclass node-set (item)
  ()
  (:documentation
   "A representation of an arbitrarily ordered set of nodes."))

;;; An XPath Object [http://www.w3.org/TR/xpath, Section 1].

(def-c-enum $xml-xpath-object-type   ;xmlXPathObjectType in `tree.h'.
  (xpath-undefined 0)
  (xpath-nodeset 1)
  (xpath-boolean 2)
  (xpath-number 3)
  (xpath-string 4)
  (xpath-point 5)
  (xpath-range 6)
  (xpath-locationset 7)
  (xpath-users 8)
  (xpath-xslt-tree 9))

(def-c-type $xml-xpath-object-ptr c-pointer) ;xmlXPathObjectPtr in `xpath.h'.

(defclass xpath-object (item)
  ()
  (:documentation
   "A representation of an XPath object."))

;;; An XPath Context [http://www.w3.org/TR/xpath, Section 1].

(def-c-type $xml-xpath-context-ptr c-pointer) ;xmlXPathContextPtr in `xpath.h'.

(defclass xpath-context (item)
  ()
  (:documentation
   "A representation of an XPath context."))

;;; An XML Output Buffer (temporary segment of memory for libxml
;;; output).

;; No `defclass' for this since it is an internal type of libxml.
(def-c-type $xml-output-buffer-ptr c-pointer) ;xmlOutputBufferPtr in `tree.h'.

;;; An XML Output Write Callback Function.

;; A foreign function that can be used to write to buffers.
(def-c-type $xml-output-write-callback-function
    (c-function (:language :stdc) (:return-type int)
                (:arguments (context c-pointer) (buffer c-pointer) (len int))))

;; The foreign address of an XML Output Write Callback Function.
(def-c-type $xml-output-write-callback ;xmlOutputWriteCallback in `xmlIO.h'.
    (c-pointer $xml-output-write-callback-function))

;;; An Encoding Handler.

;; xmlCharEncodingHandlerPtr in `encoding.h'.
(def-c-type $xml-char-encoding-handler-ptr c-pointer)

;;; Parser Option.

(def-c-enum $xml-parser-option ;xmlParserOption in `parser.h'.
  (xml-parse-recover 1) 
  (xml-parse-noent 2) 
  (xml-parse-dtdload 4) 
  (xml-parse-dtdattr 8) 
  (xml-parse-dtdvalid 16) 
  (xml-parse-noerror 32) 
  (xml-parse-nowarning 64) 
  (xml-parse-pedantic 128) 
  (xml-parse-noblanks 256) 
  (xml-parse-sax1 512) 
  (xml-parse-xinclude 1024)
  (xml-parse-nonet 2048)
  (xml-parse-nodict 4096)
  (xml-parse-nsclean 8192)
  (xml-parse-nocdata 16384)
  (xml-parse-noxincnode 32768)
  (xml-parse-compact 65536))

;;; A Text Reader.

;; xmlTextReaderPtr in `xmlreader.h'.
(def-c-type $xml-text-reader-ptr c-pointer)

(def-c-enum $xml-reader-types   ;xmlReaderTypes in `xmlreader.h'.
  (xml-reader-type-none 0)
  (xml-reader-type-element 1)
  (xml-reader-type-attribute 2)
  (xml-reader-type-text 3)
  (xml-reader-type-cdata 4)
  (xml-reader-type-entity-reference 5)
  (xml-reader-type-entity 6)
  (xml-reader-type-processing-instruction 7)
  (xml-reader-type-comment 8)
  (xml-reader-type-document 9)
  (xml-reader-type-document-type 10)
  (xml-reader-type-document-fragment 11)
  (xml-reader-type-notation 12)
  (xml-reader-type-whitespace 13)
  (xml-reader-type-significant-whitespace 14)
  (xml-reader-type-end-element 15)
  (xml-reader-type-end-entity 16)
  (xml-reader-type-xml-declaration 17))

(defclass text-reader (item)
  ()
  (:documentation
   "A representation of an XML text reader."))



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; types.lisp ends here