;;;; sexp.lisp --- example for libxml-clisp tutorial

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-09-20
;;; 
;;; $Hg: sexp.lisp,v a5b014b13f84 2009-09-22T13:30:29+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP.EXAMPLES")



(defun xml-sexp (filename &optional catalog debug-catalog)
  "Return a sexp representing the document parsed from FILENAME.
CATALOG must be a pathname designator for an XML catalog.  If
DEBUG-CATALOG is true, then verbose information about XML catalog
usage is provided."
  (with-input-from-string (in (with-output-to-string (out)
                                (sexp-reader filename out
                                             catalog debug-catalog)))
    (when in
      (loop for sexp = (read in nil in)
         until (eql sexp in)
         collect sexp))))

(defun sexp-reader (filename stream catalog debug-catalog)
  "Write to STREAM a sexp representing the document parsed from FILENAME.
CATALOG must be a pathname designator for an XML catalog.  If
DEBUG-CATALOG is true, then verbose information about XML catalog
usage is provided."
  (let ((old-value (ext:getenv "XML_DEBUG_CATALOG")))
    (when debug-catalog
      (setf (ext:getenv "XML_DEBUG_CATALOG") "1"))
    (unwind-protect
      (progn
        (when catalog
          (initialize-catalog)
          (load-catalog (namestring catalog)))
        (with-text-reader (reader filename :options '(xml-parse-noent
                                                      xml-parse-dtdload
                                                      xml-parse-dtdattr
                                                      xml-parse-xinclude
                                                      xml-parse-nocdata))
          (loop for reader-status = (text-reader-read reader)
             while reader-status
             do (node-sexp reader stream))))
      (setf (ext:getenv "XML_DEBUG_CATALOG") old-value))))

(defun node-sexp (reader stream)
  "Write to STREAM a sexp representing the current node of READER."
  (case (text-reader-node-type reader)
    (xml-reader-type-element (process-element reader stream))
    (xml-reader-type-text (process-text reader stream))
    (xml-reader-type-end-element (process-etag reader stream))
    (xml-reader-type-processing-instruction (process-pi reader stream))
    (otherwise nil)))

;;; Process the various types of nodes.

(defun process-element (reader stream)
  (process-stag reader stream)
  (loop for attr-status = (text-reader-move-to-next-attribute reader)
     while attr-status
     do (process-attribute reader stream))
  (text-reader-move-to-element reader)
  (when (text-reader-empty-element-p reader)
    (process-etag reader stream)))

(defun process-stag (reader stream)
  (format stream "(~A~%"
          (xml-string-to-string (text-reader-local-name reader)))
  (let ((ns-name (text-reader-namespace-name reader)))
    (unless (null-item-p ns-name)
      (format stream "(xml-namespace~%~S~%) ;xml-namespace~%"
              (xml-string-to-string ns-name)))))

(defun process-etag (reader stream)
  (format stream ") ;~A~%"
          (xml-string-to-string (text-reader-local-name reader))))

(defun process-attribute (reader stream)
  (unless (text-reader-namespace-declaration-p reader)
    (process-stag reader stream)
    (process-text reader stream)
    (process-etag reader stream)))

(defun process-text (reader stream)
  (let ((xml-text (text-reader-text reader)))
    (unless (null-item-p xml-text)
      (escape (xml-string-to-string xml-text) stream))))

(defun escape (string stream)
  (format stream "~S~%" string))

(defun process-pi (reader stream)
  (process-stag reader stream)
  (format stream "(processing-instruction~%")
  (process-text reader stream)
  (format stream ") ;processing-instruction~%")
  (process-etag reader stream))

(defvar *xml-sexp-test-value*
  '((DRAFT (PROCESSING-INSTRUCTION "omit=\"author\""))
    (DOC (XML-NAMESPACE "urn:example:ns:doc") (ID "sexp") (LANG "en-GB")
     (LATEX (PROCESSING-INSTRUCTION "class=\"article\" fontsize=\"12pt\""))
     (DOCINFO (XML-NAMESPACE "urn:example:ns:doc")
      (AUTHOR (XML-NAMESPACE "urn:example:ns:doc")
       (FIRSTNAME (XML-NAMESPACE "urn:example:ns:doc") "Schöne")
       (SURNAME (XML-NAMESPACE "urn:example:ns:doc") "Grüße"))
      (TITLE (XML-NAMESPACE "urn:example:ns:doc") "An Example Article")
      (TITLEABBREV (XML-NAMESPACE "urn:example:ns:doc") (ROLE "running-title")
       "Example art")
      (KEYWORDSET (XML-NAMESPACE "urn:example:ns:doc")
       (KEYWORD (XML-NAMESPACE "urn:example:ns:doc")
                (TYPE (XML-NAMESPACE "urn:example:ns:keyword") "ai")
                "an articleinfo keyword")
       (KEYWORD (XML-NAMESPACE "urn:example:ns:doc")
                (TYPE (XML-NAMESPACE "urn:example:ns:keyword") "ai")
                "another articleinfo keyword")))
     (SECTION
      (SECTIONINFO
       (KEYWORDSET
        (KEYWORD (TYPE (XML-NAMESPACE "urn:example:ns:keyword") "si")
                 "a sectioninfo keyword")))
      (TITLE "A Section")
      (PARA
       "This is a paragraph in a section.  It does not quote any one,
  so we don't need any `\"' characters.")
      (PARA "The canonical URI for this document is "
       (ULINK (URL "http://www.example.org/sexp"))
       ".  Please check there for the
  latest updates.  Meanwhile, here is Äñ example of
  character entity references.")))))

(defun test-sexp ()
  (equal (xml-sexp (merge-pathnames "sexp.xml" *test-directory*)
                   (merge-pathnames "sexp.cat" *test-directory*))
         *xml-sexp-test-value*))

(provide-example 'sexp)



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; sexp.lisp ends here